/*
 * $Id: WebResourcesDecorator.java,v 1.6 2004/04/26 06:46:11 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui;

import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.core.runtime.Preferences.PropertyChangeEvent;
import org.eclipse.jface.viewers.*;

import com.narucy.webpub.core.WebProject;

public class WebResourcesDecorator extends LabelProvider implements ILightweightLabelDecorator, IResourceChangeListener {

	final public static String
		ID_DECORATOR = "com.narucy.webpub.ui.WebResourcesDecorator";
	
	static String[] folderKeys = {
		WebProject.KEY_HTSOURCES_FOLDER,
		WebProject.KEY_SCRIPTS_FOLDER,
		WebProject.KEY_PUBLISH_FOLDER,
	};
	
	static int getKeyIndex(String k){
		for(int i=0; i<folderKeys.length; i++){
			if(folderKeys[i].equals(k)){
				return i;
			}
		}
		throw new IllegalArgumentException("invalid key:" + k);
	}
	
	HashMap listenersCache = new HashMap();
	
	public WebResourcesDecorator(){
		try{
			IProject[] projs = ResourcesPlugin.getWorkspace().getRoot().getProjects();
			for (int i = 0; i < projs.length; i++) {
				IProject proj = projs[i];
				if(proj.hasNature(WebProject.ID_NATURE)){
					regist(proj);
				}
			}
		}catch(CoreException ex){
			WebpubUIPlugin.handleException(ex);
		}
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
	}
	
	void regist(final IProject proj) throws CoreException {
		final WebProject wp = (WebProject)proj.getNature(WebProject.ID_NATURE);
		if(wp == null || listenersCache.containsKey(proj)){
			return;
		}
		Preferences.IPropertyChangeListener listener = new Preferences.IPropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent event) {
				refresh(wp, event.getProperty());
			}
		};
		wp.addPropertyChangeListener(listener);
		
		Object[] entry = {
			wp.getFolder(WebProject.KEY_HTSOURCES_FOLDER),
			wp.getFolder(WebProject.KEY_SCRIPTS_FOLDER),
			wp.getFolder(WebProject.KEY_PUBLISH_FOLDER),
			listener,
		};
		listenersCache.put(proj, entry);
		
		IResource[] refreshResources = new IResource[3];
		System.arraycopy(entry, 0, refreshResources, 0, 3);
		fireLabelProviderChanged(new LabelProviderChangedEvent(this, refreshResources));
//		fireLabelProviderChanged(new LabelProviderChangedEvent(this, entry));
	}
	
	Preferences.IPropertyChangeListener getCachedListener(IProject proj){
		Object[] entry = (Object[])listenersCache.get(proj);
		return entry != null ? (Preferences.IPropertyChangeListener)entry[3] : null;
	}
	
	void refresh(WebProject wp, String key){
		Object[] entry = (Object[])listenersCache.get(wp.getProject());
		int entryKeyIndex = getKeyIndex(key);
		
		// erase decorate old container and new decorate new container
		IContainer[] refreshContainers = {
			(IContainer)entry[entryKeyIndex],
			wp.getFolder(key),
		};
		fireLabelProviderChanged(new LabelProviderChangedEvent(this, refreshContainers));
		
		// update cached entry
		entry[entryKeyIndex] = refreshContainers[1];
	}

	public void decorate(Object element, IDecoration decoration) {
		if(!(element instanceof IAdaptable)){
			return;
		}
		IContainer container = element instanceof IContainer ?
			(IContainer)element :
			(IContainer)((IAdaptable) element).getAdapter(IContainer.class);
		
		if(container != null){
			Object[] projects = listenersCache.keySet().toArray();
			for (int i = 0; i < projects.length; i++) {
				IProject proj = (IProject)projects[i];
				Object[] entry = (Object[])listenersCache.get(proj);
				
				if(proj.equals(container)){
					decoration.addOverlay(WebpubUIPlugin.getImageDescriptor("ovr16/webproject_ovr.gif"));
				} else if (entry[0].equals(container)){
					decoration.addOverlay(WebpubUIPlugin.getImageDescriptor("ovr16/htsource_folder_ovr.gif"));
				} else if (entry[1].equals(container) ){
					decoration.addOverlay(WebpubUIPlugin.getImageDescriptor("ovr16/scripts_folder_ovr.gif"));
				} else if( entry[2].equals(container) ){
					decoration.addOverlay(WebpubUIPlugin.getImageDescriptor("ovr16/publish_folder_ovr.gif"));
				}
			}
		}
	}

	public void resourceChanged(IResourceChangeEvent event) {
		if(event.getDelta() == null){
			return;
		}
		try {
			final ArrayList refreshResources = new ArrayList();
			event.getDelta().accept(new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta delta) throws CoreException {
					IResource r = delta.getResource();
					if(r instanceof IProject){
						if(delta.getKind() == IResourceDelta.ADDED){
							regist((IProject)r);
						}else if(delta.getKind() == IResourceDelta.REMOVED){
							unregist((IProject)r);
						}
					}
					return true;
				}
			});
		} catch (CoreException ex) {
			WebpubUIPlugin.handleException(ex);
		}
	}
	
	void unregist(IProject proj){
		Object[] entry = (Object[])listenersCache.remove(proj);
		if(entry != null && proj.isAccessible()){
			try{
				WebProject wp = (WebProject)proj.getNature(WebProject.ID_NATURE);
				if(wp != null){
					wp.removePropertyChangeListener( (Preferences.IPropertyChangeListener)entry[3] );
				}
			}catch(CoreException ex){
				WebpubUIPlugin.handleException(ex);
			}
		}
	}

	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		Object[] projects = listenersCache.keySet().toArray();
		for (int i=0; i<projects.length; i++) {
			unregist((IProject)projects[i]);
		}
	}
	

}


