/*
 * $Id: PresetSelectionWizardPage.java,v 1.1 2004/01/17 12:08:07 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.wizards;

import java.io.*;
import java.net.URL;
import java.util.ArrayList;
import java.util.regex.Pattern;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;

import com.narucy.webpub.core.FileUtilities;
import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.preset.Preset;
import com.narucy.webpub.core.preset.PresetManager;
import com.narucy.webpub.core.toc.*;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.views.PresetViewer;

/**
 * 
 */
class PresetSelectionWizardPage extends WizardPage {
	
	final static String PAGE_NAME = "Preset";
	
	Pattern[] ignoreFilePatterns = new Pattern[]{
		Pattern.compile("CVS"),
	};
	
	Preset[] presets;
	PresetViewer viewer;
	Combo combo;
	
	public PresetSelectionWizardPage() {
		super(PAGE_NAME);
		setTitle("Preset");
		setDescription("Define web site structure. You can select presets that for easy to initialization the store files.");
		setImageDescriptor(WebpubUIPlugin.getDefault().getImageDescriptor("wizban/newwebproj_wiz.gif"));
	}

	public void createControl(Composite parent) {
		presets = PresetManager.getInstance().getPrests();
		
		Composite topComp = new Composite(parent, SWT.NONE);
		topComp.setLayout(new GridLayout(2,false));
		
		// preset selection combo box setting
		new Label(topComp, SWT.NONE).setText("Preset:");
		combo = new Combo(topComp, SWT.READ_ONLY);
		combo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.GRAB_HORIZONTAL));
		
		for (int i = 0; i < presets.length; i++) {
			combo.add( presets[i].getName() );
		}
		combo.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				int index = combo.getSelectionIndex();
				viewer.setPreset(presets[index]);
			}
			
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		// table tree settings
		viewer = new PresetViewer(topComp, new String[]{
			PresetViewer.KEY_LABEL,
			PresetViewer.KEY_HREF,
			PresetViewer.KEY_INITIAL_HT});
		
		GridData gd = new GridData(GridData.FILL_BOTH | GridData.FILL);
		gd.horizontalSpan = 2;
		viewer.getControl().setLayoutData(gd);
		
		// init selection
		viewer.setPreset( presets[0] );
		combo.select(0);
		
		setControl( topComp);
	}
	
	public void storePresets(WebProject desc, IProgressMonitor monitor) throws CoreException {
		try{
			monitor.beginTask("Apply preset", 2);
			ResourcesPlugin.getWorkspace().run( new PresetStoreProc(desc), new SubProgressMonitor(monitor, 1));
			desc.getProject().refreshLocal(IResource.DEPTH_INFINITE, new SubProgressMonitor(monitor, 1));
		}finally{
			monitor.done();
		}
	}
	
	class PresetStoreProc implements IWorkspaceRunnable{
		
		WebProject webProject;
		
		PresetStoreProc(WebProject webProj){
			this.webProject = webProj;
		}
		
		public void run(IProgressMonitor monitor) throws CoreException {
			try{
				monitor.beginTask(
					"Distribute preset HTML files and folders",
					IProgressMonitor.UNKNOWN);
			
				Preset preset = presets[ combo.getSelectionIndex() ];
			
				IContainer distHtFolder = webProject.getFolder(WebProject.KEY_HTSOURCES_FOLDER);
				IContainer distScriptFolder = webProject.getFolder(WebProject.KEY_SCRIPTS_FOLDER);

				// copy files
				URL baseResourcesDir = preset.getInitialResourcesDirectory();
				if(baseResourcesDir != null){
					File resourceDir = new File( baseResourcesDir.getFile() );
					if( resourceDir.isDirectory() ){
						FileUtilities.copyWithDirContainFiles(
							resourceDir,
							webProject.getProject().getLocation().toFile(),
							ignoreFilePatterns);
					}
				}

				URL htDirUrl = preset.getInitialHtsDirectory();
				if( htDirUrl != null){
					File htDir = new File( htDirUrl.getFile() );
					if( htDir.isDirectory() ){
						FileUtilities.copyWithDirContainFiles(
							htDir,
							distHtFolder.getLocation().toFile(),
							ignoreFilePatterns);
					}
				}
			
				URL scriptDirUrl = preset.getScriptsDirectory();
				if( scriptDirUrl != null){
					File scriptDir = new File(scriptDirUrl.getFile() );
					if( scriptDir.isDirectory() ){
						FileUtilities.copyWithDirContainFiles(
							scriptDir,
							distScriptFolder.getLocation().toFile(),
							ignoreFilePatterns);
					}
				}
				monitor.worked(10);
			
				Toc toc = viewer.getToc();
				TocStore.storeToc(toc, webProject.getProject().getFile("toc.xml") );
			
				Topic[] topics = toc.getAllTopics();
				for (int i = 0; i < topics.length; i++) {
					Topic topic = topics[i];
					String href = topic.getHref();
					if( href != null){
						IFile file = distHtFolder.getFile(new Path(href));
						URL initUrl = viewer.getInitialURL((Topic)topic);
					
						ArrayList folders = new ArrayList(0);
						for(IResource res = file; (res = res.getParent()) instanceof IFolder && !res.exists(); ){
							folders.add(res);
						}
						for(int j=folders.size()-1; j>=0; j--){
							((IFolder)folders.get(j)).create(true, false, new SubProgressMonitor(monitor, 1));
						}
					
						InputStream is = initUrl.openStream();
						try{
							file.create(is, true, new SubProgressMonitor( monitor, 1) );
						}finally{
							if( is != null){
								is.close();
							}
						}
					}
				}
			} catch(IOException e){
				WebpubUIPlugin.handleException(e);
			} finally{
				monitor.done();
			}
		}
	}
	
}
