/*
 * $Id: PresetViewer.java,v 1.1 2004/01/17 12:08:11 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.views;

import java.io.IOException;
import java.net.URL;
import java.util.*;

import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.core.runtime.*;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.*;
import org.xml.sax.SAXException;

import com.narucy.webpub.core.preset.Preset;
import com.narucy.webpub.core.toc.Toc;
import com.narucy.webpub.core.toc.Topic;
import com.narucy.webpub.ui.WebpubUIPlugin;



/**
 * The role of the class is to edit toc model.
 * 
 */
public class PresetViewer extends TableTreeViewer {

	static DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
	
	final public static String
		KEY_LABEL = "label",
		KEY_HREF = "href",
		KEY_INITIAL_HT = "initalHt";
	
	final static HashMap columnNames = new HashMap(3);
	
	static {
		columnNames.put(KEY_LABEL, "Label");
		columnNames.put(KEY_HREF, "Href");
		columnNames.put(KEY_INITIAL_HT, "Initial HTML");
	}
	
	HashMap selectedInitialHtdocs = null;
	ArrayList initialHtdocs = new ArrayList();
		
	Preset  currentPreset = null;
	Toc currentToc = null;

	class TocCellModifier implements ICellModifier{
		
		public boolean canModify(Object element, String property) {
			if (!(element instanceof Topic)){
				return false;
			}
			Topic topic = (Topic)element;
			if(property.equals(KEY_INITIAL_HT)){
				return (topic.getHref() != null);
			}
			return true;
		}

		public Object getValue(Object element, String property) {
			if (element instanceof Topic){
				Topic topic = (Topic)element;
				if ( property.equals(KEY_LABEL)){
					return topic.getLabel();
				}
				else if( property.equals(KEY_HREF) ){
					String href = topic.getHref();
					return (href != null) ? href : "";
				}
				else if( property.equals(KEY_INITIAL_HT) ){
					URL f = getInitialURL(topic);
					int index = initialHtdocs.indexOf(f);
					return new Integer(index);
				}
			}
			return null;
		}

		public void modify(Object elem, String property, Object value) {
			if(elem instanceof TableTreeItem){
				Object obj = ((TableTreeItem)elem).getData();
				if( obj instanceof Topic){
					ArrayList modifyProperties = new ArrayList(3);
					
					Topic topic = (Topic)obj;
					if ( property.equals(KEY_LABEL)){
						// modify label
						topic.setLabel((String)value);
						modifyProperties.add(KEY_LABEL);
					}
					else if( property.equals(KEY_HREF) ){
						// modify href
						String p = (String)value;
						if( p.length() <= 0){
							topic.setHref(null);
							selectedInitialHtdocs.remove(topic);
							modifyProperties.add(KEY_HREF);
							modifyProperties.add(KEY_INITIAL_HT);
						}
						else if( true ){
							topic.setHref(p);
							modifyProperties.add(KEY_HREF);
						}
						else{
							showError("Invalid path: " + p, IStatus.WARNING);
						}
						
					}
					else if (property.equals(KEY_INITIAL_HT)){
						// modify template
						int index = ((Integer)value).intValue();
						selectedInitialHtdocs.put(
							topic,
							initialHtdocs.get(index) );
						
						modifyProperties.add(KEY_INITIAL_HT);
					}
					update(topic, (String[])modifyProperties.toArray(new String[modifyProperties.size()]));
				}
			}
		}
	}
	
	static String getFileName(URL url){
		String line = url.toString();
		int index = line.lastIndexOf('/');
		return index != -1 ? line.substring(index+1) : "(foo)";
	}
	
	class TocLabelProvider extends LabelProvider implements ITableLabelProvider {
	
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		public String getColumnText(Object elem, int columnIndex) {
			if(elem instanceof Topic){
				Topic topic = (Topic)elem;
				switch (columnIndex) {
					case 0:
						return topic.getLabel();
					case 1:
						String href = topic.getHref();
						return href != null ? href : "(none)";
					case 2:
						URL url = getInitialURL(topic);
						return (url != null && topic.getHref() != null) ? getFileName(url) : "(none)";
				}
			}
			return null;
		}

		public boolean isLabelProperty(Object element, String property) {
			Object[] columnProperties = getColumnProperties();
			for(int i=0; i<columnProperties.length; i++){
				if( columnProperties[i].equals(property) ){
					return true;
				}
			}
			return false;
		}

	}
	
	public PresetViewer(Composite composite){
		this(composite,
			new String[]{
				PresetViewer.KEY_LABEL,
				PresetViewer.KEY_HREF});
	}
	
	public PresetViewer(Composite composite, String[] columnProp){
		super(composite,
			SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL |
			SWT.BORDER | SWT.FULL_SELECTION);
		
		// sets header property
		setColumnProperties(columnProp);
		
		// if template editing mode is enabled, these for value initialize.
		if( isContainHeaderProperty(KEY_INITIAL_HT) ){
			initialHtdocs = new ArrayList();
			selectedInitialHtdocs = new HashMap();
		}
		
		// init table
		TableTree tableTree = getTableTree();
		Table table = tableTree.getTable();
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		for (int i = 0; i < columnProp.length; i++) {
			String key = columnProp[i];
			TableColumn col = new TableColumn(table, SWT.NONE);
			col.setText( (String)columnNames.get(key) );
		}
		
		// viewer settings
		setContentProvider(new TocContentProvider());
		setLabelProvider(new TocLabelProvider());
		setCellModifier(new TocCellModifier());
	}
	
	boolean isContainHeaderProperty(String mode){
		Object[] columnProperties = getColumnProperties();
		for (int i = 0; i <columnProperties.length; i++) {
			if( columnProperties[i].equals(mode) ){
				return true;
			}
		}
		return false;
	}
				
	
	public void setColumnProperties(String[] columnProps) {
		assertHeaderProperties(columnProps);
		super.setColumnProperties((String[])columnProps.clone());
	}
	
	static void assertHeaderProperties(String[] props){
		Set keys = columnNames.keySet();
		for(int i=0; i<props.length; i++){
			if(!keys.contains(props[i])){
				throw new IllegalArgumentException("Illegal header property key:" + props[i]);
			}
		}
	}

	public void setPreset(Preset preset) {
		currentPreset = preset;
		try {
			currentToc = preset.createToc();
			if( isContainHeaderProperty(KEY_INITIAL_HT) ){
				initInitialHtList(preset);
			}
		} catch (IOException e) {
			WebpubUIPlugin.handleException(e);
		} catch (SAXException e) {
			WebpubUIPlugin.handleException(e);
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
		if(currentToc == null){
			return;
		}
		
		// init editors
		Table table = getTableTree().getTable();
		Object[] columnProperties = getColumnProperties();
		CellEditor[] editors = new CellEditor[columnProperties.length];
		for(int i=0; i<columnProperties.length; i++){
			String key = (String)columnProperties[i];
			if(key.equals(KEY_LABEL) || key.equals(KEY_HREF)){
				editors[i] = new TextCellEditor(table);
			}
			else if (key.equals(KEY_INITIAL_HT)){
				URL[] initalURLs = (URL[])initialHtdocs.toArray(new URL[initialHtdocs.size()]);
				String[] initalURLNames = new String[initalURLs.length];
				for (int j=0; j<initalURLs.length; j++) {
					initalURLNames[j] = getFileName( initalURLs[j] );
				}
				editors[i] = new ComboBoxCellEditor(table, initalURLNames);
			}
		}
		setCellEditors(editors);
		setInput(currentToc);

		// expand tree node and pack compponent size
		expandToLevel(TreeViewer.ALL_LEVELS);
		TableColumn[] columns = table.getColumns();
		for (int i = 0; i < columns.length; i++) {
			columns[i].pack();
		}
	}
	
	void initInitialHtList(Preset preset) throws IOException, CoreException{
		initialHtdocs.clear();
		initialHtdocs.addAll( Arrays.asList(preset.getHtSelectionURLs()) );
				
		URL defaultInitalURL = preset.getDefaultSelectionURL();
		if( defaultInitalURL == null){
			defaultInitalURL = (URL)initialHtdocs.get(0);
		}
				
		Topic[] topics = currentToc.getAllTopics();
		selectedInitialHtdocs.clear();
		for(int i=0; i<topics.length; i++){
			if( topics[i].getHref() != null){
				URL url = null;
				if(topics[i] instanceof Topic){
					Topic t = (Topic)topics[i];
					String item = t.getAttribute("selected");
					if( item != null){
						URL base = preset.getBaseDirectory();
						int index = initialHtdocs.indexOf( new URL(base, item) );
						if( index == -1){
							throw new IllegalArgumentException(
								"Invalid attribute found as inital toc: selected=\"" + item + "\"");
						}
						url = (URL)initialHtdocs.get(index);
					}
				}
				if( url == null){
					url = defaultInitalURL;
				}
				selectedInitialHtdocs.put( topics[i], url);
			}
		}
	}
		
	void showError(String message, int severity) {
		Status status = new Status(
			severity,
			WebpubUIPlugin.ID_PLUGIN,
			IStatus.OK,
			message,
			null);
		
		ErrorDialog.openError(
			getControl().getShell(),
			null,
			null,
			status);
	}

	public Toc getToc() {
		return currentToc;
	}

	public URL getInitialURL(Topic topic){
		return (URL)selectedInitialHtdocs.get(topic);
	}


}
