/*
 * $Id: HTPreviewLinkWithEditorManager.java,v 1.1 2004/01/17 12:08:11 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.views;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.*;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.*;

import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.ui.MnemonicNamer;
import com.narucy.webpub.ui.WebpubUIPlugin;

/**
 * PreviewAutoActivedManager provides editor part bridge.
 * 
 */
class HTPreviewLinkWithEditorManager {

	final public static String
		M_PREVIEW_TARGETS = "previewTargets",
		M_PREVIEW_BEHAVIOR = "previewBehavior";

	IPreferenceStore preferenceStore = WebpubUIPlugin.getDefault().getPreferenceStore();
	HTPreView htPreview;
	
	MnemonicNamer mnemonicNamer = new MnemonicNamer() {
		public String toLabel(Object obj) {
			IFile file = (IFile)obj;
			String name = file.getName();
			String fullPath = file.getFullPath().toString().substring(1);
			int nameLen = name.length();
			int fullLen = fullPath.length();
			if( nameLen + fullLen > 32){
				fullPath = fullPath.substring(0, fullLen-nameLen) + "...";
			}
			return name + "  [" + fullPath + "]";
		}
	};

	HashMap fileActions = new HashMap();
	Action linkWithEditorAction;

	IResourceChangeListener resourceListener = new IResourceChangeListener() {
		final ArrayList removeTargets = new ArrayList();

		IResourceDeltaVisitor visitor = new IResourceDeltaVisitor() {
			public boolean visit(IResourceDelta delta) throws CoreException {
				IResource r = delta.getResource();
				if( r instanceof IFile && delta.getKind() == IResourceDelta.REMOVED){
					removeTargets.add(r);
				}
				return true;
			}
		};
		
		public void resourceChanged(IResourceChangeEvent event) {
			removeTargets.clear();
			try{
				IResourceDelta delta = event.getDelta();
				if( delta != null ){
					delta.accept(visitor);
				}
				final IFile[] removes = (IFile[])removeTargets.toArray(new IFile[removeTargets.size()]);
				if( removes.length > 0){
					removeTargets.clear();
					asyncExec(new Runnable() {
						public void run() {
							undefinePreviewAction(removes);
						}
					});
				}
			}catch(CoreException e){
				WebpubUIPlugin.handleException(e);
			}
		}
	};

	/**
	 * Call back from prefereance.
	 */
	IPropertyChangeListener preferenceChangeListener = new IPropertyChangeListener() {
		public void propertyChange(PropertyChangeEvent event) {
			if( event.getProperty().equals(WebpubUIPlugin.HT_PREVIEW_LINK_WITH_EDITOR) ){
				linkWithEditorAction.setChecked( ((Boolean)event.getNewValue()).booleanValue() );
			}
		}
	};	

	/**
	 * Call back from part.
	 */
	IPartListener partListener = new IPartListener() {
		public void partBroughtToTop(IWorkbenchPart part) {}
		public void partDeactivated(IWorkbenchPart part) {}

		public void partClosed(IWorkbenchPart part) {
			if( part instanceof IEditorPart){
				unregistObserveEditor( (IEditorPart)part);
			}
		}

		public void partOpened(IWorkbenchPart part) {
			if( part instanceof IEditorPart){
				registObserveEditor( (IEditorPart)part);
			}
		}
		
		public void partActivated(IWorkbenchPart part) {
			if(isLinkWithEditor() && part instanceof IEditorPart){
				IFile editFile = getFile( (IEditorPart)part);
				changeBrowseFile(editFile);
			}
		}
	};
	
	/**
	 * Construct PrivewAutoActivedManager.
	 */
	public HTPreviewLinkWithEditorManager(HTPreView view) {
		super();
		this.htPreview = view;
		IMenuManager mgr = htPreview.getMenuManager();

		// addition editor menu item. 
		mgr.add( new Separator(M_PREVIEW_TARGETS));
		// create auto mate preview changer action
		mgr.add( new Separator(M_PREVIEW_BEHAVIOR));
		linkWithEditorAction = new Action("&Link with Editor", IAction.AS_CHECK_BOX ){
			public void run(){
				handleLinkWithEditorCheck();
			}
		};
		linkWithEditorAction.setToolTipText("Link with Editor");
		linkWithEditorAction.setImageDescriptor(WebpubUIPlugin.getDefault().getImageDescriptor("ctool16/synced.gif"));
		
		mgr.add(linkWithEditorAction);
		
		// init for current open editors 
		IEditorReference[] refs = htPreview.getSite().getPage().getEditorReferences(); 
		for (int i = 0; i < refs.length; i++) {
			registObserveEditor( refs[i].getEditor(true) );
		}

		// init state.
		linkWithEditorAction.setChecked( preferenceStore.getBoolean( WebpubUIPlugin.HT_PREVIEW_LINK_WITH_EDITOR) );
		
		view.getWebBrowser().addPropertyChangeListener(new IPropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent event) {
				if( event.getProperty().equals(AbstractWebBrowser.EVENT_NAME_URL_CHANGE_STARTED) ){
					try {
						handleWebBorwserUrlChanged(event);
					} catch (Exception e) {
						WebpubUIPlugin.handleException(e);
					}
				}
			}
		});
		view.getSite().getPage().addPartListener(partListener);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceListener);
	}
	
	void handleLinkWithEditorCheck(){
		boolean checked = linkWithEditorAction.isChecked();
		preferenceStore.setValue(WebpubUIPlugin.HT_PREVIEW_LINK_WITH_EDITOR, checked);
		if(!checked){
			return;
		}
				
		IFile[] registedFiles = getRegistedFiles();
		if( registedFiles.length > 0 ){
			IFile currentBrowseFile = htPreview.getCurrentPublishFrom();
			if( currentBrowseFile != null){
				
				IFile editFile = getFile( htPreview.getSite().getPage().getActiveEditor() );
				for (int i = 0; i < registedFiles.length; i++) {
					IFile r = registedFiles[i];
					if( currentBrowseFile.equals(r) ){
						// already opened the editor side file.
						return;
					}
					if( editFile.equals(r) ){
						changeBrowseFile(r);
					}
				}
			}
		}else{
			// link with editor, but editor is not opened.
			htPreview.browsePage(null);
		}
	}
	
	void handleWebBorwserUrlChanged(PropertyChangeEvent event) throws CoreException{
		URL after = (URL)event.getNewValue();
		IFile from = new SourceFileSeeker(after).getPublishFrom();
		
		URL before = (URL)event.getOldValue();

		IFile[] files = getRegistedFiles();
		for (int i = 0; i < files.length; i++) {
			IFile f = files[i];
 				getOpenAction(f).setChecked(f.equals(from));
		}

		// auto activiation browser url changed, notify workbench part.
		if(before != after &&
			(before != null ? !before.equals(after) : !after.equals(before)) &&
			isLinkWithEditor() &&
			fileActions.containsKey(from) ){
			
			htPreview.getSite().getPage().openEditor(from);
		}
	}
	
	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}

	public IFile[] getRegistedFiles(){
		return (IFile[])fileActions.keySet().toArray(new IFile[fileActions.size()]);
	}
	
	void changeBrowseFile(IFile editFile){
		if( editFile != null &&
			htPreview.isPreviewble(editFile) &&
			fileActions.containsKey(editFile) &&
			!getOpenAction(editFile).isChecked() ){
			
			// do browse and checke state changing.
			browseFileAsHTSource(editFile);
		}
	}

	public void definePreviewAction(final IFile file){
		if(!fileActions.containsKey(file)){
			Action action = new Action(mnemonicNamer.getLabel(file), IAction.AS_RADIO_BUTTON){
				public void run(){
					browseFileAsHTSource(file);
				}
			};
			ActionContributionItem actionItem = new ActionContributionItem(action);
			fileActions.put(file, actionItem);
			htPreview.getMenuManager().appendToGroup(M_PREVIEW_TARGETS , actionItem);
		}
	}
	
	IAction getOpenAction(IFile file){
		return fileActions.containsKey(file) ?
			((ActionContributionItem)fileActions.get(file)).getAction() :
			null;
	}

	IFile getFile(IEditorPart editor){
		IEditorInput input = editor.getEditorInput();
		return (input instanceof IFileEditorInput) ?
			((IFileEditorInput)input).getFile() :
			null;
	}

	void registObserveEditor(IEditorPart editor){
		IFile file = getFile(editor);
		if(file != null && htPreview.isPreviewble(file)){
			definePreviewAction(file);
		}
	}
	
	void unregistObserveEditor(IEditorPart editor){
		IFile file = getFile(editor);
		if(file != null){
			undefinePreviewAction(file);
		}
		if( isLinkWithEditor() && fileActions.isEmpty() ){
			htPreview.browsePage(null);
		}
	}
	
	void undefinePreviewAction(IFile file){
		undefinePreviewAction(new IFile[]{file} );
	}

	void undefinePreviewAction(final IFile[] files){
		IMenuManager mgr = htPreview.getMenuManager();
		
		for(int i=0; i<files.length; i++){
			ActionContributionItem actionItem = (ActionContributionItem)fileActions.remove(files[i]);
			if( actionItem != null){
				mgr.remove(actionItem);
			}
			mnemonicNamer.release(files[i]);
		}
	}
	
	static URL getPublishUrl(IFile file){
		if( file != null){
			PublishDescription desc = getPublishDescription(file);
			if( desc != null){
				try {
					return desc.getPublishTo().getLocation().toFile().toURL();
				} catch (MalformedURLException e) {
					WebpubUIPlugin.handleException(e);
				}
			}
		}
		
		return null;
	}
	
	public static PublishDescription getPublishDescription(IFile file){
		try{
			return PublishDescriptionFactory.getInstance().create(file);
		} catch (IllegalConfigurationException e) {
			WebpubUIPlugin.handleException(e);
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		} catch (IOException e) {
			WebpubUIPlugin.handleException(e);
		}
		return null;
	}
	
	boolean isLinkWithEditor(){
		return linkWithEditorAction != null && linkWithEditorAction.isChecked();
	}

	void asyncExec( Runnable proc){
		IWorkbenchPartSite site = htPreview.getSite();
		if( site != null ){
			Shell sh = site.getShell();
			if( sh != null && !sh.isDisposed()){
				Display disp = sh.getDisplay();
				if( disp != null && !disp.isDisposed()){
					disp.asyncExec(proc);
				}
			}
		}
	}
	
	void browseFileAsHTSource(IFile file){
		PublishDescription desc = getPublishDescription(file);
		try {
			IFile f = file;
			if( desc != null ){
				IResource res = desc.getPublishTo();
				if( res instanceof IFile){
					f = (IFile)res;
				}
			}
			htPreview.browsePage( f.getLocation().toFile().toURL() );
		} catch (MalformedURLException e) {
		}
	}

	public void dispose(){
		htPreview.getSite().getPage().removePartListener(partListener);
		preferenceStore.removePropertyChangeListener(preferenceChangeListener);
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceListener);
	}
	
	public Action getLinkWithEditorAction() {
		return linkWithEditorAction;
	}

}
