/*
 * $Id: HTPreView.java,v 1.2 2004/01/18 08:01:47 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.views;

import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.*;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.StringConverter;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;
import org.eclipse.ui.part.ViewPart;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.publish.SourceFileSeeker;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.actions.OpenEditorFromURLAction;
import com.narucy.webpub.ui.actions.RepublishAction;

/**
 * <p>
 * HTPreView role is to define of browser selection 
 * and indication a HTML document and display Title.
 * 
 * @TODO supports other Win32 system.
 * @TODO only the publish folder?
 */
public class HTPreView extends ViewPart {

	final static String KEY_URL_HISTORY = "url_history";

	IPreferenceStore preferenceStore;
	AbstractWebBrowser webBrowser;

	Label notAvailableLabel;
	Control browserControl;

	boolean available = true;

	Combo urlInput;
	Action republishAction = null;
	OpenEditorFromURLAction openEditorAction = null;
	
	HTPreviewLinkWithEditorManager linkWithEditorMgr;
	String originalTitle, loadingTitle;
	
	int maxItemCount = 10;
	String[] initialUrls = null;
	
	/**
	 * Returns publish original file if url file is publishable.
	 * If other case, return null.
	 */
	public IFile getCurrentPublishFrom() {
		IFile from = null;
		try{
			URL url = webBrowser.getCurrentURL();
			if (url != null) {
				from = new SourceFileSeeker(url).getPublishFrom();
			}
		}catch(CoreException e){
			WebpubUIPlugin.handleException(e);
		}
		return from;
	}
	
	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}
	
	/**
	 * <p>
	 * Call back from eclipse resources. Reload web browser when changed resource is
	 * current browse file reloational project.
	 */
	IResourceChangeListener resourceListener = new IResourceChangeListener() {
		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta delta = event.getDelta();
			if (delta == null) {
				return;
			}
			try {
				final boolean[] needReload = new boolean[]{false};
				
				delta.accept(new IResourceDeltaVisitor() {
					public boolean visit(IResourceDelta delta) throws CoreException {
						if ( isRefresh(delta.getResource()) ) {
							needReload[0] = true;
						}
						return true;
					}
				});
				
				if(needReload[0]){
					runReloadProc();
				}
				
			} catch (CoreException e) {
				WebpubUIPlugin.handleException(e);
			}
		}
		
		boolean isRefresh(IResource r) throws CoreException{
			final IFile from = getCurrentPublishFrom();
			if(r instanceof IFile && from != null){
				WebProject wp = (WebProject)from.getProject().getNature(WebProject.ID_NATURE);
				if( wp == null){
					return from.getProject().equals(r.getProject());
				}else{
					IContainer pubFolder = wp.getFolder(WebProject.KEY_PUBLISH_FOLDER);
					for(IContainer c = r.getParent(); !(c instanceof IWorkspaceRoot); c = c.getParent() ){
						if(c.equals(pubFolder)){
							return true;
						}
					}
				}
			}
			return false;
		}

		void runReloadProc() {
			asyncExec(new Runnable() {
				public void run() {
					if(isAvailable()){
						webBrowser.reload();
					}
				}
			});
		}
	};

	void asyncExec(Runnable proc) {
		IWorkbenchPartSite site = getSite();
		if (site != null) {
			Shell sh = site.getShell();
			if (sh != null && !sh.isDisposed()) {
				Display disp = sh.getDisplay();
				if (disp != null) {
					disp.asyncExec(proc);
					return;
				}
			}
		}
		throw new RuntimeException("Ran not invoke a runnable item:" + proc);
	}

	/**
	 * short cut:
	 * getViewSite().getActionBars().getMenuManager()
	 */
	public IMenuManager getMenuManager() {
		return getViewSite().getActionBars().getMenuManager();
	}

	/**
	 * Create controls.
	 */
	public void createPartControl(final Composite parent) {
		originalTitle = getTitle();
		loadingTitle = originalTitle + " [loading...]";
		
		preferenceStore = WebpubUIPlugin.getDefault().getPreferenceStore();
		
		// create control.
		urlInput = new Combo(parent, SWT.BORDER | SWT.DROP_DOWN);
		if(initialUrls != null){
			urlInput.setItems(initialUrls);
		}
		
		if (System.getProperty("os.name").indexOf("Windows") != -1) {
			webBrowser = new IEWebBrowser();
		} else {
			// @TODO: other os support.
			throw new RuntimeException("Other Win32 platform is not supported");
		}

		// create browser widgets and label. 
		notAvailableLabel = new Label(parent, SWT.NONE);
		notAvailableLabel.setText("A preview is not available.");

		browserControl = webBrowser.createControl(parent);
		parent.setLayout(new Layout() {
			protected Point computeSize(Composite composite, int wHint, int hHint, boolean flushCache) {
				return new Point(wHint, hHint);
			}

			protected void layout(Composite composite, boolean flushCache) {

				Rectangle area = composite.getClientArea();
				Point inputSize = urlInput.computeSize(SWT.DEFAULT, SWT.DEFAULT);
				
				urlInput.setLocation(0, 0);
				urlInput.setSize(area.width, inputSize.y);

				area.y = inputSize.y + 2;
				area.height -= inputSize.y;

				Control enable = available ? browserControl : notAvailableLabel;
				Control disable = available ? notAvailableLabel : browserControl;
				
				disable.setEnabled(false);
				disable.setSize(0, 0);
				enable.setEnabled(true);
				enable.setBounds(area);
			}
		});

		// init menu.
		IMenuManager menuMgr = getMenuManager();
		menuMgr.add(new Action("&Deselect Target") {
			public void run() {
				browsePage(null);
			}
		});
		linkWithEditorMgr = new HTPreviewLinkWithEditorManager(this);

		// create republish action
		final IActionBars bars = getViewSite().getActionBars();
		
		republishAction = new Action() {
			public void run() {
				IFile republishTarget = getCurrentPublishFrom();
				if (republishTarget != null) {
					RepublishAction act = new RepublishAction();
					act.init(getSite().getWorkbenchWindow());
					act.selectionChanged(this, new StructuredSelection(republishTarget));
					act.run(this);
				}
			}
		};
		republishAction.setToolTipText("Publish");
		republishAction.setImageDescriptor(WebpubUIPlugin.getDefault().getImageDescriptor("ctool16/republish.gif"));
		
		// toolbar
		IToolBarManager toolBar = bars.getToolBarManager();
		toolBar.add(republishAction);
		
		openEditorAction = new OpenEditorFromURLAction();
		toolBar.add(openEditorAction);
		toolBar.add(linkWithEditorMgr.getLinkWithEditorAction());
		
		// handle global action.
		bars.setGlobalActionHandler(IWorkbenchActionConstants.CUT, new Action() {
			public void run() {
				if( urlInput.isFocusControl()){
					urlInput.cut();
				}
			}
		});
		bars.setGlobalActionHandler(IWorkbenchActionConstants.COPY, new Action() {
			public void run() {
				if( urlInput.isFocusControl()){
					urlInput.copy();
				}
			}
		});
		bars.setGlobalActionHandler(IWorkbenchActionConstants.PASTE, new Action() {
			public void run() {
				if( urlInput.isFocusControl()){
					urlInput.paste();
				}
			}
		});
		
		//
		getSite().setSelectionProvider(new HTPreviewSelectionProvider(this));
		
		// listener setting
		urlInput.addKeyListener(new KeyListener() {
			public void keyPressed(KeyEvent e) {
				if (e.character == '\r' || e.character == '\n') {
					enterUrl();
				}
			}
			public void keyReleased(KeyEvent e) {}
		});
		urlInput.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				browsePage( createUrl(urlInput.getText()) );
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceListener);
		webBrowser.addPropertyChangeListener(new IPropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent event) {
				if (event.getProperty().equals(AbstractWebBrowser.EVENT_NAME_URL_CHANGE_STARTED)) {
					if( event.getNewValue() != null){
						setTitle(loadingTitle);
					}
				}else if( event.getProperty().equals(AbstractWebBrowser.EVENT_NAME_URL_CHANGE_COMPLETED) ){
					URL url = (URL)event.getNewValue();
					urlInput.setText(url.toString());
					
					String title = originalTitle;
					String pageTitle = webBrowser.getTitle();
					if( pageTitle != null){
						title += " - " + pageTitle;
					}
					setTitle(title);
					refreshActionsState();
					getSite().getSelectionProvider().setSelection(new StructuredSelection(url));
				}
			}
		});
		
		browsePage(null);
		urlInput.forceFocus();
		
		refreshActionsState();
	}
	
	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
		if(memento != null){
			String urlListStr = memento.getString(KEY_URL_HISTORY);
			if(urlListStr != null){
				initialUrls = urlListStr.split("\\s+");
			}
		}
	}

	void enterUrl(){
		String text = urlInput.getText();
		URL url = createUrl(text);
		if( url != null){
			String urlText = url.toString();
			if( !text.equals(urlText)){
				urlInput.setText(urlText);
			}
			if( !isContainURLHistory(url) ){
				urlInput.add(urlText, 0);
			}
			urlInput.setText(urlText);

			if( urlInput.getItemCount() > maxItemCount){
				urlInput.remove(urlInput.getItemCount()-1);
			}
			browsePage(url);
		}else{
			ErrorDialog.openError(
				getSite().getShell(),
				"Error",
				"Specify URL is invalid: " + text,
				null);
		}
	}

	static URL createUrl(String text){
		try {
			return new URL(text);
		} catch (MalformedURLException e) {
		}
		try {
			return new URL("http://" + text);
		} catch (MalformedURLException e1) {
		}
		return null;
	}
			
	void refreshActionsState() {
		URL url = webBrowser.getCurrentURL();
		if( url != null){
			IFile pubFrom = getCurrentPublishFrom();
			republishAction.setEnabled(pubFrom != null && url.getProtocol().equals("file") );
			openEditorAction.setOpenFile(pubFrom);
		}
	}

	boolean isContainURLHistory(URL url){
		String[] items = urlInput.getItems();
		for(int i=0; i<urlInput.getItemCount(); i++){
			if( urlInput.getItem(i).equals(url.toString()) ){
				return true;
			}
		}
		return false;
	}

	public boolean isPreviewble(IFile f) {
		String[] previewExts = StringConverter.asArray( preferenceStore.getString(WebpubUIPlugin.HT_PREVIEW_EXTENSIONS) );
		String ext = f.getFileExtension();
		for (int i = 0; i < previewExts.length; i++) {
			if (ext != null && ext.equals(previewExts[i])) {
				return true;
			}
		}
		return false;
	}

	public void browsePage(URL url) {
		available = (url != null);
		browserControl.getParent().layout();
		
		webBrowser.browsePage(url);

		String urlStr = available ? url.toString() : "";
		urlInput.setText(urlStr);
		
		if( !available ){
			setTitle(originalTitle);
		}
	}

	public boolean isAvailable() {
		return browserControl != null && !browserControl.isDisposed() && available;
	}

	public void setFocus() {
	}

	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceListener);
		linkWithEditorMgr.dispose();
		browserControl.dispose();
		super.dispose();
	}

	public AbstractWebBrowser getWebBrowser() {
		return webBrowser;
	}
	
	public void saveState(IMemento memento) {
		String[] urls = urlInput.getItems();
		StringBuffer buff = new StringBuffer();
		for (int i = 0; i < urls.length; i++) {
			buff.append( urls[i] );
			if(i != urls.length - 1){
				buff.append(' ');
			}
		}
		memento.putString(KEY_URL_HISTORY, buff.toString());
	}

}
