/*
 * $Id: WebProjectStructureBlock.java,v 1.1 2004/01/17 12:08:02 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */

package com.narucy.webpub.ui.properties;
import java.util.*;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.dialogs.ContainerSelectionDialog;

import com.narucy.webpub.core.StatusChangeListener;
import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.ui.WebpubUIPlugin;

/**
 * <p>
 * WebProjectPathStructureBlock roles to create ui components
 * for web project directory structure.
 * 
 * <p>
 * It is not depend on any container classes, only to create.
 * It used in WebProjectStructureWizardPage, WebProjectStructurePropertyPage.
 */
public class WebProjectStructureBlock {

	StatusChangeListener statusListener;
	WebProject webProj;

	Composite base;
	HashMap widgets = null;
	
	WebProject createdDescription = null;
	
	/**
	 * Listener for text modify
	 */
	ModifyListener modifyListener = new ModifyListener() {
		public void modifyText(ModifyEvent e) {
			checkValues();
		}
	};

	/**
	 * Call publish target container dialog.
	 */
	SelectionListener browseButtonListener = new SelectionListener() {
		public void widgetSelected(SelectionEvent e) {
			Object[] ks = widgets.keySet().toArray();
			for (int i = 0; i < ks.length; i++) {
				String k = (String)ks[i];
				if( getButton(k).equals(e.widget)){
					callContainerSelectionDialog(k);
				}
			}
		}

		public void widgetDefaultSelected(SelectionEvent e) {}
	};

	public WebProjectStructureBlock(StatusChangeListener statusListener) {
		this.statusListener = statusListener;
	}
	
	/**
	 * Return new Composite instance.
	 */
	public Composite createControl(Composite parent, final IProject project) throws CoreException {
		this.webProj = project != null ? (WebProject)project.getNature(WebProject.ID_NATURE) : null;
		
		base = new Composite(parent,SWT.NONE);
		
		// create widgets
		GridLayout layout = new GridLayout(2, false);
		layout.verticalSpacing = 9;
		base.setLayout(layout);

		createText(WebProject.KEY_HTSOURCES_FOLDER, "HTML source folder:");
		createText(WebProject.KEY_SCRIPTS_FOLDER, "&Script directory:");
		createText(WebProject.KEY_PUBLISH_FOLDER,  "&Publishing folder:");
		
		if( webProj != null){
			setFolderData(
				webProj.getString(WebProject.KEY_HTSOURCES_FOLDER),
				webProj.getString(WebProject.KEY_SCRIPTS_FOLDER),
				webProj.getString(WebProject.KEY_PUBLISH_FOLDER) );
		}
		// returns composite
		return base;
	}
	
	public void forceFocus(){
		getTextControl(WebProject.KEY_HTSOURCES_FOLDER).forceFocus();
	}
	
	void setPath(String key, IPath fullPath){
		
		Text text = getTextControl(key);
		if( webProj.getProject().getFullPath().isPrefixOf(fullPath) ){
			text.setText(fullPath.removeFirstSegments(1).toString());
		}else{
			text.setText(fullPath.toString());
		}
	}
	
	void callContainerSelectionDialog(String key){
		IProject project = webProj.getProject();

		// user selection link target resource
		ContainerSelectionDialog dialog =
			new ContainerSelectionDialog(
				base.getShell(),
				project,
				true,
				"Select container");
		
		if( dialog.open() == Window.OK){
			Object[] result = dialog.getResult();
			setPath(key, (IPath)result[0]);
		}
	}
	
	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}
	
	void createText(String key, String textData){
		new Label(base, SWT.NULL).setText(textData);
		
		Composite inputBase = new Composite(base, SWT.NONE);
		GridLayout gl = new GridLayout(2,false);
		gl.marginHeight = 0;
		inputBase.setLayout(gl);
		inputBase.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		Text text = new Text(inputBase, SWT.BORDER | SWT.SINGLE);
		text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		text.addModifyListener(modifyListener);
		
		Button button = new Button(inputBase, SWT.PUSH);
		button.setText("Browse...");
		button.addSelectionListener(browseButtonListener);
		
		if(widgets == null){
			widgets = new HashMap();
		}
		widgets.put(key, new Object[]{text,button});
	}
		
	public void setFolderData(String htFolderName, String scriptFolderName, String publishFolderName){
		getTextControl(WebProject.KEY_HTSOURCES_FOLDER).setText(htFolderName);
		getTextControl(WebProject.KEY_SCRIPTS_FOLDER).setText(scriptFolderName);
		getTextControl(WebProject.KEY_PUBLISH_FOLDER).setText(publishFolderName);
		
		checkValues();
	}
	
	Text getTextControl(String key){
		return (Text)((Object[])widgets.get(key))[0];
	}
	
	public String getText(String key){
		return getTextControl(key).getText();
	}
	
	public Button getButton(String key){
		return (Button)((Object[])widgets.get(key))[1];
	}

	void checkValues(){
		String htFolderName = getText(WebProject.KEY_HTSOURCES_FOLDER);
		String scriptFolderName = getText(WebProject.KEY_SCRIPTS_FOLDER);
		String publishFolderName = getText(WebProject.KEY_PUBLISH_FOLDER);

		if (publishFolderName.equals( htFolderName) ){
			updateMessage("Base HTML folder and publish folder must be different.", IStatus.ERROR);
		}else if (publishFolderName.equals(scriptFolderName) ){
			updateMessage("Script folder and publish folder must be different.", IStatus.ERROR);
		}else if(htFolderName.length() == 0){
			updateMessage("Recommend specify base HTML folder.", IStatus.WARNING);
		}else if(scriptFolderName.length() == 0){
			updateMessage("Recommend specify script folder.", IStatus.WARNING);
		}else if(htFolderName.charAt(0) == '/'){
			updateMessage("Recommends HTML source folder included to same project.", IStatus.WARNING);
		}else if(scriptFolderName.charAt(0) == '/'){
			updateMessage("Recommends script folder included to same project.", IStatus.WARNING);
		}else if(scriptFolderName.equals(htFolderName)){
			updateMessage("Recommend base HTML folder and script folder specifiy different.", IStatus.WARNING);
		}else{
			updateMessage(null, IStatus.OK);
		}
	}
	
	void updateMessage(String msg, int severity){
		IStatus status = new Status(severity, WebpubUIPlugin.ID_PLUGIN, IStatus.OK, msg != null ? msg : "", null);
		statusListener.statusChanged(status);
	}

	static Map getLanguages(){
		Locale[] locales = Locale.getAvailableLocales();
		HashMap hash = new HashMap();
		for (int i=0; i<locales.length; i++) {
			Locale locale = locales[i];
			String lang = locale.getLanguage();
			if(!hash.containsKey(lang)){
				hash.put(lang, locale.getDisplayLanguage());
			}
		}
		
		return hash;
	}

	public WebProject getCreatedDescription() {
		return createdDescription;
	}

	public void setEditable(boolean editable){
		if(widgets != null){
			Object[] keys = widgets.keySet().toArray();
			for (int i = 0; i < keys.length; i++) {
				String k = (String)keys[i];
				getTextControl(k).setEnabled(editable);
				getButton(k).setEnabled(editable);
			}
		}
	}

	public Composite getControl() {
		return base;
	}

}
