/*
 * $Id: WebProjectPage.java,v 1.1 2004/01/17 12:08:02 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.properties;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.dialogs.PropertyPage;

import com.narucy.webpub.core.*;
import com.narucy.webpub.ui.JFaceUtilities;
import com.narucy.webpub.ui.WebpubUIPlugin;


/**
 * WebProjectPage.
 */
public class WebProjectPage extends PropertyPage {
	
	WebProject webProject;

	WebProjectStructureBlock block;
	Combo charsetsCombo;
	ListViewer extsListViewer, mappedUrlViewer;
	Button addExtButton, removeExtButton, editExtButton, addUrlButton, removeUrlButton, editUrlButton;
	
	SelectionListener buttonListener = new SelectionListener() {
		public void widgetSelected(SelectionEvent e) {
			if(e.widget == addExtButton) handleAddExtension();
			if(e.widget == removeExtButton) handleRemoveExtension();
			if(e.widget == editExtButton) handleRemoveExtension();
			if(e.widget == addUrlButton) handleAddUrl();
			if(e.widget == removeUrlButton) handleRemoveUrl();
			if(e.widget == editUrlButton) handleEditUrl();
		}

		public void widgetDefaultSelected(SelectionEvent e) {
		}
	};

	/*
	 * Note: HTML extension used to template synchronization target,
	 * and inner document publish way definication
	 * (as '<?publish by=\"...\" ?>').
	 */
	protected Control createContents(Composite parent) {
		setDescription("Web Project directory structure setting");
		IProject proj = (IProject)getElement().getAdapter(IProject.class);
		webProject = (WebProject)proj.getAdapter(WebProject.class);
		
		//
		// create base widget
		//
		block = new WebProjectStructureBlock(new StatusChangeListener() {
			public void statusChanged(IStatus status) {
				int serverity = status.getSeverity();
				switch (serverity) {
					case IStatus.ERROR:
						setErrorMessage(status.getMessage());
						break;
					case IStatus.WARNING:
					case IStatus.INFO:
						setMessage(status.getMessage(), serverity);
						break;
					case IStatus.OK:
						setErrorMessage(null);
						setMessage(null, serverity);
						break;
				}
				
				updateApplyButton();
			}
		});
		
		Composite base = null;
		try {
			base = block.createControl(parent, proj);
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
			return new Label(parent, SWT.NONE);
		}
		
		// char code input controls
		new Label(base, SWT.NONE).setText("HTML char code:");
		charsetsCombo = new Combo(base, SWT.BORDER | SWT.DROP_DOWN);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalIndent = 5;
		charsetsCombo.setLayoutData(gd);
		String[] supportsCharCodes = WebpubPlugin.getSupportCharsets();
		for(int i=0; i<supportsCharCodes.length; i++){
			charsetsCombo.add(supportsCharCodes[i]);
		}
		
		// extensions widgets
		Label labe = new Label(base, SWT.NONE);
		labe.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
		labe.setText("HTML file &type extension:");
		createExtensionViewer(base)
			.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// Mapped URL definication
		labe = new Label(base, SWT.NONE);
		labe.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
		labe.setText("&Mapped URL:");
		createMappedUrlViewer(base)
			.setLayoutData(new GridData(GridData.FILL_BOTH));		
		
		// apply web project values to control
		reload();
		
		return base;
	}
	
	Control createExtensionViewer(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		
		base.setLayout(new GridLayout(2, false));
		extsListViewer = new ListViewer(base);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 3;
		extsListViewer.getControl().setLayoutData(gd);
		extsListViewer.setContentProvider(new ArrayContentProvider());
		extsListViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				boolean selectedItem = !event.getSelection().isEmpty();
				editExtButton.setEnabled(selectedItem);
				removeExtButton.setEnabled(selectedItem);
			}
		});
		
		addExtButton = createButton(base, "&Add");
		editExtButton = createButton(base, "&Edit");
		removeExtButton = createButton(base, "&Remove");
		
		return base;
	}
	
	Control createMappedUrlViewer(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(2, false));
		
		mappedUrlViewer = new ListViewer(base);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 3;
		mappedUrlViewer.getControl().setLayoutData(gd);
		mappedUrlViewer.setContentProvider(new ArrayContentProvider());
		mappedUrlViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				boolean selectedItem = !event.getSelection().isEmpty();
				editUrlButton.setEnabled(selectedItem);
				removeUrlButton.setEnabled(selectedItem);
			}
		});
		
		addUrlButton = createButton(base, "A&dd");
		editUrlButton = createButton(base, "Ed&it");
		removeUrlButton = createButton(base, "Re&move");
		
		return base;
	}
	
		Button createButton(Composite parent, String text){
		Button button = new Button(parent, SWT.PUSH);
		button.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.HORIZONTAL_ALIGN_FILL));
		button.setText(text);
		button.addSelectionListener(buttonListener);
		return button;
	}

	String inputExtension(String initialExt){
		InputDialog dialog = new InputDialog(
			getShell(),
			"File Extension",
			"Input HTML file type extension",
			initialExt,
			JFaceUtilities.FILE_EXTENSION_VALIDATOR);
		
		return (dialog.open() == InputDialog.OK) ? dialog.getValue() : null;
	}
	
	URL inputUrl(String initialUrl){
		InputDialog dialog = new InputDialog(
			getShell(),
			"URL",
			"Input project public document mapped url",
			initialUrl,
			null);
		
		return (dialog.open() == InputDialog.OK) ? toURL(dialog.getValue()) : null;
	}
	
	static URL toURL(String urlStr){
		try {
			return new URL(urlStr);
		} catch (MalformedURLException e) {
			try {
				return new File(urlStr).toURL();
			} catch (MalformedURLException e1) {
			}
		}
		return null;
	}

	void reload(){
		// init char code settings value
		String charset = webProject.getString(WebProject.KEY_HT_CHARSET);
		setCharset(charset);
		
		extsListViewer.setInput( new ArrayList(Arrays.asList(webProject.getArray(WebProject.KEY_HT_EXTENSIONS))) );
		extsListViewer.refresh();
		
		mappedUrlViewer.setInput( new ArrayList(Arrays.asList(webProject.getArray(WebProject.KEY_MAPPED_URL))) );
		mappedUrlViewer.refresh();
	}
	
	void setCharset(String charset){
		int index = charsetsCombo.indexOf(charset);
		if( index != -1){
			charsetsCombo.select(index);
		}else{
			charsetsCombo.add(charset, 0);
			charsetsCombo.select(0);
		}
	}
	
	void handleAddExtension(){
		ArrayList list = (ArrayList)extsListViewer.getInput();
		Object obj = inputExtension(null);
		if(obj != null){
			Object sel = ((IStructuredSelection)extsListViewer.getInput()).getFirstElement();
			if(sel == null){
				list.add(obj);
			}else{
				list.add(list.indexOf(sel), obj);
			}
		}
		extsListViewer.refresh();
	}
	
	void handleRemoveExtension(){
		ArrayList list = (ArrayList)extsListViewer.getInput();
		list.removeAll( ((IStructuredSelection)extsListViewer.getSelection()).toList() );
		extsListViewer.refresh();
	}
	
	void handleEditExtension(){
		ArrayList list = (ArrayList)extsListViewer.getInput();
		String sel = (String)((IStructuredSelection)extsListViewer.getSelection()).getFirstElement();
		String newExt = inputExtension(sel);
		if( newExt != null){
			int index = list.indexOf(sel);
			list.remove(index);
			list.add(index, newExt);
			extsListViewer.refresh();
		}
	}
	
	void handleAddUrl(){
		ArrayList list = (ArrayList)mappedUrlViewer.getInput();
		
		URL url = inputUrl(null);
		if(url != null){
			Object sel = ((IStructuredSelection)mappedUrlViewer.getSelection()).getFirstElement();
			if(sel == null){
				list.add(url.toString());
			}else{
				list.add(list.indexOf(sel), url.toString());
			}
			mappedUrlViewer.refresh();
		}
	}
	
	void handleRemoveUrl(){
		ArrayList list = (ArrayList)mappedUrlViewer.getInput();
		list.removeAll( ((IStructuredSelection)mappedUrlViewer.getSelection()).toList() );
		mappedUrlViewer.refresh();
	}
	
	void handleEditUrl(){
		ArrayList list = (ArrayList)mappedUrlViewer.getInput();
		String sel = (String)((IStructuredSelection)mappedUrlViewer.getSelection()).getFirstElement();
		URL newUrl = inputUrl(sel);
		
		if( newUrl != null){
			int index = list.indexOf(sel);
			list.remove(index);
			list.add(index, newUrl.toString());
			mappedUrlViewer.refresh();
		}
	}

	protected void performDefaults() {
		setCharset(webProject.getString(WebProject.KEY_HT_CHARSET));
		extsListViewer.setInput( new ArrayList(Arrays.asList(webProject.getDefaultArray(WebProject.KEY_HT_EXTENSIONS))) );
		extsListViewer.refresh();
		
		mappedUrlViewer.setInput( new ArrayList(Arrays.asList(webProject.getDefaultArray(WebProject.KEY_MAPPED_URL))) );
		mappedUrlViewer.refresh();
	}

	public boolean performOk() {
		webProject.setValue(
			WebProject.KEY_HTSOURCES_FOLDER,
			block.getText(WebProject.KEY_HTSOURCES_FOLDER));
		
		webProject.setValue(
			WebProject.KEY_SCRIPTS_FOLDER,
			block.getText(WebProject.KEY_SCRIPTS_FOLDER));

		webProject.setValue(
			WebProject.KEY_PUBLISH_FOLDER,
			block.getText(WebProject.KEY_PUBLISH_FOLDER));

		webProject.setValue(WebProject.KEY_HT_CHARSET, charsetsCombo.getText() );
		
		ArrayList exts = ((ArrayList)extsListViewer.getInput());
		webProject.setValue(WebProject.KEY_HT_EXTENSIONS, (String[])exts.toArray(new String[exts.size()]) );
		
		ArrayList urls = (ArrayList)mappedUrlViewer.getInput();
		webProject.setValue(WebProject.KEY_MAPPED_URL, (String[])urls.toArray(new String[urls.size()]) );
		
		try {
			webProject.storePreferences();
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
		
		return true;
	}

}
