/*
 * $Id: PublishDescriptorDifinicationBlock.java,v 1.1 2004/01/17 12:08:02 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.properties;

import java.io.IOException;
import java.net.URL;
import java.util.*;

import javax.xml.parsers.*;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.swt.widgets.Text;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import com.narucy.webpub.core.StatusChangeListener;
import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.ui.TableLabelProvider;
import com.narucy.webpub.ui.WebpubUIPlugin;


/**
 * 
 * @see PropertyPage
 */
public class PublishDescriptorDifinicationBlock {

	final static String
		COLKEY_PATTERN = "pattern",
		COLKEY_BY = "by",
		COLKEY_PUBLISHTO = "publishTo",
		COLKEY_ARGUMENTS = "arguments";

	static String[] specialAttributes = {
		"by", "publish_to"
	};

	static HashMap propertyNames = new HashMap();
	static {
		// set column names
		propertyNames.put( COLKEY_PATTERN, "File Match Pattern");
		propertyNames.put( COLKEY_BY, "Publish By");
		propertyNames.put( COLKEY_PUBLISHTO, "Publish To");
		propertyNames.put( COLKEY_ARGUMENTS, "Arguments");
		
		// sort attributes for binary search.
		Arrays.sort( specialAttributes );
	}
	
	static class PublishDescriptionsDocumentsFacade {
		Document doc;
		
		PublishDescriptionsDocumentsFacade(Document doc){
			this.doc = doc;
		}
		
		int getDescriptionCount(){
			return getPublishElements().length;
		}
		
		int nodeIndex(Element e){
			Element[] elems = getPublishElements();
			for(int i=0; i<elems.length; i++){
				if(elems[i].equals(e)){
					return i;
				}
			}
			return -1;
		}

		Element[] getPublishElements(){
			Element rootElem = doc.getDocumentElement();
			NodeList nodes = rootElem.getElementsByTagName("mapping");
			
			int len = nodes.getLength();
			Element[] elems = new Element[len];
			for(int i=0; i<len; i++){
				elems[i] = (Element)nodes.item(i);
			}
			return elems;
		}
		
		void moveElement(Element moveTargetElem, int moveCount){
			int index = nodeIndex(moveTargetElem);
			Element rootElem = doc.getDocumentElement();
			Element[] pubElems = getPublishElements();
			
			boolean upCmd = 0 > moveCount;
			for(int i=0, j=upCmd ? index : index+1; i<Math.abs(moveCount); i++){
				if( upCmd){
					// up
					rootElem.insertBefore(moveTargetElem, pubElems[--j] );
				}else{
					// down
					if(++j<pubElems.length){
						rootElem.insertBefore( moveTargetElem, pubElems[j] );
					}else{
						rootElem.appendChild(moveTargetElem);
					}
				}
			}
		}
		
		void removeElement(Element elem){
			Element rootElem = doc.getDocumentElement();
			rootElem.removeChild(elem);
		}
		
		void addInitialPublishDescription(){
			addInitialPublishDescription(null);
		}

		void addInitialPublishDescription(Element insertBeforeElem){
			Element rootElem = doc.getDocumentElement();
			
			Element newMappingElem = doc.createElement("mapping");
			newMappingElem.setAttribute("pattern", "**/*");
			if( insertBeforeElem != null){
				rootElem.insertBefore(newMappingElem, insertBeforeElem);
			}else{
				rootElem.appendChild(newMappingElem);
			}
		
			Element newPublishElem = doc.createElement("publish");
			newMappingElem.appendChild(newPublishElem);
			newPublishElem.setAttribute("by", PublishDescription.BY_COPY);
		}
		
		Document getDocument(){
			return doc;
		}
	
	}
	
	PublisherRegistory registory = PublisherRegistory.getInstance();

	String[] publishKeys;
	StatusChangeListener statusListener;
	
	DocumentBuilderFactory docBuilderFac = DocumentBuilderFactory.newInstance();

	Composite base;	
	TableViewer publishDescriptionViewer;
	
	Button
		addDescItemButton,
		removeDescItemButton,
		upDescItemButton,
		downDescItemButton;

	Text publishDescriptionText;

	String[] columnProps = new String[]{
		COLKEY_PATTERN,
		COLKEY_BY,
		COLKEY_PUBLISHTO,
		COLKEY_ARGUMENTS
	};
	
	IFile editTargetResource;

	public PublishDescriptorDifinicationBlock(Composite parent, IFile editTargetResource, StatusChangeListener statusListener) {
		// keys settings.
		publishKeys = registory.getPublishByKeys();
		Arrays.sort( publishKeys);
		
		this.editTargetResource = editTargetResource;
		this.statusListener = statusListener;
		
		//
		// create control.
		//
		base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(2, false) );
		
		// top label
		Label labe = new Label(base, SWT.NONE);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		labe.setLayoutData(gd);
		labe.setText("Define publisher &mappings:");
		labe = new Label(base, SWT.NONE);

		// create table viewer
		publishDescriptionViewer = new TableViewer(
			base, SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		
		// table settings
		Table table = publishDescriptionViewer.getTable();
		gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 4;
		table.setLayoutData(gd);
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// configuration providers
		publishDescriptionViewer.setColumnProperties(columnProps);
		publishDescriptionViewer.setContentProvider( new PublishDescriptionContentProvider() );
		publishDescriptionViewer.setLabelProvider( new PublishDescriptionLabelProvider() );
		publishDescriptionViewer.setCellModifier( new PublishDescriptionCellModifier() );
		publishDescriptionViewer.setCellEditors( createCellEditors() );
		
		// column settings
		for (int i = 0; i < columnProps.length; i++) {
			String key = columnProps[i];
			TableColumn col = new TableColumn(table, SWT.LEFT);
			col.setText( (String)propertyNames.get(key) );
		}

		// create buttons
		addDescItemButton = createButton( base, "&Add", new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				PublishDescriptionsDocumentsFacade facade =
					(PublishDescriptionsDocumentsFacade)publishDescriptionViewer.getInput();
				facade.addInitialPublishDescription( getSelectedElement() );
				refreshAll();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		removeDescItemButton = createButton( base, "&Remove", new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				getPublishDescriptionDocumentsFacade().removeElement( getSelectedElement() );
				refreshAll();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		upDescItemButton = createButton( base, "&Up", new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				getPublishDescriptionDocumentsFacade().moveElement( getSelectedElement(), -1);
				refreshAll();
			}

			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		downDescItemButton = createButton( base, "&Down", new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				getPublishDescriptionDocumentsFacade().moveElement( getSelectedElement(), 1);
				refreshAll();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		// create bottom label.
		new Label(base, SWT.NONE).setText("Publisher description:");
		new Label(base, SWT.NONE);
		publishDescriptionText = new Text(
			base,
			SWT.MULTI | SWT.BORDER | SWT.READ_ONLY | SWT.WRAP | SWT.V_SCROLL | SWT.H_SCROLL);
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 2;
		publishDescriptionText.setLayoutData(gd);
		
		publishDescriptionViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				refreshPublishDescriptionText();
				refreshButtonState();
			}
		});
		
		initContent();
	}

	public IFile getEditTarget() {
		return editTargetResource;
	}

	/**
	 * Return true if all of publish to use publish element attribute name.
	 * (therefore return false if publish specifed argument.)
	 */
	static boolean isSpecialAttribute(String name){
		return Arrays.binarySearch(specialAttributes, name) >= 0;
	}
	
	int getColumnIndex(String name){
		for(int i=0; i<columnProps.length; i++){
			if( columnProps[i].equals(name) ){
				return i;
			}
		}
		return -1;
	}

	class PublishDescriptionContentProvider implements IStructuredContentProvider {
		public Object[] getElements(Object inputElement) {
			PublishDescriptionsDocumentsFacade docFacade =
				(PublishDescriptionsDocumentsFacade)inputElement;
			return docFacade.getPublishElements();
		}

		public void dispose() {
			// do nothing
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			// do nothing
		}
	}
	
	class PublishDescriptionLabelProvider extends TableLabelProvider {	
		public String getColumnText(Object element, int columnIndex) {
			Element mappingElem = (Element)element;
			Element publishElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			switch (columnIndex) {
				case 0:
					return mappingElem.getAttribute("pattern");
				case 1:
					String by = publishElem.getAttribute("by");
					return by;
				case 2:
					String to = publishElem.getAttribute("publish_to");
					return to != null && to.length() > 0 ? to : "(default)";
				case 3:
					StringBuffer buff = new StringBuffer("{");
							
					NamedNodeMap attrs = publishElem.getAttributes();
					for(int i=0, j=0, len=attrs.getLength(); i<len; i++){
						Node n = attrs.item(i);
						String name = n.getNodeName();
						if( !isSpecialAttribute(name) ){
							if( j++ > 0){
								buff.append(',');
							}
							buff.append( name );
							buff.append( '=');
							buff.append( n.getNodeValue() );
						}
					}
					buff.append('}');
					return buff.toString();
			}
			return "(" + columnIndex + " " + element.getClass().toString() + ")";
		}
	
		public boolean isLabelProperty(Object element, String property) {
			return getColumnIndex(property) != -1;
		}
	}
	

	/**
	 * fix eclipse platform api missed.
	 * @see https://bugs.eclipse.org/bugs/show_bug.cgi?id=34819
	 */
	static abstract class Fix34819iDialogCellEditor extends DialogCellEditor{
		
		boolean dialogOpened = false;
		
		Fix34819iDialogCellEditor(Table table){
			super(table);
		}
		
		protected void focusLost() {
			deactivate();
		}		
		protected Button createButton(Composite parent) {
			Button dialogButton = super.createButton(parent);
			dialogButton.addListener(SWT.FocusOut, new Listener() {
				public void handleEvent(Event event) {
					if (!dialogOpened) {
						Fix34819iDialogCellEditor.this.focusLost();
						
					}
				}
			});
			return dialogButton;
		}

		final protected Object openDialogBox(Control parent) {
			dialogOpened = true;
			
			Dialog dialog = createDialog(parent);
			int res = dialog.open();
			dialogOpened = false;
			if( res == Dialog.OK){
				return getResult(dialog);
			}
			return null;
		}
		
		abstract Dialog createDialog(Control parent);
		
		abstract Object getResult(Dialog dialog);
	}
	
	class PublishDescriptionCellModifier implements ICellModifier {

		public boolean canModify(Object element, String property) {
			return getColumnIndex(property) != -1;
		}

		public Object getValue(Object element, String property) {
			Element mappingElem = (Element)element;
			if (property.equals(COLKEY_PATTERN) ){
				return mappingElem.getAttribute("pattern");
			}
			Element publishElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			if( property.equals(COLKEY_PUBLISHTO)){
				return publishElem.getAttribute("publish_to");
			}
			if( property.equals(COLKEY_BY)){
				String by = publishElem.getAttribute("by");
				int index = Arrays.binarySearch(publishKeys, by);
				return new Integer(index);
			}
			if( property.equals(COLKEY_ARGUMENTS)){
				NamedNodeMap attrs = publishElem.getAttributes();
				HashMap map = new HashMap();
				for(int i=0, len=attrs.getLength(); i<len; i++){
					Node n = attrs.item(i);
					String name = n.getNodeName();
					if( !isSpecialAttribute(name) ){
						map.put( name, n.getNodeValue() );
					}
				}
				return map;
			}
			return null;
		}

		public void modify(Object element, String property, Object value) {
			if( element instanceof TableItem){
				element = ((TableItem)element).getData();
			}
			Element mappingElem = (Element)element;
			Element pubElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			if( property.equals(COLKEY_PATTERN) ){
				mappingElem.setAttribute("pattern", (String)value);
			}else if( property.equals(COLKEY_BY)){
				int index = ((Integer)value).intValue();
				pubElem.setAttribute("by", publishKeys[index]);
			}else if( property.equals(COLKEY_PUBLISHTO)){
				pubElem.setAttribute("publish_to", (String)value);
			}else if( property.equals(COLKEY_ARGUMENTS)){
				// clear attributes
				NamedNodeMap nm = pubElem.getAttributes();
				for(int i=0; i<nm.getLength(); i++){
					Node n = nm.item(i);
					String name = n.getNodeName();
					if( !isSpecialAttribute(name)){
						pubElem.removeAttribute(name);
					}
				}
				
				// add attributes
				Map newAttributes = (Map)value;
				Object[] keys = newAttributes.keySet().toArray();
				for (int i = 0; i < keys.length; i++) {
					String name = (String)keys[i];
					pubElem.setAttribute(name, (String)newAttributes.get(name) );
				}
			}
			publishDescriptionViewer.refresh();
		}
	}
	
	CellEditor[] createCellEditors(){
		Table table = publishDescriptionViewer.getTable();
		
		CellEditor[] editors = new CellEditor[ columnProps.length ];
		for(int i=0; i<columnProps.length; i++){
			String prop = columnProps[i];
			CellEditor ed;
			if( prop.equals(COLKEY_PATTERN) ){
				ed = new Fix34819iDialogCellEditor(table) {
					Dialog createDialog(Control c) {
						String pattern = (String)doGetValue();
						return new ResourceMatchDialog(
							c.getShell(),
							pattern,
							editTargetResource.getParent() );
					}

					Object getResult(Dialog dialog) {
						return ((ResourceMatchDialog)dialog).getFilter();
					}
				};
			}else if(prop.equals(COLKEY_BY) ){
				ed = new ComboBoxCellEditor(table, publishKeys);
			}else if(prop.equals(COLKEY_PUBLISHTO)){
				ed = new TextCellEditor(table);
			}else if(prop.equals(COLKEY_ARGUMENTS)){
				ed = new Fix34819iDialogCellEditor(table) {
					Dialog createDialog(Control c) {
						return new MapEditDialog(
							c.getShell(),
							(Map)doGetValue() );
					}

					Object getResult(Dialog dialog) {
						return doGetValue();
					}
				};
			}else{;
				ed = null;
			}
			editors[i] = ed;
		}
		return editors;
	}
	
	
	void refreshAll(){
		publishDescriptionViewer.refresh();
		refreshButtonState();
		refreshPublishDescriptionText();
	}
	
	void refreshButtonState(){
		boolean enable = publishDescriptionViewer.getTable().getEnabled();
		Element selectedElem = getSelectedElement();
		PublishDescriptionsDocumentsFacade docFacade = getPublishDescriptionDocumentsFacade();
		int index = docFacade.nodeIndex(selectedElem);
		
		addDescItemButton.setEnabled(enable);
		removeDescItemButton.setEnabled(enable && selectedElem != null);
		
		upDescItemButton.setEnabled(enable && index >= 1);
		downDescItemButton.setEnabled(enable && index < docFacade.getDescriptionCount()-1);
	}
	
	void refreshPublishDescriptionText(){
		Element mappingElem = getSelectedElement();
		if( mappingElem != null){
			Element publishElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			String by = publishElem.getAttribute("by");
				
			String description = registory.getPublishDescription(by);
			publishDescriptionText.setText(
				description != null ?
					description :
					"Specify publisher (" + by + ") is not exist description.");
		}else{
			publishDescriptionText.setText("");
		}
	}
	
	PublishDescriptionsDocumentsFacade getPublishDescriptionDocumentsFacade(){
		return (PublishDescriptionsDocumentsFacade)publishDescriptionViewer.getInput();
	}
	
	Element getSelectedElement(){
		IStructuredSelection sel =
			(IStructuredSelection)publishDescriptionViewer.getSelection();
		return !sel.isEmpty() ? (Element)sel.getFirstElement() : null;
	}
	
	static Button createButton(Composite parent, String title, SelectionListener listener){
		Button button = new Button(parent, SWT.PUSH);
		button.setText(title);
		button.setLayoutData( new GridData(
			GridData.HORIZONTAL_ALIGN_FILL |
			GridData.VERTICAL_ALIGN_BEGINNING));
		button.addSelectionListener(listener);
		return button;
	}

	/**
	 * create document object from edit target publish descriptjion file
	 * and set document object to ui widgets.
	 */
	void initContent(){
		// create publisher builder document object
		// and error handling.
		String errorMessage = null;
		try{
			Document doc;
			DocumentBuilder docBuilder = docBuilderFac.newDocumentBuilder();
			if( editTargetResource.exists() ){
				doc = docBuilder.parse(editTargetResource.getContents());
			}else{
				URL url = WebpubUIPlugin.getResource("resources/inital_publish_description");
				doc = docBuilder.parse(url.openStream() );
			}
			publishDescriptionViewer.setInput( new PublishDescriptionsDocumentsFacade(doc) );
		} catch (SAXException e) {
			errorMessage = "configuration file \".publish\" is invalid.";
		} catch (IOException e) {
			errorMessage = "can not read \".publish\".";
		} catch (ParserConfigurationException e) {
			errorMessage = "publish file read error that is parser configuration exception.";
			WebpubUIPlugin.handleException(e);
		} catch (CoreException e) {
			errorMessage = "can not found publish configuration file \".publish\", (that file is not synchronize in workbench resource, or that file is not exist or not local)";
			WebpubUIPlugin.handleException(e);
		}
		
		Table table = publishDescriptionViewer.getTable();
		for (int i = 0, len = table.getColumnCount(); i < len; i++) {
			TableColumn col = table.getColumn(i);
			col.pack();
			if( col.getWidth() > 100){
				col.setWidth(100);
			}
		}

		boolean enabled = (errorMessage == null);
		setEnabled(enabled);
		if( statusListener != null ){
			if( !enabled){
				statusListener.statusChanged( createStatus( IStatus.ERROR, errorMessage) );
			}else{
				statusListener.statusChanged( createStatus( IStatus.OK, "") );
			}
		}
	}
	
	static IStatus createStatus(int severity, String message){
		return new Status(
			severity,
			WebpubUIPlugin.ID_PLUGIN,
			severity,
			message,
			null);
	}
	
	public void setEnabled(boolean bool){
		Control[] controls = base.getChildren();
		for (int i = 0; i < controls.length; i++) {
			controls[i].setEnabled(bool);
		}
	}

	/**
	 * Store the edited document instance.
	 */
	public boolean store() throws CoreException {
		PublishPropertyStore storer = new PublishPropertyStore();
		Document doc = getPublishDescriptionDocumentsFacade().getDocument();
		
		storer.store( editTargetResource.getLocation().toFile(), doc);
		editTargetResource.refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor() );
		return true;
	}

	public Composite getControl() {
		return base;
	}

}
