/*
 * $Id: MapEditDialog.java,v 1.1 2004/01/17 12:08:02 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.properties;

import java.util.Arrays;
import java.util.Map;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;

import com.narucy.webpub.ui.TableLabelProvider;

/**
 * MapEditDialog roles to edit given Map instance.
 */
class MapEditDialog extends Dialog {

	final static String[] propNames = { "Key", "Value" };

	Map editTarget;
	String[] listProperties = new String[]{ "key", "value" };

	int initialWidth = 400, initialHeight = 300;
	TableViewer tableViewer;
	Button addButton, removeButton;
	
	public MapEditDialog(Shell shell, Map editTarget) {
		super(shell);
		setShellStyle(getShellStyle() | SWT.MAX | SWT.RESIZE);
		this.editTarget = editTarget;
	}
	
	public Control createDialogArea(Composite parent) {
		Composite base = (Composite)super.createDialogArea(parent);
		base.setLayout(new GridLayout(2, false));
		
		new Label(base,SWT.NONE).setText("Maps keys to values:");
		new Label(base, SWT.NONE);
		
		// create table
		tableViewer = new TableViewer(
			base, SWT.BORDER | SWT.FULL_SELECTION | SWT.V_SCROLL | SWT.H_SCROLL);

		// innter table setting
		Table table = tableViewer.getTable();
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 2;
		table.setLayoutData(gd);
		for(int i=0; i<listProperties.length; i++){
			TableColumn col = new TableColumn(table, SWT.LEFT);
			col.setText(propNames[i]);
		}
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		
		// cell settings,
		tableViewer.setColumnProperties(listProperties);
		
		// content provider settign
		tableViewer.setContentProvider(new IStructuredContentProvider() {
			public Object[] getElements(Object inputElement) {
				Object[] keys = editTarget.keySet().toArray();
				Arrays.sort(keys);
				
				Object[] entryDatas = new Object[keys.length];
				for (int i = 0; i < keys.length; i++) {
					String k = (String)keys[i];
					entryDatas[i] = new String[]{k, (String)editTarget.get(k) };
				}
				return entryDatas;
			}
			public void dispose() {}
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {}
		});
		
		// label provider settings.
		tableViewer.setLabelProvider(new TableLabelProvider() {
			public String getColumnText(Object element, int columnIndex) {
				return ((String[])element)[columnIndex];
			}
		});
		
		// cell modifier settings
		tableViewer.setCellModifier(new ICellModifier() {
			public boolean canModify(Object element, String property) {
				return true;
			}

			public Object getValue(Object element, String property) {
				String[] kv = (String[])element;
				return kv[ property.equals("key") ? 0 : 1 ];
			}

			public void modify(Object element, String property, Object value) {
				if (element instanceof Item) {
					element = ((Item) element).getData();
				}
				String[] kv = (String[])element;
				if( property.equals("key") ){
					editTarget.remove(kv[0]);
					editTarget.put((String)value, kv[1]);
				}else{
					editTarget.put(kv[0], (String)value);
				}
				tableViewer.refresh();
			}
		});
		tableViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				refreshButtonStatus();
			}
		});
		
		// input data
		tableViewer.setInput(editTarget);
		
		// sets column size
		// minimun width to define refer initial width size. (initial width divide by three).
		TableColumn[] columns = table.getColumns();
		for (int i = 0, minSize = initialWidth / 3; i < columns.length; i++) {
			TableColumn column = columns[i];
			column.pack();
			if( column.getWidth() < minSize ){
				column.setWidth(minSize);
			}
		}
		TextCellEditor editor = new TextCellEditor(table);
		tableViewer.setCellEditors(new CellEditor[]{ editor, editor });
		
		// set buttons
		addButton = createButton( base, "&Add",new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				callAddDialog();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		removeButton = createButton( base, "&Remove", new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				editTarget.remove( getCurrentSelection()[0] );
				tableViewer.refresh();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		refreshButtonStatus();
		return base;
	}
	
	static Button createButton(Composite base, String text, SelectionListener listener){
		Button button = new Button(base, SWT.PUSH);
		button.setLayoutData(
			new GridData(
				GridData.HORIZONTAL_ALIGN_FILL |
				GridData.VERTICAL_ALIGN_BEGINNING) );
		button.setText(text);
		button.addSelectionListener(listener);
		
		return button;
	}
	
	void refreshButtonStatus(){
		String[] value = getCurrentSelection();
		removeButton.setEnabled(value != null);
	}
	
	String[] getCurrentSelection(){
		IStructuredSelection sel = (IStructuredSelection)tableViewer.getSelection();
		return (String[])sel.getFirstElement();
	}
	
	void callAddDialog(){
		String newKey = null;
		InputDialog dialog = new InputDialog(getShell(), "New Item Entry", "Input key of new item", null, null);
		if( dialog.open() ==  Window.OK){
			newKey = dialog.getValue();
			if( newKey.length() > 0){
				editTarget.put(newKey, "");
				tableViewer.refresh();
			}
		}
	}

	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("Edit Key/Value");
	}

	protected Point getInitialSize() {
		return new Point(initialWidth, initialHeight);
	}

}
