package com.narucy.webpub.ui.preferences;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

import com.narucy.webpub.core.InterpreterCheckResult;
import com.narucy.webpub.core.InterpreterCheckerManager;
import com.narucy.webpub.ui.WebpubUIPlugin;

/**
 * RubyInterpreterPreferencePage
 */
public class RubyInterpreterPreferencePage extends PreferencePage implements IWorkbenchPreferencePage {

	final static String[] COLUMNS = {
		"Name", "Message", "State"
	};
	
	InterpreterCheckerManager checkerManager = InterpreterCheckerManager.getInstance();
	
	Button interpreterBrowseButton, useRdtInterpreterButton;
	Text interpreterLocation;
	
	Table infoTable;
	Text infoNote;
	
	protected Control createContents(Composite parent) {
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		base.setLayout(new GridLayout(1, false));
		
		createInterpreterSelectionControl(base);
		
		new Label(base, SWT.NONE).setText("Ruby infomation:");
		
		// create ruby infomation state.
		infoTable = new Table(base, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION);
		for(int i=0; i<COLUMNS.length; i++){
			TableColumn col = new TableColumn(infoTable, SWT.LEFT);
			col.setText(COLUMNS[i]);
		}
		infoTable.setLinesVisible(true);
		infoTable.setHeaderVisible(true);
		infoTable.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				TableItem item = (TableItem)e.item;
				String text = item.getText();
				String desc = checkerManager.getDescription(text);
				setNoteText(desc);
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.heightHint = 100;
		infoTable.setLayoutData(gd);
		
		// create note
		infoNote = new Text(base, SWT.READ_ONLY | SWT.NO_FOCUS | SWT.MULTI | SWT.READ_ONLY | SWT.WRAP);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.heightHint = 100;
		infoNote.setLayoutData(gd);
		
		// initial values
		load();
		
		return base;
	}
	
	/**
	 * Create ruby interperter setting controls.
	 */
	Composite createInterpreterSelectionControl(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(3, false));
		base.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		new Label(base, SWT.NONE).setText("Interpreter:");
		
		interpreterLocation = new Text(base, SWT.BORDER);
		interpreterLocation.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		interpreterLocation.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				interpreterCheck();
			}
		});
		
		interpreterBrowseButton = new Button(base, SWT.PUSH);
		interpreterBrowseButton.setText("&Browse...");
		interpreterBrowseButton.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				callInterperterBrowseDialog();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		interpreterBrowseButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		
		new Label(base, SWT.NONE);
		new Label(base, SWT.NONE);
		
		useRdtInterpreterButton = new Button(base, SWT.PUSH);
		useRdtInterpreterButton.setText("Use &RDT Settings");
		useRdtInterpreterButton.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				useRdtInterpreterLocation();
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});
		
		useRdtInterpreterButton.setEnabled(WebpubUIPlugin.getDefault().getRDTSelectedInterpreterLocation() != null);
		useRdtInterpreterButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		
		return base;
	}

	void callInterperterBrowseDialog(){
		FileDialog dialog = new FileDialog(getShell(), SWT.OPEN);
		
		dialog.setFilterNames(new String[]{"Ruby Interpreter"});
		dialog.setFilterExtensions(
			System.getProperty("os.name").indexOf("Windows") != -1 ?
				new String[]{"ruby.exe"} :
				new String[]{"ruby"});
		
		String file = dialog.open();
		if( file != null ){
			interpreterLocation.setText(file);
		}
	}

	void useRdtInterpreterLocation(){
		String path = WebpubUIPlugin.getDefault().getRDTSelectedInterpreterLocation();
		if( path != null){
			interpreterLocation.setText(path);
		}
	}

	void setNoteText(String text){
		infoNote.setText( text != null ? "Note: " + text : "");
	}
	
	IRunnableWithProgress checkProc = new IRunnableWithProgress(){
		public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException{
			try{
				setErrorMessage(null);
				infoTable.removeAll();
				String path = interpreterLocation.getText();
				try {
					InterpreterCheckResult result =
						checkerManager.check(path);
					String[] keys = result.getKeys();
					for (int i = 0; i < keys.length; i++) {
						String k = keys[i];
						String[] r = { k, result.getMessage(k), toStateString(result.getState(k))};
						new TableItem(infoTable, SWT.NONE).setText(r);
					}
				} catch (CoreException e) {
					e.printStackTrace();
					setErrorMessage(e.getLocalizedMessage());
				}
		
				TableColumn[] cols = infoTable.getColumns();
				for (int i = 0; i < cols.length; i++) {
					cols[i].pack();
				}
			}
			finally{
				monitor.done();
			}
		}
	};
	
	void interpreterCheck() {
		try {
			new ProgressMonitorDialog(getShell()).run(false, true, checkProc);
		} catch (InvocationTargetException e) {
			WebpubUIPlugin.handleException(e);
		} catch (InterruptedException e) {
		}
	}
	
	static String toStateString(int v){
		switch (v) {
			case NONE:
				return "OK";
			case INFORMATION:
				return "Infomation";
			case WARNING:
				return "Warning";
			case ERROR:
				return "Error";
			default :
				throw new IllegalArgumentException("Invalid value:" + v);
		}
	}

	public boolean performOk() {
		IPreferenceStore store = getPreferenceStore();
		String path = interpreterLocation.getText();
		store.setValue(WebpubUIPlugin.RUBY_INTERPRETER_LOCATION, path);
		
		return true;
	}

	public void init(IWorkbench workbench) {
		setPreferenceStore(WebpubUIPlugin.getDefault().getPreferenceStore());
	}

	protected void performDefaults() {
		IPreferenceStore store = getPreferenceStore();
		interpreterLocation.setText(
			store.getDefaultString(WebpubUIPlugin.RUBY_INTERPRETER_LOCATION) );
	}
	
	void load(){
		IPreferenceStore store = getPreferenceStore();
		interpreterLocation.setText(
			 store.getString(WebpubUIPlugin.RUBY_INTERPRETER_LOCATION) );
	}
	

}
