/*
 * $Id: TocEditBlock.java,v 1.1 2004/01/17 12:08:08 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.editors.toc;

import java.util.*;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.internal.dialogs.ResourceSorter;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;

import com.narucy.webpub.core.toc.*;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.views.TocContentProvider;

/**
 * 
 */
public class TocEditBlock {

	Cursor cursor;

	TreeViewer pubFolderTree, tocTree;

	Button
		addButton, removeButton,
		addAllButton, removeAllButton,
		upButton, leftButton, rightButton, downButton,
		createEmptyButton;

	Composite base, buttonBase;

	Toc toc = null;
	IContainer publishFolder;
	TocController controller;
	
	TocListener tocListener = new TocListener() {
		public void topicChanged(TocEvent[] events) {
			base.getDisplay().asyncExec(new Runnable() {
				public void run() {
					if(!base.isDisposed()){
						tocTree.refresh();
						refreshButtonState();
					}
				}
			});
		}
	};
	
	SelectionListener buttonListener = new SelectionListener() {
		public void widgetSelected(SelectionEvent e) {
			Object[] expanded = tocTree.getExpandedElements();
			
			if(e.widget == addButton) handleAddButton();
			if(e.widget == removeButton) handleRemoveButton();
			if(e.widget == addAllButton) handleAddAllButton();
			if(e.widget == removeAllButton) handleRemoveAllButton();
			if(e.widget == upButton) handleUpButton();
			if(e.widget == leftButton) handleLeftButton();
			if(e.widget == rightButton) handleRightButton();
			if(e.widget == downButton) handleDownButton();
			if(e.widget == createEmptyButton) handleCreateEmptyButton();
			
			toc.fireChanged();
			
			// expand node
			Topic[] ts = toc.getAllTopics();
			for (int i = 0; i < ts.length; i++) {
				Topic t = ts[i];
				for (int j=0; j<expanded.length; j++) {
					if(t.equals(expanded[j])){
						tocTree.expandToLevel(t, 1);
					}
				}
			}
		}
		public void widgetDefaultSelected(SelectionEvent e) {}
	};
	
	public TocEditBlock(Composite parent) {
		base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(3, false));
		
		cursor = new Cursor(base.getDisplay(), SWT.CURSOR_SIZEWE);
		
		new Label(base, SWT.NONE).setText("Resources");
		new Label(base, SWT.NONE);
		new Label(base, SWT.NONE).setText("Sitemap");
		
		// create publish folder tree.
		pubFolderTree = new TreeViewer(base);
		pubFolderTree.getControl().setLayoutData(new GridData(GridData.FILL_VERTICAL));
		pubFolderTree.setLabelProvider(new WorkbenchLabelProvider());
		pubFolderTree.setContentProvider(new WorkbenchContentProvider());
		pubFolderTree.setSorter(new ResourceSorter(ResourceSorter.TYPE));
		pubFolderTree.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				refreshButtonState();
			}
		});
		
		// create buttns
		buttonBase = new Composite(base, SWT.NONE);
		GridLayout gl = new GridLayout(1, false);
		gl.marginHeight = 4;
		buttonBase.setLayout(gl);
		
		addButton = createButton(">");
		removeButton = createButton("<");
		addAllButton = createButton(">>");
		removeAllButton = createButton("<<");
		
		new Label(buttonBase, SWT.NONE);
		upButton = createButton("&Up");
		leftButton = createButton("&Left");
		rightButton = createButton("&Right");
		downButton = createButton("&Down");

		new Label(buttonBase, SWT.NONE);
		createEmptyButton = createButton("&Create Empty");

		// create toc tree.		
		tocTree = new TreeViewer(base);
		tocTree.getControl().setLayoutData(new GridData(GridData.FILL_BOTH));
		tocTree.setLabelProvider(new WorkbenchLabelProvider(){
			public String decorateText(String text, Object e){
				String href = ((Topic)e).getHref();
				return (href != null) ?
					text + " (" + href + ")" : text;
			}
		});
		tocTree.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				refreshButtonState();
			}
		});
		tocTree.setContentProvider(new TocContentProvider());
		tocTree.getControl().addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				if(toc != null){
					toc.removeTocListener(tocListener);
					cursor.dispose();
				}
			}
		});

		base.addMouseListener(new MouseListener() {
			public void mouseDoubleClick(MouseEvent e) {
			}
			
			int bx;

			public void mouseDown(MouseEvent e) {
				bx = e.x;
				base.setCursor(cursor);
			}

			public void mouseUp(MouseEvent e) {
				Control control = pubFolderTree.getControl();
				GridData gd = (GridData)control.getLayoutData();
				if(gd.widthHint == SWT.DEFAULT){
					gd.widthHint = control.getSize().x;
				}
				gd.widthHint += (e.x - bx);
				base.layout();
				base.setCursor(null);
			}
		});
		
		setEnable(false);
	}
	
	public void setInput(IContainer pubFolder, Toc t){
		if(toc == t || (t != null ? t.equals(toc) : toc.equals(t)) ){
			return;
		}
		
		if(toc != null){
			toc.removeTocListener(tocListener);
		}
		
		publishFolder = pubFolder;
		toc = t;
	
		if(publishFolder != null && toc != null){
			pubFolderTree.setInput(pubFolder);
			tocTree.setInput(toc);
			tocTree.expandAll();
			
			controller = new TocController(pubFolder, toc);
			toc.addTocListener(tocListener);
			
			setEnable(true);
			refreshButtonState();
		}else{
			setEnable(false);
		}
	}
	
	void setEnable(boolean b){
		notifyEnable(base, b);
	}
	
	static void notifyEnable(Composite p, boolean b){
		Control[] children = p.getChildren();
		for (int i = 0; i < children.length; i++) {
			Control control = children[i];
			control.setEnabled(b);
			if(control instanceof Composite){
				notifyEnable((Composite)control, b);
			}
		}
	}
	
	void refreshButtonState(){
		// add, remove button refresh.
		IResource[] resources = getSelectedResources();
		addButton.setEnabled( resources.length == 0 ? false : !controller.isAlreadyAdded(resources) );
		addAllButton.setEnabled( !controller.isAllResourcesAdded() );
		
		Topic[] topics = getSelectedTopics();
		removeButton.setEnabled(topics.length > 0);
		removeAllButton.setEnabled(toc.hasSubtopics());

		boolean selected = topics.length > 0;
		createEmptyButton.setEnabled(selected);
		
		boolean
			up = selected,
			left = selected,
			right = selected,
			down = selected;
		
		if(selected){
			for (int i = 0; i < topics.length; i++) {
				if(topics[i].getNextTopic(false) == null){
					down = false;
				}
				if(topics[i].getPrevTopic(false) == null){
					up = false;
					right = false;
				}
				if( topics[i].getParent() instanceof Toc){
					left = false;
				}
			}
		}
		upButton.setEnabled(up);
		leftButton.setEnabled(left);
		rightButton.setEnabled(right);
		downButton.setEnabled(down);
	}

	Button createButton(String name){
		Button button = new Button(buttonBase, SWT.FLAT);
		button.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_END));
		button.setText(name);
		button.addSelectionListener(buttonListener);
		return button;
	}

	IResource[] getSelectedResources(){
		List l = ((IStructuredSelection) pubFolderTree.getSelection()).toList();
		return (IResource[])l.toArray(new IResource[l.size()]);
	}

	Topic[] getSelectedTopics(){
		List l = ((IStructuredSelection)tocTree.getSelection()).toList();
		return (Topic[])l.toArray( new Topic[l.size()]);
	}
	
	protected void handleCreateEmptyButton() {
		InputDialog dialog = new InputDialog(
			base.getShell(),
			"New Node",
			"Create empty node",
			null,
			null);
		
		if(dialog.open() == Window.OK){
			createEmptyNode(dialog.getValue());
		}
	}

	void createEmptyNode(String name){
		Topic[] ts = getSelectedTopics();
		Topic emptyNode = (ts.length > 0 ? ts[0] : toc).createTopic(null, name);
		tocTree.setSelection(new StructuredSelection(emptyNode));
	}

	protected void handleDownButton() {
		Topic[] topics = getSelectedTopics();
		for (int i = topics.length-1; i >= 0; i--) {
			topics[i].moveDown();
		}
	}

	protected void handleLeftButton(){
		Topic[] topics = getSelectedTopics();
		for (int i = 0; i < topics.length; i++) {
			topics[i].levelDown();
		}
	}
	
	protected void handleRightButton(){
		Topic[] topics = getSelectedTopics();
		for (int i = 0; i < topics.length; i++) {
			topics[i].levelUp();
		}
	}
	
	protected void handleUpButton() {
		Topic[] topics = getSelectedTopics();
		for (int i = 0; i < topics.length; i++) {
			topics[i].moveUp();
		}
	}

	protected void handleRemoveAllButton() {
		toc.removeSubtopics();
	}

	protected void handleRemoveButton() {
		Topic[] removeTopics = getSelectedTopics();
		Topic nextSelection = controller.findNextSelectionTopic(removeTopics);
		
		for(int i=0; i<removeTopics.length; i++){
			Topic t = removeTopics[i];
			t.getParent().removeTopic(t);
		}
		if(nextSelection != null){
			tocTree.setSelection(new StructuredSelection(nextSelection));
		}
	}

	protected void handleAddButton() {
		try {
			invokeInsertResourceToToc(getSelectedResources());
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
	}

	protected void handleAddAllButton() {
		try {
			invokeInsertResourceToToc(publishFolder.members());
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
	}

	void invokeInsertResourceToToc(IResource[] resources) throws CoreException{
		Topic[] topics = getSelectedTopics();
		
		ArrayList added = new ArrayList();
		Topic parent = topics.length > 0 ? topics[0] : toc;
		for (int i = 0; i < resources.length; i++) {
			added.addAll( Arrays.asList(controller.insertResourceToToc(resources[i], parent)) );
		}
		tocTree.setSelection(new StructuredSelection(added));
	}
	
	public void setBackground(Color color){
		base.setBackground(color);
		buttonBase.setBackground(color);
		Control[] controls = base.getChildren();
		for (int i = 0; i < controls.length; i++) {
			Control c = controls[i];
			if(c instanceof Label){
				((Label)c).setBackground(color);
			}
		}
	}
	
	public Composite getControl() {
		return base;
	}

	public TreeViewer getTocTree() {
		return tocTree;
	}

}