 /*
 * $Id: HTEditor.java,v 1.1 2004/01/17 12:08:03 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.editors.html;

import java.util.Arrays;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.text.*;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.texteditor.*;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.template.Template;
import com.narucy.webpub.core.template.TemplateParser;
import com.narucy.webpub.ui.WebpubUIPlugin;
import com.narucy.webpub.ui.editors.ColorManager;

/**
 * HTMLEditor.
 */
public class HTEditor extends TextEditor implements IPropertyChangeListener {

	final public static String
		ID_KEYBIND_SCOPE = "com.narucy.webpub.ui.editors.html.HTEditorScope";

	final static String[] relationalPreferenceKeys = {
		WebpubUIPlugin.HT_EDITOR_COLOR_DEFAULT,
		WebpubUIPlugin.HT_EDITOR_COLOR_TAG,
		WebpubUIPlugin.HT_EDITOR_COLOR_COMMENT,
		WebpubUIPlugin.HT_EDITOR_COLOR_ERB,
		WebpubUIPlugin.HT_EDITOR_COLOR_ATTR,
		WebpubUIPlugin.HT_EDITOR_COLOR_PROCESSOR,
	};
	
	static {
		Arrays.sort(relationalPreferenceKeys);
	}

	ColorManager colorManager;
	HTDocumentProvider docProvider;
	HTOutlinePage outlinePage = null;

	public HTEditor() {
		super();
		setKeyBindingScopes(new String[]{ID_KEYBIND_SCOPE});

		// initialize instance values.
		colorManager = new ColorManager();
		// init preferenancesss
		IPreferenceStore prefStore = WebpubUIPlugin.getDefault().getPreferenceStore();
		prefStore.addPropertyChangeListener(this);
		setPreferenceStore(prefStore);
		
		WorkbenchChainedTextFontFieldEditor.startPropagate(prefStore, JFaceResources.TEXT_FONT);
		
		// configration editor
		HTViewerConfiguration conf = new HTViewerConfiguration(colorManager);
		setSourceViewerConfiguration(conf);
		
		setRangeIndicator(new DefaultRangeIndicator());

		docProvider = new HTDocumentProvider();
		setDocumentProvider(docProvider);
	}

	public Object getAdapter(Class required) {
		if (IContentOutlinePage.class.equals(required)) {
			return getOutlinePage();
		}
		return super.getAdapter(required);
	}
	
	HTOutlinePage getOutlinePage(){
		if (outlinePage == null) {
			outlinePage = new HTOutlinePage(getSourceViewer().getDocument());
			outlinePage.addSelectionChangedListener(new ISelectionChangedListener() {
				public void selectionChanged(SelectionChangedEvent event) {
					outlineSelectionChanged(event);
				}
			});
		}
		return outlinePage;
	}

	public void propertyChange(PropertyChangeEvent event) {
		HTViewerConfiguration conf = (HTViewerConfiguration)getSourceViewerConfiguration();
		if( conf != null){
			String prop = event.getProperty();
			IPreferenceStore store = getPreferenceStore();

			StyledText styledText = getSourceViewer().getTextWidget();
			if(prop.equals(WebpubUIPlugin.HT_EDITOR_DISPLAYED_TAB_WIDTH)){
				styledText.setTabs(store.getInt(prop));
			}else if(prop.equals(WebpubUIPlugin.HT_EDITOR_WORDWRAP)){
				styledText.setWordWrap(store.getBoolean(prop));
			}else if(prop.equals(WebpubUIPlugin.HT_EDITOR_COLOR_BACKGROUND)){
				styledText.setBackground( colorManager.getColor(prop));
			}else {
				conf.propertyChange(event);
			}
		}
	}
	
	void outlineSelectionChanged(SelectionChangedEvent event){
		ISelection selection = event.getSelection();
		if(selection instanceof IStructuredSelection){
			IStructuredSelection sel = (IStructuredSelection)selection;
			Object obj = sel.getFirstElement();
			if(obj instanceof Template){
				Template tmpl = (Template)obj;
				Integer lineNumber =
					(Integer)tmpl.getProperty( TemplateParser.FIND_LINE_NUMBER);
				
				if( lineNumber != null ){
					int lineNum = lineNumber.intValue();
					jumpLine(lineNum);
				}
			}
		}
	}
	
	void jumpLine(int lineNum){
		IDocument doc = getSourceViewer().getDocument();
		if( doc != null){
			try {
				int offset = doc.getLineOffset(lineNum);
				TextSelection textSelection = new TextSelection(offset, 0);
				getSelectionProvider().setSelection(textSelection);
			} catch (BadLocationException e) {
				WebpubUIPlugin.handleException(e);
			}
		}
	}

	protected boolean affectsTextPresentation(PropertyChangeEvent event) {
		return Arrays.binarySearch(relationalPreferenceKeys, event.getProperty()) >= 0;
	}

	protected void createActions() {
		super.createActions();
		installEncodingSupport();
		IEditorInput input = getEditorInput();
		if(input instanceof IFileEditorInput){
			try{
				WebProject wp = (WebProject)((IFileEditorInput)input).getFile().getProject().getNature(WebProject.ID_NATURE);
				if(wp != null){
					String charset = wp.getString(WebProject.KEY_HT_CHARSET);
					fEncodingSupport.setEncoding(charset);
				}
			}catch(CoreException e){
				WebpubUIPlugin.handleException(e);
			}
		}

		ContentAssistAction action = new ContentAssistAction(
			WebpubUIPlugin.getDefault().getResourceBundle(),
			"ContentAssistProposal.",
			this);

		// This action definition is associated with the accelerator Ctrl+Space
		action.setActionDefinitionId(ITextEditorActionDefinitionIds.CONTENT_ASSIST_PROPOSALS);
		setAction("ContentAssistProposal", action);
	}

	public void dispose() {
		IPreferenceStore prefStore = getPreferenceStore();
		prefStore.removePropertyChangeListener(this);
		colorManager.dispose();
		super.dispose();
	}

}
