/*
 * $Id: LinkCheckConfigurationTab.java,v 1.3 2004/01/17 11:50:30 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.linkcheck.ui;

import java.util.*;
import java.util.List;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;

import com.narucy.webpub.debug.WebpubDebugPlugin;
import com.narucy.webpub.linkcheck.core.LinkCheckConstants;

/**
 * LinkCheckLaunchConfigurationTab.
 */
class LinkCheckConfigurationTab extends AbstractLaunchConfigurationTab implements LinkCheckConstants{

	final static ArrayList EMPTY_LIST = new ArrayList();

	final static String[] ORIGINAL_CHECKERS = {
		"http", "ftp",
	};

	ArrayList ignoreRegexps = new ArrayList(), checkSchemes = new ArrayList();

	Image iconImage = null;
	
	ListViewer ignoreRegexpViewer;
	CheckboxTableViewer schemeViewer;

	Button addRegexpButton, removeRegexpButton, addCheckerButton, removeCheckerButton, backgroundButton;

	SelectionListener listener = new SelectionListener() {
		public void widgetSelected(SelectionEvent e) {
			if(e.widget == addRegexpButton) handleAddRegexp();
			if(e.widget == removeRegexpButton) handleRemoveRegexp();
			if(e.widget == addCheckerButton) handleAddChecker();
			if(e.widget == removeCheckerButton) handleRemoveChecker();
		}

		public void widgetDefaultSelected(SelectionEvent arg0) {
		}
	};
	
	public LinkCheckConfigurationTab(){
		checkSchemes.addAll(Arrays.asList(ORIGINAL_CHECKERS));
	}

	public void createControl(Composite parent) {
		// create icon
		iconImage = WebpubDebugPlugin.getImageDescriptor("ctool16/link_checker.gif").createImage();
		
		// create base
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(1, false));

		backgroundButton = new Button(base, SWT.CHECK);
		backgroundButton.setText("&Background");
		
		new Label(base, SWT.NONE).setText("&Check Schemes:");
		createCheckSchemesSelectionControl(base)
			.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		new Label(base, SWT.NONE).setText("&Ignore Regexps:");
		createIgnoreRegexpsControl(base)
			.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		Button clearMarkerButton = new Button(base, SWT.PUSH);
		clearMarkerButton.setText("&Clear Link Check Markers");
		clearMarkerButton.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent arg0) {
				deleteAllMarkers();
			}

			public void widgetDefaultSelected(SelectionEvent arg0) {}
		});
		
		schemeViewer.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(CheckStateChangedEvent e) {
				getLaunchConfigurationDialog().updateButtons();
			}
		});
		
		// create
		setControl(base);
	}
	
	void deleteAllMarkers(){
		try{
			ResourcesPlugin.getWorkspace().run(new IWorkspaceRunnable() {
				public void run(IProgressMonitor monitor) throws CoreException {
					IMarker[] mks = ResourcesPlugin.getWorkspace().getRoot().findMarkers(null, false, IResource.DEPTH_INFINITE);
					for (int i = 0; i < mks.length; i++) {
						IMarker m = mks[i];
						if(Boolean.TRUE.equals(m.getAttribute(ID_LINKCHECK_MARKER))){
							m.delete();
						}
					}
				}
			}, new NullProgressMonitor());
		}catch(CoreException e){
			WebpubDebugPlugin.handleException(e);
		}
	}

	Control createCheckSchemesSelectionControl(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(2, false));
		
		schemeViewer = CheckboxTableViewer.newCheckList(
			base, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 2;
		schemeViewer.getControl().setLayoutData(gd);
		schemeViewer.setContentProvider(new ArrayContentProvider());
		schemeViewer.setInput(checkSchemes);
		
		addCheckerButton = createButton(base, "&Add Checker");
		addCheckerButton.setEnabled(false);
		
		removeCheckerButton = createButton(base, "&Remove Checker");
		removeCheckerButton.setEnabled(false);
		
		return base;
	}
	
	Control createIgnoreRegexpsControl(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		
		base.setLayout(new GridLayout(2, false));
		ignoreRegexpViewer = new ListViewer(base);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 2;
		ignoreRegexpViewer.getControl().setLayoutData(gd);
		
		ignoreRegexpViewer.setContentProvider(new ArrayContentProvider());
		ignoreRegexpViewer.setInput(ignoreRegexps);
		
		addRegexpButton = createButton(base, "A&dd Ignore URL Regexp");
		removeRegexpButton = createButton(base, "R&emove Ignore URL Regexp");
		
		return base;
	}
	
	Button createButton(Composite parent, String name){
		Button button = new Button(parent, SWT.PUSH);
		button.setText(name);
		button.addSelectionListener(listener);
		
		button.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_BEGINNING));
		return button;
	}

	void handleAddRegexp(){
		InputDialog input = new InputDialog(
			getShell(),
			"Regexp",
			"Input ignore target for a checker appears url.",
			null,
			new IInputValidator() {
				public String isValid(String v) {
					if(v.length() == 0){
						return "Input any values";
					}
					try{
						Pattern.compile(v);
					}catch(PatternSyntaxException e){
						return e.getMessage();
					}
					return null;
				}
			});
		
		if(input.open() == Window.OK){
			ignoreRegexps.add(input.getValue());
			ignoreRegexpViewer.refresh();
			getLaunchConfigurationDialog().updateButtons();
		}
	}
	
	void handleRemoveRegexp(){
		List ls = ((IStructuredSelection) ignoreRegexpViewer.getSelection()).toList();
		if ( ignoreRegexps.removeAll(ls) ){
			ignoreRegexpViewer.refresh();
		}
	}
	
	void handleAddChecker(){
	}
	
	void handleRemoveChecker(){
	}
	
	public void setDefaults(ILaunchConfigurationWorkingCopy wc) {
		wc.setAttribute(ATTR_IGNORE_REGEXPS, EMPTY_LIST);
		wc.setAttribute(ATTR_CHECK_SCHEMES, Arrays.asList(ORIGINAL_CHECKERS));
		wc.setAttribute(ATTR_BACKGROUND, true);
	}

	public void initializeFrom(ILaunchConfiguration conf) {
		try {
			ignoreRegexps.clear();
			ignoreRegexps.addAll( conf.getAttribute(ATTR_IGNORE_REGEXPS, EMPTY_LIST) );
			List schemes = conf.getAttribute(ATTR_CHECK_SCHEMES, Arrays.asList(ORIGINAL_CHECKERS));
			schemeViewer.setCheckedElements(schemes.toArray());
			backgroundButton.setSelection(conf.getAttribute(ATTR_BACKGROUND, true));
		} catch (CoreException e) {
			WebpubDebugPlugin.handleException(e);
		}
		ignoreRegexpViewer.refresh();
		schemeViewer.refresh();
	}

	public void performApply(ILaunchConfigurationWorkingCopy wc) {
		wc.setAttribute(ATTR_IGNORE_REGEXPS, ignoreRegexps);
		wc.setAttribute(ATTR_CHECK_SCHEMES, Arrays.asList(schemeViewer.getCheckedElements()));
		wc.setAttribute(ATTR_BACKGROUND, backgroundButton.getSelection());
	}

	public String getName() {
		return "&Link Check";
	}

	public Image getImage() {
		return iconImage;
	}

	public void dispose() {
		if(iconImage != null){
			iconImage.dispose();
		}
		super.dispose();
	}

}
