/*
 * $Id: LinkCheckConfigurationFactory.java,v 1.1 2004/01/17 16:49:33 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.linkcheck.core;

import java.io.IOException;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.debug.core.*;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.debug.WebpubDebugPlugin;

/**
 * LinkCheckLaunchConfigurationFactory
 */
public class LinkCheckConfigurationFactory implements LinkCheckConstants {

	String[] extensions;

	public LinkCheckConfigurationFactory(){
		setExtensions( new String[]{
			"htm", "html",
		});
	}

	public ILaunchConfigurationWorkingCopy createLaunchConfiguration(IResource[] resources, boolean convertToPublishLocation) throws CoreException{
		resources = surgeryResourceList(resources, convertToPublishLocation);
		
		// convert to device location pathes
		String[] resourceLocations = new String[resources.length];
		for (int i = 0; i < resourceLocations.length; i++) {
			IResource r = resources[i];
			resourceLocations[i] = r.getLocation().toString();
		}
		
		// create launch configuration
		ILaunchManager launchManager = DebugPlugin.getDefault().getLaunchManager();
		ILaunchConfigurationType configType =
			launchManager.getLaunchConfigurationType(ID_LINKCHECK_LAUNCH_CONFIGURATION);
		ILaunchConfigurationWorkingCopy wc = configType.newInstance(
			null, launchManager.generateUniqueLaunchConfigurationNameFrom("Link Test"));
		
		wc.setAttribute(ATTR_CHECK_RESOURCES, Arrays.asList(resourceLocations));
		wc.setAttribute(ATTR_EXTENSIONS, Arrays.asList(extensions));
		wc.setAttribute(ATTR_BACKGROUND, true);
		
		return wc;
	}
	
	public IResource[] surgeryResourceList(IResource[] rs, boolean convertToPublishLocation) throws CoreException{
		rs = removeOverwrap(rs);
		if(convertToPublishLocation){
			convertToPublishLocation(rs);
			rs = removeOverwrap(rs);
		}
		sortResources(rs);
		return rs;
	}
	
	boolean anyAdded(List list, IContainer container) throws CoreException{
		IResource[] rs = container.members();
		for (int j = 0; j < rs.length; j++) {
			IResource r = rs[j];
			if(r instanceof IFile && !isCheckFile((IFile)r)){
				continue;
			}
			
			if(!list.contains(r)){
				return false;
			}
		}
		return true;
 	}
	
	boolean isCheckFile(IFile f){
		String ext = f.getFileExtension();
		return (ext != null && Arrays.binarySearch(extensions, ext) >= 0) && f.isAccessible();
	}

	IResource[] removeOverwrap(IResource[] resources) throws CoreException{
		Arrays.sort(resources, new Comparator() {
			public boolean equals(Object obj) {
				return false;
			}

			public int compare(Object a, Object b) {
				IPath pa = ((IResource)a).getFullPath();
				IPath pb = ((IResource)b).getFullPath();
				int sa = pa.segmentCount();
				int sb = pb.segmentCount();
				if(sa == sb){
					for(int i=0; i<sa; i++){
						int v = pa.segment(i).compareTo(pb.segment(i));
						if(v != 0){
							return v;
						}
					}
					return 0;
				}else {
					return sa < sb ? 1 : -1;
				}
			}
		});
		
		ArrayList targets = new ArrayList( Arrays.asList(resources) );
		
		for (int i = 0; i < resources.length; i++) {
			IContainer c = resources[i].getParent();
			if(!targets.contains(c) && anyAdded(targets, c)){
				targets.removeAll(Arrays.asList(c.members()));
				targets.add(c);
			}
		}
		
		resources = (IResource[]) targets.toArray(new IResource[targets.size()]);
		for (int i = 0; i < resources.length; i++) {
			IResource r = resources[i];
			if(!r.isAccessible()){
				targets.remove(r);
			}else if(r instanceof IContainer){
				removeOverrapResources(targets, (IContainer)r);
			}else if(r instanceof IFile && !isCheckFile((IFile)r)){
				targets.remove(r);
			}
		}
		
		return (IResource[])targets.toArray(new IResource[targets.size()]);
	}
	
	void sortResources(IResource[] resources){
		Arrays.sort(resources, new Comparator() {
			public boolean equals(Object obj) {
				return false;
			}
	
			public int compare(Object a, Object b) {
				if( a.getClass().equals(b.getClass()) ){
					String pa = ((IResource)a).getLocation().toString();
					String pb = ((IResource)b).getLocation().toString();
					return pa.compareTo(pb);
				}else{
					return (a instanceof IFile) ? 1 : -1;
				}
			}
		});
	}
	
	void convertToPublishLocation(IResource[] resources) throws CoreException{
		PublishDescriptionFactory fac = PublishDescriptionFactory.getInstance();
		for (int i = 0; i < resources.length; i++) {
			IResource resource = resources[i];
			WebProject wp = (WebProject)resource.getProject().getNature(WebProject.ID_NATURE);
			
			if(wp == null){
				// not convert if specified resource is not belong web project.
				continue;
			}
			
			if(resource instanceof IContainer){
				// if specified resource included in source folder, sets mapped publish folder path.
				IContainer htSourceFolder = wp.getFolder(WebProject.KEY_HTSOURCES_FOLDER);
				IContainer pubTarget = (IContainer)resource;
				for(IContainer c = pubTarget; !(c instanceof IProject); c = c.getParent()){
					if(c.equals(htSourceFolder)){
						IPath relPath = pubTarget.getFullPath().removeFirstSegments( htSourceFolder.getFullPath().segmentCount() ); 
						IFolder folder = wp.getFolder(WebProject.KEY_PUBLISH_FOLDER).getFolder(relPath);
						if(folder.isAccessible()){
							resources[i] = folder;
						}
						break;
					}
				}
			} else if(resource instanceof IFile){
				PublishDescription desc = null;
				try {
					desc = fac.create(resource);
				} catch (IllegalConfigurationException e) {
					WebpubDebugPlugin.handleException(e);
				} catch (IOException e) {
					WebpubDebugPlugin.handleException(e);
				}
				if(desc != null){
					IResource r = desc.getPublishTo();
					if(r != null){
						resources[i] = r;
					}
				}
			}
		}
	}

	void removeOverrapResources(List targets, IContainer container) throws CoreException{
		IResource[] rs = container.members();
		for (int i = 0; i < rs.length; i++) {
			IResource r = rs[i];
			targets.remove(r);
			if(r instanceof IContainer){
				removeOverrapResources(targets, (IContainer)r);
			}
		}
	}

	public String[] getExtensions() {
		return (String[])extensions.clone();
	}

	public void setExtensions(String[] exts) {
		extensions = (String[])exts.clone();
		Arrays.sort(extensions);
	}

}
