/*
 * $Id: ResourceSelectionTab.java,v 1.3 2004/01/17 11:50:31 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.debug.ui;

import java.util.*;
import java.util.List;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.dialogs.ResourceSelectionDialog;
import org.eclipse.ui.model.WorkbenchLabelProvider;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.WebpubPlugin;
import com.narucy.webpub.debug.WebpubDebugPlugin;
import com.narucy.webpub.linkcheck.core.LinkCheckConfigurationFactory;
import com.narucy.webpub.linkcheck.core.LinkCheckConstants;

/**
 * ResourceSelectionTabGroup
 */
public class ResourceSelectionTab extends AbstractLaunchConfigurationTab implements LinkCheckConstants {

	final static ArrayList EMPTY_LIST = new ArrayList();
	
	LinkCheckConfigurationFactory configFactory = new LinkCheckConfigurationFactory();
	
	Image iconImage;
	
	TableViewer resourcesViewer;
	ListViewer extsViewer;
	Button addResourceButton, removeResourceButton, addExtsButton, removeExtsButton;

	SelectionListener buttonListener = new SelectionListener() {
		public void widgetSelected(SelectionEvent e) {
			if(e.widget == addResourceButton) handleAddResources();
			if(e.widget == removeResourceButton) handleRemoveResources();
			if(e.widget == addExtsButton) handleAddExts();
			if(e.widget == removeExtsButton) handleRemoveExts();
		}

		public void widgetDefaultSelected(SelectionEvent arg0) {}
	};

	public ResourceSelectionTab(){
	}

	public void createControl(Composite parent) {
		// create icon
		iconImage = WebpubDebugPlugin.getImageDescriptor("ctool16/resources.gif").createImage();
		
		//// create control
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(1, false));
		
		new Label(base, SWT.NONE).setText("Check &Resource:");
		Control resourceBase = createResourceSelectionControl(base);
		resourceBase.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		new Label(base, SWT.NONE).setText("File &Extensions:");
		Control extsBase = createExtsSelectionControl(base);
		extsBase.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		setControl(base);
	}
	
	Control createResourceSelectionControl(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(2, false));
		
		// create viewer
		resourcesViewer = new TableViewer(base);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 2;
		resourcesViewer.getControl().setLayoutData(gd);
		resourcesViewer.setContentProvider(new ArrayContentProvider());
		resourcesViewer.setLabelProvider(new WorkbenchLabelProvider(){
			protected String decorateText(String input, Object element) {
				if(element instanceof IResource){
					return ((IResource)element).getLocation().toString();
				}
				return input;
			}
		});
		
		// create button
		addResourceButton = createButton(base, "&Add");
		removeResourceButton = createButton(base, "&Remove");
		removeResourceButton.setEnabled(false);
		
		resourcesViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				removeResourceButton.setEnabled( !event.getSelection().isEmpty() );
			}
		});
		return base;
	}
	
	Control createExtsSelectionControl(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(2, false));
		
		// create viewer
		extsViewer = new ListViewer(base);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 2;
		extsViewer.getControl().setLayoutData(gd);
		extsViewer.setContentProvider(new ArrayContentProvider());
		extsViewer.setLabelProvider(new LabelProvider());
		
		// create button
		addExtsButton = createButton(base, "A&dd");
		removeExtsButton = createButton(base, "R&emove");
		removeExtsButton.setEnabled(false);
		
		extsViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				removeExtsButton.setEnabled( !event.getSelection().isEmpty() );
			}
		});
		
		return base;
	}

	Button createButton(Composite parent, String name){
		Button button = new Button(parent, SWT.PUSH);
		button.setText(name);
		button.addSelectionListener(buttonListener);
		
		button.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_BEGINNING));
		return button;
	}
		
	IResource[] getSelectedResource(){
		List rs = ((IStructuredSelection)resourcesViewer.getSelection()).toList();
		return (IResource[]) rs.toArray(new IResource[rs.size()]);
	}
	
	void handleAddResources(){
		ResourceSelectionDialog dialog = new ResourceSelectionDialog(
			getShell(),
			ResourcesPlugin.getWorkspace().getRoot(),
			"Select to check resources");
		
		if(dialog.open() == Window.OK){
			Object[] rs = dialog.getResult();
			addResources(Arrays.asList(rs));
		}
	}

	void addResources(List rs){
		addResources((IResource[])rs.toArray(new IResource[rs.size()]));
	}
	
	void addResources(IResource[] rs){
		ArrayList list = (ArrayList)resourcesViewer.getInput();
		// add resources
		for (int i = 0; i < rs.length; i++) {
			if(!list.contains(rs[i])){
				list.add(rs[i]);
			}
		}
		
		// sugery
		rs = (IResource[])list.toArray(new IResource[list.size()]);
		try {
			rs = configFactory.surgeryResourceList(rs, true);
			resourcesViewer.setInput( new ArrayList(Arrays.asList(rs)) );
		} catch (CoreException e) {
			WebpubPlugin.handleException(e);
		}
		
		// update tree
		setDirty(true);
		resourcesViewer.refresh();
		getLaunchConfigurationDialog().updateButtons();
	}
	
	void handleRemoveResources(){
		IResource[] rs = getSelectedResource();
		ArrayList list = (ArrayList)resourcesViewer.getInput();
		for (int i = 0; i < rs.length; i++) {
			list.remove(rs[i]);
		}
		setDirty(true);
		resourcesViewer.setInput(list);
		resourcesViewer.refresh();
		getLaunchConfigurationDialog().updateButtons();
	}

	void handleAddExts(){
		InputDialog input = new InputDialog(
			getShell(),
			"File Extension",
			"Input to check target that is workbench resources extension",
			null,
			FILE_EXTENSION_VALIDATOR);
		
		if(input.open() == Window.OK){
			String[] exts = configFactory.getExtensions();
			String[] newExts = new String[exts.length+1];
			System.arraycopy(exts, 0, newExts, 0, exts.length);
			newExts[newExts.length - 1] = input.getValue();
			setExtensions(newExts);
		}
	}
	
	void setExtensions(List newExts){
		setExtensions((String[])newExts.toArray(new String[newExts.size()]));
	}
	
	void setExtensions(String[] newExts){
		configFactory.setExtensions(newExts);
		extsViewer.setInput(newExts);
		extsViewer.refresh();

		refreshResourceList();
	}
	
	void handleRemoveExts(){
		List exts = new ArrayList( Arrays.asList(configFactory.getExtensions()) );
		exts.removeAll(((IStructuredSelection) extsViewer.getSelection()).toList());
		setExtensions(exts);
	}
	
	void refreshResourceList(){
		addResources(new IResource[]{});
	}
	
	public void setDefaults(ILaunchConfigurationWorkingCopy wc) {
		ArrayList defaultResources = new ArrayList();
		try {
			IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
			for (int i = 0; i < projects.length; i++) {
				WebProject wp = (WebProject)projects[i].getNature(WebProject.ID_NATURE);
				if(wp != null){
					defaultResources.add(wp.getFolder(WebProject.KEY_PUBLISH_FOLDER).getLocation().toString());
				}
			}
		} catch (CoreException e) {
			WebpubDebugPlugin.handleException(e);
		}
		wc.setAttribute(ATTR_CHECK_RESOURCES, defaultResources);
		wc.setAttribute(ATTR_EXTENSIONS, Arrays.asList(new String[]{"html", "htm"}));
	}

	public void initializeFrom(ILaunchConfiguration conf) {
		try {
			// init resources
			List resources = conf.getAttribute(ATTR_CHECK_RESOURCES, EMPTY_LIST);
			ArrayList rs = new ArrayList();
			
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			for(int i=0; i<resources.size(); i++){
				Path path = new Path((String)resources.get(i));
				
				IResource r = root.getFileForLocation(path);
				if(r == null || !r.isAccessible()){
					r = root.getContainerForLocation(path);
				}
				if(r != null && r.isAccessible()){
					rs.add(r);
				}
			}
			resourcesViewer.setInput(rs);

			// init exts
			List exts = conf.getAttribute(ATTR_EXTENSIONS, EMPTY_LIST);
			setExtensions(exts);
			
		} catch (CoreException e) {
			WebpubDebugPlugin.handleException(e);
		}
	}

	public void performApply(ILaunchConfigurationWorkingCopy wc) {
		ArrayList locations = new ArrayList();
		ArrayList resourceList = (ArrayList)resourcesViewer.getInput();
		for(int i=0; i<resourceList.size(); i++){
			IResource r = (IResource)resourceList.get(i);
			locations.add( r.getLocation().toString() );
		}
		wc.setAttribute(ATTR_CHECK_RESOURCES, locations);
	}

	public String getName() {
		return "&Target Resources";
	}

	public Image getImage() {
		return iconImage;
	}

	public void dispose() {
		if(iconImage != null){
			iconImage.dispose();
		}
		super.dispose();
	}

	final public static IInputValidator FILE_EXTENSION_VALIDATOR = new IInputValidator() {
		char[] invalidChars = "\\/:;*,?<>| \t\r\n".toCharArray();
		
		public String isValid(String newName) {
			for (int i = 0; i < invalidChars.length; i++) {
				char ch = invalidChars[i];
				if(newName.indexOf(ch) != -1){
					return "Invalid char code \"" + ch + "\"";
				}
			}
			if (newName.length() > 0 && newName.charAt(0) == '.'){
				return "Not need to specify first char \".\"";
			}
			return null;
		}
	};

}
