/*
 * $Id: TemplateParser.java,v 1.1 2004/01/17 15:51:52 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.template;

import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;

import com.narucy.webpub.core.TextReader;
import com.narucy.webpub.core.WebProject;

/**
 * TemplateParser role is to create root templates.
 */
public class TemplateParser {

	final static Pattern
		beginPattern = Pattern.compile("^\\s*<!--\\s*\\$begin:(.+)\\$\\s*-->\\s*$"),
		endPattern = Pattern.compile("^\\s*<!--\\s*\\$end:(.+)\\$\\s*-->\\s*$");

	static TemplateParser instance = new TemplateParser();

	public static TemplateParser getInstance(){
		return instance;
	}

	/**
	 * Template property value of find line number.
	 * value type of Integer that represents find line number.
	 */
	final public static String
		FIND_LINE_NUMBER = "find_line_number";
	
	private TemplateParser(){
	}
	
	public Template parse(IFile f) throws TemplateParseException, IOException, CoreException {
		WebProject wp = (WebProject)f.getProject().getNature(WebProject.ID_NATURE);
		String charset = wp.getString(WebProject.KEY_HT_CHARSET);
		
		return parse(f.getContents(), charset, "Root");
	}
		
	/**
	 * Parse the template instance. 2nd argument name is root node name for
	 * new template.
	 */
	public Template parse(InputStream stream, String charset, String rootTmplName) throws TemplateParseException, IOException {
		TextReader reader = new TextReader(stream, charset);
		try{
			return parse(reader, rootTmplName);
		} finally {
			if( reader != null){
				reader.close();
			}
		}
	}

	public Template parse(Iterator iter, String rootTmplName) throws TemplateParseException {
		Template root = new Template(rootTmplName);
		
		Stack stack = new Stack();
		Template template = root;
		for (int i=0; iter.hasNext(); i++) {
			String line = (String)iter.next();
			
			String name = matchBeginTmplateLine(line);
			if(name != null){
				Template newTemplate = new Template(name);
				newTemplate.setProperty(FIND_LINE_NUMBER,  new Integer(i));
				
				stack.add(template);
				template.addTemplate(newTemplate);
				template = newTemplate;
			}
			if( template != null){
				template.addLine(line);
			}
			else{
				root.addLine(line);
			}
			if( matchEndTemplateLine(line) != null){
				if( template == null){
					throw new TemplateParseException(line, i);
				}
				if( stack.isEmpty() ){
					throw new TemplateParseException("close template error", i);
				}
				template = (Template)stack.pop();
			}
		}
		
		return root;
	}
	
	public static String matchBeginTmplateLine(String line){
		Matcher beginMatch = beginPattern.matcher(line);
		if(beginMatch.matches()){
			return beginMatch.group(1);
		}
		return null;
	}
	
	public static String matchEndTemplateLine(String line){
		Matcher endMatch = endPattern.matcher(line);
		if(endMatch.matches()){
			return endMatch.group(1);
		}
		return null;
	}


}
