/*
 * $Id: TemplateManager.java,v 1.1 2004/01/17 15:51:52 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.template;

import java.io.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.WebpubPlugin;

/**
 * <p>
 * TemplateManager roles are to keep template datas for a project,
 * and modify templates.
 * 
 * TODO: manage encoding.
 * TODO: Locking template archtecture.
 * TODO: defines up to date behavior.
 * TODO: refactor template name
 */
final public class TemplateManager {

	// project par one instance.

	final public static String
		ID_MARKER_TEMPLATE_ERROR = WebpubPlugin.ID_PLUGIN + ".template.error";

	static HashMap registedProjects = new HashMap();

	public static IProject[] getRegistedProjects(){
		return (IProject[])registedProjects.keySet().toArray(new IProject[registedProjects.size()]);
	}
	
	public static TemplateManager getManagerInstance(IProject proj) throws CoreException {
		TemplateManager manager = (TemplateManager)registedProjects.get(proj);
		if(manager == null){
			manager = new TemplateManager(proj);
			registedProjects.put(proj, manager);
		}
		return manager;
	}

	static TemplateParser parser = TemplateParser.getInstance();

	// instance methods.

	IProject project;

	HashMap
		globalTemplates = new HashMap(),
		templateDefinedLocation = new HashMap();
	
	ArrayList listeners = new ArrayList();	

	private TemplateManager(IProject proj) throws CoreException {
		this.project = proj;
		researchAll( new NullProgressMonitor() );
	}

	/**
	 * Template defined files modifier for synchronize in project templates
	 * consistent.
	 */
	class ModifyTemplateProc implements IWorkspaceRunnable {
	
		IFile[] checksFiles;
		ArrayList distedFiles = new ArrayList();
		String charset;
		
		ModifyTemplateProc(IFile[] checksFiles) throws CoreException{
			this.checksFiles = checksFiles;
			WebProject wp = (WebProject)checksFiles[0].getProject().getNature(WebProject.ID_NATURE);
			charset = wp.getString(WebProject.KEY_HT_CHARSET);
		}
		
		public void run(IProgressMonitor monitor) throws CoreException {
			monitor.beginTask("Research templates", 1000);
			try{
				for (int i = 0, task = 1000 / checksFiles.length; i < checksFiles.length; i++) {
					IFile f = checksFiles[i];
					f.refreshLocal(IResource.DEPTH_ZERO, new SubProgressMonitor(monitor, task) );
					if( f.exists() ){
						try {
							deleteMakers(f);
							autoModify(f, new SubProgressMonitor(monitor, task));
							addGlobalTemplate(f);
						} catch (TemplateParseException e) {
							handleException(f, e);
						} catch (IOException e) {
							handleException(f, e);
						} catch (TemplateManageException e) {
							handleException(f, e);
						}
					}
					monitor.worked(task);
				}
			}
			finally{
				monitor.done();
			}
		}

		void autoModify(IFile file, IProgressMonitor monitor) throws TemplateParseException, IOException, CoreException, TemplateManageException{
			monitor.beginTask(
				"Modify that content changed file has template that relational resources",
				1);
			
			try {
				IProject proj = file.getProject();
				if( !file.isSynchronized(IResource.DEPTH_ZERO) ){
					file.refreshLocal(IResource.DEPTH_ZERO, new SubProgressMonitor(monitor, 1));
				}
				Template[] maybeModifiedTemplates = parser.parse(
					file.getContents(),
					charset,
					file.getFullPath().toString()).getAllTemplates();

				// search modify target files.
				ArrayList relationFiles = new ArrayList();
				ArrayList changeTemplates = new ArrayList();
				tmplLoop: for (int i=0; i<maybeModifiedTemplates.length; i++) {
					Template maybeNewTmpl = maybeModifiedTemplates[i];
					
					// ignore template type is jump.
					String name = maybeNewTmpl.getName();
					if( name.equals(Template.NAME_CONTENTS) ){
						continue;
					}
					
					// search changed tmplates.
					Template globalSideTmpl = getGlobalTemplate(name);
					if( globalSideTmpl == null){
						// new template
						globalTemplates.put(name, maybeNewTmpl);
						templateDefinedLocation.put(name, new ArrayList() );
					} else if( !globalSideTmpl.equals(maybeNewTmpl) ){
						// If changed file has templates is different with global template
						// and, already changed flags is on, batting a changed file, thrown
						// exception.
						for(int j=0; j<changeTemplates.size(); j++){
							Template t = (Template)changeTemplates.get(j);
							if( t.getName().equals(name) ){
								if(!t.equals(maybeNewTmpl)){
									IMarker marker = file.createMarker(IMarker.PROBLEM);
									marker.setAttribute(ID_MARKER_TEMPLATE_ERROR, true);
									marker.setAttribute(IMarker.MESSAGE, "Updated template found more two points in same template name:" + name);
									throw new TemplateManageException(file, name);
								}
								continue tmplLoop;
							}
						}

						// modify template. get modify target template relational files.
						IFile[] definedFiles = getTemplateDefinedFiles(name);
						for (int j=0; j<definedFiles.length; j++) {
							IFile f = definedFiles[j];
							if( !relationFiles.contains(f) ){
								relationFiles.add(f);
							}
						}
						changeTemplates.add(maybeNewTmpl);
					}
				}
				
				// regist to global template.
				if( !changeTemplates.isEmpty() && !relationFiles.isEmpty() ){
					Template[] changeTmpls = (Template[])changeTemplates.toArray(new Template[changeTemplates.size()]);
					for (int i = 0; i < changeTmpls.length; i++) {
						Template t = changeTmpls[i];
						globalTemplates.put( t.getName(), t);
					}
					IFile[] files = (IFile[])relationFiles.toArray(new IFile[relationFiles.size()]);
					doModify(files, changeTmpls, new SubProgressMonitor(monitor, 1));
				}
			} finally {
				monitor.done();
			}
		}
		
		IFile[] getDistributedFiles(){
			return (IFile[])distedFiles.toArray( new IFile[distedFiles.size()]);
		}
		
		void doModify(IFile[] relFiles, Template[] changeTmpls, IProgressMonitor monitor) throws TemplateParseException, IOException, CoreException{
			try{
				monitor.beginTask("Do modify for template synchronization", 1000);
				for (int i = 0, task = 1000 / relFiles.length; i < relFiles.length; i++) {
					IFile f = relFiles[i];
					deleteMakers(f);
					
					Template tmpl = parser.parse(f.getContents(), charset, "Root");
					for (int j = 0; j < changeTmpls.length; j++) {
						tmpl.modifyTmpl( changeTmpls[j] );
					}
					distribute(f, tmpl);
					distedFiles.add(f);
					f.refreshLocal(IResource.DEPTH_ZERO, new SubProgressMonitor(monitor, task));
	        			}
			}finally{
				monitor.done();
			}
		}
	}
	
	/**
	 * Delete after appends attributes in template manager.
	 */
	static void deleteMakers(IResource res) throws CoreException{
		IMarker[] markers = res.findMarkers(IMarker.PROBLEM, false, IResource.DEPTH_ZERO);
		for (int j = 0; j < markers.length; j++) {
			IMarker m = markers[j];
			if( m.getAttribute(ID_MARKER_TEMPLATE_ERROR) != null){
				m.delete();
			}
		}
	}
	
	static void handleException(IFile file, Exception error) throws CoreException{
		IMarker marker = file.createMarker(IMarker.PROBLEM);
		marker.setAttribute(ID_MARKER_TEMPLATE_ERROR, true);
		if( error instanceof TemplateParseException){
			TemplateParseException e = (TemplateParseException)error;
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker.setAttribute(IMarker.LINE_NUMBER, e.getLineNum());
		}
		marker.setAttribute( IMarker.MESSAGE, error.getMessage());
	}
	
	void distribute(IFile file, Template tmpl) throws IOException, CoreException {
		OutputStreamWriter writer = null;
		try {
			WebProject wp = (WebProject)file.getProject().getNature(WebProject.ID_NATURE);
			String charset = wp.getString(WebProject.KEY_HT_CHARSET);
			writer = new OutputStreamWriter( new FileOutputStream(file.getLocation().toFile()), charset);
			String[] lines = tmpl.getLines();
			for (int i = 0; i < lines.length; i++) {
				writer.write(lines[i]);
				writer.write('\n');
			}
		}finally{
			if(writer != null){
				writer.close();
			}
		}
	}
	
	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}
	
	/**
	 * modify templates cache.
	 */
	public synchronized IFile[] syncronizeTemplate(IFile[] changedFiles, IFile[] removedFile, IProgressMonitor monitor) throws CoreException {
		IFile[] distributed;

		for(int i=0; i<removedFile.length; i++){
			removeGlobalTemplate(removedFile[i] );
		}

		if( changedFiles.length > 0){
			ModifyTemplateProc proc = new ModifyTemplateProc(changedFiles);
			ResourcesPlugin.getWorkspace().run(proc, new SubProgressMonitor( monitor, 1) );
			distributed = proc.getDistributedFiles();
		}else{
			distributed = new IFile[0];
		}
		
		fireTemplateModified();
		return distributed;
	}
	
	/**
	 * Throws CoreException that thrown when in occse search html template
	 * resoruces.
	 */
	public synchronized void researchAll(IProgressMonitor monitor) throws CoreException{
		try{
			monitor.beginTask("Researching templates", IProgressMonitor.UNKNOWN);
			WebProject webProj = (WebProject)project.getNature(WebProject.ID_NATURE);
			if(webProj == null){
				return;
			}
			// TODO: selectable any folder.
			IFile[] files = webProj.findFileMembers(WebProject.KEY_HTSOURCES_FOLDER);
			for (int i = 0; i < files.length; i++) {
				IFile f = files[i];
				try {
					if( webProj.isHTExtension(f.getFileExtension()) && f.exists() ){
						f.refreshLocal(IResource.DEPTH_ZERO, new SubProgressMonitor(monitor, 1));
						deleteMakers(f);
						addGlobalTemplate(f);
						
						monitor.worked(1);
					}
				} catch (TemplateManageException e) {
					handleException(f, e);
				} catch (TemplateParseException e) {
					handleException(f, e);
				} catch (IOException e) {
					handleException(f, e);
				}
			}
			fireTemplateModified();
		}
		finally{
			monitor.done();
		}
	}

	/**
	 * Load templat from speicfy file into global templates stack.
	 */
	synchronized void addGlobalTemplate(IFile file)
			throws TemplateManageException, TemplateParseException, IOException, CoreException {
		
		HashMap notifyTargets = new HashMap();
		
		Template rootTmpl = parser.parse(file);
		Template[] tmpls = rootTmpl.getAllTemplates();
		
		for (int i=0; i<tmpls.length; i++) {
			Template tmpl = tmpls[i];
			String name = tmpl.getName();
			// none addition name of CONTENTS.
			if( name.equals(Template.NAME_CONTENTS)){
				continue;
			}
			if( !isContainGlobalTemplate(name) ){
				globalTemplates.put(name, tmpl.clone() );
				templateDefinedLocation.put(name, new ArrayList() );
			}
			List list = (List)templateDefinedLocation.get(name);
			if( !list.contains(file)){
				list.add(file);
			}
			// if same name template but contain is different,
			// raise exception.
			Template addedTmpl = getGlobalTemplate(name);
			if( !addedTmpl.equals(tmpl)){
				throw new TemplateManageException(file, name);
			}
		}
	}
	
	/**
	 * Remove referer target specify file form templates cache.
	 */
	synchronized void removeGlobalTemplate(IFile file){
		Template[] tmpls = getTemplatesFromCache(file);
		for (int i = 0; i < tmpls.length; i++) {
			Template template = tmpls[i];
			
			String tmplName = template.getName();
			List files = (List)templateDefinedLocation.get(tmplName);
			files.remove(file);
			if( files.size() == 0){
				globalTemplates.remove(tmplName);
				templateDefinedLocation.remove(tmplName);
			}
		}
	}
	
	/**
	 * Returns specifed file has template.
	 * None parse process, get from on memory cache.
	 */
	public Template[] getTemplatesFromCache(IFile file){
		ArrayList tmpls = new ArrayList();
		Template[] globalTmpls = getGlobalTemplates();
		for (int i = 0; i < globalTmpls.length; i++) {
			Template tmpl = globalTmpls[i];
			IFile[] definedFiles = getTemplateDefinedFiles(tmpl.getName());
			for (int j = 0; j < definedFiles.length; j++) {
				if( definedFiles[j].equals(file) ){
					tmpls.add(tmpl);
				}
			}
		}
		return (Template[])tmpls.toArray(new Template[tmpls.size()]);
	}

	/**
	 * Returns is resisted specify template name in this template manager.
	 */
	public boolean isContainGlobalTemplate(String name){
		// generally both HashMap instances are same.
		return globalTemplates.containsKey(name) && templateDefinedLocation.containsKey(name);
	}
	
	public Template getGlobalTemplate(String name){
		return (Template)globalTemplates.get(name);
	}
	
	public String[] getGlobalTemplateNames(){
		return (String[])globalTemplates.keySet().toArray(new String[globalTemplates.size()]);
	}
	
	public Template[] getGlobalTemplates(){
		return (Template[])globalTemplates.values().toArray(new Template[globalTemplates.size()]);
	}
	
	public IFile[] getTemplateDefinedFiles(String name){
		List files = (List)templateDefinedLocation.get(name);
		return (files != null) ? (IFile[])files.toArray( new IFile[files.size()] ) : new IFile[0];
	}

	public void addListener( TemplateManagerListener l){
		listeners.add(l);
	}
	
	public void removeListener( TemplateManagerListener l){
		listeners.remove(l);
	}
	
	protected void fireTemplateModified(){
		for (int i = 0; i < listeners.size(); i++) {
			((TemplateManagerListener)listeners.get(i)).templateModified(project);
		}
	}	
	
	/**
	 * Return true if specified file is ignore html template search target
	 * in web project.
	 */	
	public static boolean isTemplateFile(IResource file) throws CoreException {
		IProject proj = file.getProject();
		if( !(file instanceof IFile) ){
			// not related project, don't check.
			return false;
		}
		
		// need to be a web project file.
		WebProject webProj = (WebProject)proj.getNature(WebProject.ID_NATURE);
		if( webProj == null || !webProj.isHTExtension(file.getFileExtension()) ){
			return false;
		}
		
		// specified file is included publish, it's ignore.
		IContainer pubFolder = webProj.getFolder(WebProject.KEY_PUBLISH_FOLDER);
		for(IResource res = file; !((res = res.getParent()) instanceof IProject);){
			if ( res.equals(pubFolder) ) {
				return false;
			}
		}
		
		return true;
	}

}
