/*
 * $Id: SourceFileSeeker.java,v 1.1 2004/01/17 15:51:55 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.publish;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.*;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.WebpubPlugin;

/**
 * PublishFromSeeker provides search publish from content file function.
 */
public class SourceFileSeeker {

	static DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();

	IFile publishFile;
	WebProject webProject = null;

	public SourceFileSeeker(String url) throws CoreException {
		this(toURL(url));
	}
	
	static URL toURL(String urlStr){
		URL url = null;
		try {
			url = new URL(urlStr);
		} catch (MalformedURLException e) {
			try {
				url = new File(urlStr).toURL();
			} catch (MalformedURLException e1) {
			}
		}
		return url;
	}

	public SourceFileSeeker(URL url) throws CoreException {
		this(url != null ? urlToFile(url) : null);
	}

	/**
	 * URL to ht resource location. returns publish file.
	 */
	static IFile urlToFile(URL url) throws CoreException {
		// to workspace file
		Path path = new Path(new File(url.getFile()).toString());
		IFile file = ResourcesPlugin.getWorkspace().getRoot().getFileForLocation(path);
		if(file != null && file.isAccessible()){
			return file;
		}
		
		// find from mapped url.
		String urlStr = url.toString();
		WebProject[] wps = getWebProjects();
		for (int i = 0; i < wps.length; i++) {
			WebProject wp = wps[i];
			String[] urls = wp.getArray(WebProject.KEY_MAPPED_URL);
			for (int j = 0; j < urls.length; j++) {
				String u = urls[j];
				if( urlStr.toLowerCase().startsWith(u.toLowerCase()) ){
					String relPath = urlStr.substring(u.length());
					if(relPath.charAt(relPath.length()-1) == '/'){
						relPath += "index.html";
					}
					IFile f = wp.getFolder(WebProject.KEY_PUBLISH_FOLDER).getFile(new Path(relPath));
					if(f != null && f.isAccessible()){
						return f;
					}
				}
			}
		}

		return null;
	}
	
	/**
	 * Collect Web Project from workspace.
	 */
	static WebProject[] getWebProjects() throws CoreException{
		ArrayList dist = new ArrayList();
		
		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		for (int i = 0; i < projects.length; i++) {
			IProject project = projects[i];
			if(project.isAccessible()){
				WebProject wp = (WebProject) project.getNature(WebProject.ID_NATURE);
				if(wp != null){
					dist.add(wp);
				}
			}
		}
		
		return (WebProject[]) dist.toArray(new WebProject[dist.size()]);
	}

	public SourceFileSeeker(IFile publishFile) throws CoreException {
		this.publishFile = publishFile;

		if( publishFile != null){
			WebProject[] wps = getWebProjects();
			for (int i = 0; i < wps.length; i++) {
				WebProject wp = wps[i];
				if( wp.getFolder(WebProject.KEY_PUBLISH_FOLDER).getProject().equals(publishFile.getProject()) &&
					isParentFolder(publishFile, wp.getFolder(WebProject.KEY_PUBLISH_FOLDER)) ){
					
					webProject = wp;
					break;
				}
			}
		}
	}
	
	/**
	 * <p>
	 * Returns publish source file of constructer specified url published file.
	 * 
	 * <ul>
	 * <li>If specify file is not inclulded in publish folder of web project. 
	 * <li>Next, search way to simple mapping "ht source folder"/"public folder".
	 * <li>If not found follow way, search publish descriptor, and map a file.
	 * </ul>
	 */
	public IFile getPublishFrom() throws CoreException {
		if(webProject == null){
			return null;
		}
		
		IContainer pubFolder = webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER);
		if( pubFolder == null || !isParentFolder(publishFile, pubFolder) ){
			return null;
		}
		
		IContainer htFolder = webProject.getFolder(WebProject.KEY_HTSOURCES_FOLDER);
		IPath publishPath = pubFolder.getFullPath();
		IPath relPath = publishFile.getFullPath()
			.removeFirstSegments(publishPath.segmentCount() );
		IFile htSrc = htFolder.getFile(relPath);
		if( htSrc.exists() ){
			return htSrc;
		}
		
		return getPublishFromAsPublishDescriptions();
	}
	
	IFile getPublishFromAsPublishDescriptions() throws CoreException {
		try{
			// if not found ht source mapping, search from publish descriptors.
			IFile[] publishDescriptions = findPublishDescriptorFiles();
			sortPublishDescriptorFiles(publishDescriptions);
			for(int i=0; i<publishDescriptions.length; i++){
				IFile pubDesc = publishDescriptions[i];
				IContainer fromBase = pubDesc.getParent();
					
				Object[][] pubDescEntries = getPublishDescriptionEntries(publishDescriptions[i]);
				for(int j=0; j<pubDescEntries.length; j++){
					String pattern = (String)pubDescEntries[j][0];
					String pubTo = (String)pubDescEntries[j][1];
					if( pubTo != null){
						IFile file = isPublishFileEntried(fromBase, pattern, pubTo, publishFile);
						if( file != null){
							return file;
						}
					}
				}
			}
		} catch (SAXException e) {
			WebpubPlugin.handleException(e);
		} catch (IOException e) {
			WebpubPlugin.handleException(e);
		} catch (ParserConfigurationException e) {
			WebpubPlugin.handleException(e);
		}
		return null;
	}
	
	/**
	 * Find with wild card pattern string.
	 */
	IFile isPublishFileEntried( IContainer fromContainer, String pattern, String publishTo, IFile publishFile) throws CoreException {
		// direct match
		IFile tobePublishedFile = webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER).getFile(new Path(publishTo));
		
		if( publishFile.equals(tobePublishedFile) ){
			IResource[] ress = fromContainer.members();
			for (int i = 0; i < ress.length; i++) {
				 if( ress[i] instanceof IFile){
					IFile fromFile = (IFile)ress[i];
					String relPath =
						fromFile.getFullPath().removeFirstSegments(
							fromContainer.getFullPath().segmentCount() ).toString();
					if( PublishDescriptionFactory.isMatch(pattern, relPath) ){
						return fromFile;
					}
				}
			}
		}else if( publishTo.charAt(publishTo.length()-1) == '/'){
			IResource[] ress = fromContainer.members();
			for (int i = 0; i < ress.length; i++) {
				IResource res = ress[i];
				if( res instanceof IFile && ((IFile)res).getName().equals(publishFile.getName()) ){
					return (IFile)res;
				}
			}
		}else if (publishTo.indexOf('*') != -1){
			// refer publish location path.
			IPath publishLocation = new Path(publishTo).removeLastSegments(1);
			IFolder parentFolder = webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER).getFolder(publishLocation);
			if(publishFile.getParent().equals(parentFolder)){
				
				// do find as from container.
				IFile[] fromFiles = findFiles(fromContainer);
				for (int i = 0; i < fromFiles.length; i++) {
					IFile fromFile = fromFiles[i];
					
					// matching path
					String relPath =
						fromFile.getFullPath().removeFirstSegments(
							fromContainer.getFullPath().segmentCount() ).toString();
					
					if( PublishDescriptionFactory.isMatch(pattern, relPath) &&
						getBaseName(fromFile.getName()).equals( getBaseName(publishFile.getName())) ){
						return fromFile;
					}
				}
			}
		}
		return null;
	}
	
	static IFile[] findFiles(IContainer container) throws CoreException{
		ArrayList dist = new ArrayList();
		collectFiles(container, dist);
		return (IFile[]) dist.toArray(new IFile[dist.size()]);
	}
	
	static void collectFiles(IContainer container, List dist) throws CoreException{
		IResource[] members = container.members();
		for (int i = 0; i < members.length; i++) {
			IResource r = members[i];
			if(r instanceof IFile){
				dist.add(r);
			}else if(r instanceof IContainer){
				collectFiles((IContainer)r, dist);
			}
		}
		
	}
	
	static String getBaseName(String file){
		int separatorIndex = file.lastIndexOf('/');
		if( separatorIndex != -1){
			file = file.substring(separatorIndex + 1);
		}
		int dotIndex = file.lastIndexOf('.');
		if( dotIndex != -1){
			file = file.substring(0, dotIndex);
		}
		return file;
	}

	static Object[][] getPublishDescriptionEntries(IFile publishDescriptionFile) throws SAXException, IOException, ParserConfigurationException, CoreException {
		Document doc = docBuilderFactory.newDocumentBuilder().parse( publishDescriptionFile.getContents() );
		Element rootElem = doc.getDocumentElement();
		
		NodeList mappings = rootElem.getElementsByTagName("mapping");
		
		int len = mappings.getLength();
		Object[][] entries = new Object[len][2];
		for(int i=0; i<len; i++){
			Object[] entry = entries[i];
			Element mappingElem = (Element)mappings.item(i);
			entry[0] = mappingElem.getAttribute("pattern");
			
			Element pubElem = (Element)mappingElem.getElementsByTagName("publish").item(0);
			String pubTo = pubElem.getAttribute("publish_to");
			entry[1] = pubTo.length() > 0 ? pubTo : null;
		}
		
		return entries;
	}

	/**
	 * Sort by scarcity folder,
	 * for search publish to location.
	 */
	void sortPublishDescriptorFiles(IFile[] files) throws CoreException {
		final IContainer htFolder = webProject.getFolder(WebProject.KEY_HTSOURCES_FOLDER);
		Arrays.sort(files, new Comparator() {
			public boolean equals(Object obj) {
				return false;
			}
			public int compare(Object o1, Object o2) {
				IFile f1 = (IFile)o1;
				IFile f2 = (IFile)o2;
				
				boolean inHtSource1 = isParentFolder(f1, htFolder);
				boolean inHtSource2 = isParentFolder(f2, htFolder);
				if( inHtSource1 != inHtSource2){
					return inHtSource1 ? 1 : -1;
				}
				
				int s1 = f1.getFullPath().segmentCount();
				int s2 = f2.getFullPath().segmentCount();
				
				return (s1 == s2) ? 0 : (s1 < s2) ? 1 : -1;
			}
		});
	}
	
	static boolean isParentFolder(IFile file, IContainer target){
		for(IResource res = file; !((res = res.getParent()) instanceof IWorkspaceRoot);){ 
			if( target.equals(res) ){
				return true;
			}
		}
		return false;
	}

	IFile[] findPublishDescriptorFiles() throws CoreException {
		ArrayList dist = new ArrayList();
		IResource[] folders = webProject.getProject().members();
		for(int i=0; i<folders.length; i++){
			if( folders[i] instanceof IFolder){
				doFindPublishDescriptorFile((IFolder)folders[i], dist);
			}
		}
		return (IFile[])dist.toArray(new IFile[dist.size()]);
	}

	static void doFindPublishDescriptorFile(IFolder folder, List dist) throws CoreException{
		IResource[] resources = folder.members();
		for (int i = 0; i < resources.length; i++) {
			IResource resource = resources[i];
			if( resource instanceof IFile && ((IFile)resource).getName().equals(PublishDescriptionFactory.PUBLISH_DESCRIPTION_FILENAME) ){
				dist.add(resource);
			}else if(resource instanceof IFolder){
				doFindPublishDescriptorFile((IFolder)resource, dist);
			}
		}
	}

}
