/*
 * $Id: PublisherBuilder.java,v 1.1 2004/01/17 15:51:55 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.publish;

import java.io.*;
import java.util.ArrayList;
import java.util.Map;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

/**
 * @see IncrementalProjectBuilder
 */
public class PublisherBuilder extends IncrementalProjectBuilder {

	final public static String
		ID_BUILDER = WebpubPlugin.ID_PLUGIN + ".publisherBuilder",
		ID_MARKER_PUBLISHER_PROBLEM = "com.narucy.webpub.core.publisher.problem";
	
	final public static File UNPARSED_LINES_TEMPORARY_FILE =
		new File(System.getProperty("java.io.tmpdir"), "com.narucy.webpub.core.publish.unparsedlines");

	String baseScriptPath = WebpubPlugin.getDefault().getDescriptor().find(new Path("scripts/bin/pubtool.rb") ).getFile();
	int readSleepMs = 10;

	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		buildPrepare();
		
		if (kind == IncrementalProjectBuilder.FULL_BUILD) {
		   fullBuild(monitor);
		} else {
		   IResourceDelta delta = getDelta(getProject());
		   if (delta == null) {
			  fullBuild(monitor);
		   } else {
			  incrementalBuild(delta, monitor);
		   }
		}
		return null;
	}

	void buildPrepare(){
		UNPARSED_LINES_TEMPORARY_FILE.delete();
	}
	
	void fullBuild(IProgressMonitor monitor) throws CoreException {
		try{
			monitor.beginTask("Publish all sources", 1024);
			IResource[] resources = findTargetResources();
			if( resources.length > 0){
				doPublish(resources, new SubProgressMonitor(monitor, 1024 / resources.length));
			}
		}
		finally{
			monitor.done();
		}
	}

	/**
	 * Choose publishing atarget files these are all of files
	 * except of script folder and publisher.
	 */	
	IResource[] findTargetResources() throws CoreException {
		IProject project = getProject();
		WebProject webProject = (WebProject)project.getNature(WebProject.ID_NATURE);
		IContainer publishFolder = webProject.getFolder(WebProject.KEY_PUBLISH_FOLDER);
		IContainer scriptsFolder = webProject.getFolder(WebProject.KEY_SCRIPTS_FOLDER);
		
		ArrayList dist = new ArrayList();
		IResource[] resources = project.members();
		for (int i = 0; i < resources.length; i++) {
			IResource res = resources[i];
			if( res instanceof IFile){
				dist.add(res);
			}
			else if(res instanceof IFolder && !(res.equals(publishFolder) || res.equals(scriptsFolder)) ) {
				findResources( (IFolder)res, dist);
			}
		}
		return (IResource[])dist.toArray(new IResource[dist.size()]);
	}
	
	static void findResources(IFolder folder, ArrayList dist) throws CoreException{
		IResource[] resources = folder.members();
		for (int i = 0; i < resources.length; i++) {
			IResource res = resources[i];
			dist.add(res);
			if(res instanceof IFolder){
				findResources( (IFolder)res, dist);
			}
		}
	}

	void incrementalBuild(IResourceDelta delta, final IProgressMonitor monitor) throws CoreException {
		try{
			monitor.beginTask("Incremental Publish", 1);
			
			final ArrayList publishTargets = new ArrayList();
			delta.accept(new IResourceDeltaVisitor() {
				PublishDescriptionFactory fac = PublishDescriptionFactory.getInstance();
				public boolean visit(IResourceDelta delta) throws CoreException {
					int kind = delta.getKind();
					if( kind == IResourceDelta.ADDED || kind == IResourceDelta.CHANGED){
						IResource res = delta.getResource();
						try{
							if(fac.create(res) != null){
								publishTargets.add(res);
							}
						} catch(IOException e){
							throw new CoreException(new Status(
								IStatus.ERROR,
								WebpubPlugin.ID_PLUGIN,
								IStatus.OK,
								"Error occastion in build process:" + res,
								e));
						} catch (IllegalConfigurationException e) {
							throw new CoreException(new Status(
								IStatus.WARNING,
								WebpubPlugin.ID_PLUGIN,
								IStatus.OK,
								"Illegal publish configuration file:" + e.getIllegalFile(),
								e));
						}
					}
					return true;
				}

			});
			if( publishTargets.size() > 0){
				doPublish( (IResource[])publishTargets.toArray(new IResource[publishTargets.size()]), new SubProgressMonitor(monitor, 1) );
			}
		}
		finally{
			monitor.done();
		}
	}
	
	WebProject getWebProject() throws CoreException {
		return (WebProject)getProject().getNature(WebProject.ID_NATURE);
	}
	
	/**
	 * Delete marker
	 */
	void deleteMarker(IResource res) throws CoreException{
		if( res.isAccessible() ){
			IMarker[] markers = res.findMarkers(null, false, IResource.DEPTH_ZERO);
			for (int i = 0; i < markers.length; i++) {
				IMarker m = markers[i];
				if( m.getAttribute(ID_MARKER_PUBLISHER_PROBLEM) != null){
					m.delete();
				}
			}
		}
	}
	
	void doPublish(IResource[] resources, IProgressMonitor monitor) throws CoreException {
		try{
			monitor.beginTask("Publish process", IProgressMonitor.UNKNOWN);
			
			// Remove old publish task data
			for (int i = 0; i < resources.length; i++) {
				deleteMarker(resources[i]);
			}
			
			IProject project = getProject();
			String charset = ((WebProject)project.getNature(WebProject.ID_NATURE)).getString(WebProject.KEY_HT_CHARSET);
			
			IPath pubConfigFile = generatePublishConfig();
			
			String command =
				baseScriptPath +
				" --workspace " + project.getWorkspace().getRoot().getLocation() +
				" --plugin_properties " + pubConfigFile +
				" --files " + joinResourceArray(resources);
			
			// exec ruby from system call
			// invoke stream reader thread thread
			Process process = WebpubPlugin.getDefault().rubyExec(command, project.getLocation());
			BuildResultReader resultReader = new BuildResultReader(process.getInputStream(), charset);
			BuildErrorReader errorReader = new BuildErrorReader(process.getErrorStream(), charset);
			
			resultReader.start();
			errorReader.start();
			
			// loop for waiting publishing process
			// exist loop when both stream is end or canceled form monitor.
			while( !(resultReader.isDone() && errorReader.isDone()) && !monitor.isCanceled() ){
				Thread.sleep(readSleepMs);
				monitor.worked(1);
			}
			
			// addition to task list if error exist.
			Object[] errorEntries = resultReader.errors.keySet().toArray();
			for (int i = 0; i < errorEntries.length; i++) {
				String[] entry = (String[])errorEntries[i];
				IResource res = findResourceFronPath( entry[0], resources);
				if(res != null && res.exists() ){
					IMarker marker = res.createMarker(IMarker.PROBLEM);
					marker.setAttribute(ID_MARKER_PUBLISHER_PROBLEM, true);
					marker.setAttribute(IMarker.USER_EDITABLE, true);
					String[] errorLines = (String[])resultReader.errors.get(entry);
					marker.setAttribute(IMarker.MESSAGE, errorLines[0]);
					marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
					// TODO: parse error and to settings task list archtecture
					// parser(res, errorLines);
				}
			}

			// store the unparsed error files.
			String[] lines = resultReader.getUnparsedLines();
			if( lines.length > 0){
				PrintWriter writer = new PrintWriter( new OutputStreamWriter(new FileOutputStream(UNPARSED_LINES_TEMPORARY_FILE), charset));
				try{
					for (int i = 0; i < lines.length; i++) {
						writer.println(lines[i]);
					}
				}finally{
					if( writer != null ){
						writer.close();
					}
				}
			}
			
			// refresh project publish folder resources.
			ArrayList refreshed = new ArrayList();
			for (int i = 0; i < resources.length; i++) {
				IResource r = resources[i];
				WebProject wp = (WebProject)r.getProject().getNature(WebProject.ID_NATURE);
				if( wp != null && !refreshed.contains(wp) ){
					refreshed.add(wp);
					wp.getFolder(WebProject.KEY_PUBLISH_FOLDER).refreshLocal(IResource.DEPTH_INFINITE, new SubProgressMonitor(monitor, 1));
				}
			}
			
			// raise if invalid stack trace exists
			String[] errorLog = errorReader.getErrorLog();
			RubyStacktrace[] traces = RubyStacktrace.createFromLines( errorLog);
			if( traces.length > 0){
				throw new CoreException(new Status(
					IStatus.ERROR,
					WebpubPlugin.ID_PLUGIN,
					IStatus.OK,
					traces[0].getMessage(),
					traces[0]));
			}
		}catch(InterruptedException e){
			WebpubPlugin.handleException(e);
		}catch(IOException e){
			IPath rubyInterpreter = WebpubPlugin.getDefault().getRubyCommandPath();
			String msg =
				!rubyInterpreter.toFile().exists() ?
					"Ruby interpreter is not exist: " + rubyInterpreter + " - Please settings in preference page \"Web Publisher > Ruby Interpreter\".":
					e.getMessage();
			
			throw new CoreException(new Status(
				IStatus.ERROR,
				WebpubPlugin.ID_PLUGIN,
				IStatus.OK,
				msg,
				e));
		}finally{
			monitor.done();
		}
	}
	
	String joinResourceArray(IResource[] resources){
		StringBuffer buff = new StringBuffer();
		for (int i = 0; i < resources.length; i++) {
			buff.append( resources[i].getLocation().toString() );
			if( i != resources.length - 1){
				buff.append(',');
			}
		}
		return buff.toString();
	}
	
	IResource findResourceFronPath( String resourcePath, IResource[] resources){
		for (int i = 0; i < resources.length; i++) {
			IResource res = resources[i];
			if( res.getLocation().toString().equals(resourcePath) ){
				return res;
			}
			
			// for cygwin, temporary storagy:
			// choose relative path from workspace name.
			// e.g, /cygdrive/c/workspace/fooproject/foofile to fooproject/foofile.
			// and choose that file from workspace root.
			IWorkspaceRoot root = res.getWorkspace().getRoot();
			String workspaceDirName = root.getLocation().lastSegment();
			int index = resourcePath.indexOf(workspaceDirName);
			if( index != -1){
				Path path = new Path( resourcePath.substring(index + workspaceDirName.length() ) );
				IFile f = root.getFile(path);
				if( f.exists() ){
					return f;
				}
			}
		}

		return null;
	}	
	
	/**
	 * Generate publish configuration csv file.
	 * for notify ruby command, extended consoles.
	 */
	IPath generatePublishConfig() throws CoreException {
		PublisherRegistory registory = PublisherRegistory.getInstance();
		
		Exception error = null;
		
		File file = new File( System.getProperty("java.io.tmpdir"), "publish.properties");
		BufferedWriter writer = null;
		try{
			FileOutputStream out = new FileOutputStream(file);
			writer = new BufferedWriter(new OutputStreamWriter(out));
			
			String[] byKeys = registory.getPublishByKeys();
			for (int i = 0; i < byKeys.length; i++) {
				String k = byKeys[i];
				writer.write(k + "," + registory.getPublishScriptFileLocation(k) + "," + registory.getPublishClassName(k) + "\n");
			}
		} catch (FileNotFoundException e) {
			error = e;
		} catch (IOException e) {
			error = e;
		} finally{
			try {
				if(writer != null){
					writer.close();
				}
			} catch (IOException e) {
				error = e;
			}
			if( error != null){
				throw new CoreException(new Status(
					IStatus.ERROR,
					WebpubPlugin.ID_PLUGIN,
					IStatus.ERROR,
					"Error occasion in publish properties file generation",
					error));
			}
		}
		
		return new Path(file.toString());
	}
	
}

