/*
 * $Id: PublishDescriptionFactory.java,v 1.1 2004/01/17 15:51:55 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.publish;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import com.narucy.webpub.core.*;

/**
 * <p>
 * PublishDescriptionFactory creates HTPublishDescriptor that for selection
 * a html document publish script file and method.
 * 
 * <p>
 * To careate, call HTpublishDescriptorFactory#createDescriptor method.
 * first, searching to way from that method specify file. If it can not
 * fount publish way, search parent directories refrect to parent.
 */
final public class PublishDescriptionFactory {

	final public static String
		PUBLISH_DESCRIPTION_FILENAME = ".publish";

	final static String
		PUBLISHCODE_BEGIN = "<?publish ",
		PUBLISHCODE_END = "?>";		
	
	static PublishDescriptionFactory instance = new PublishDescriptionFactory();
	
	public static PublishDescriptionFactory getInstance(){
		return instance;
	}

	DocumentBuilderFactory docBuilderFac = DocumentBuilderFactory.newInstance();
	String[] ignorePatterns = {"CVS", ".publish"};
	
	private PublishDescriptionFactory(){}
	
	/**
	 * <p>
	 * To create PublishDescription from iterator. 
	 * This method for on memory document data.
	 * 
	 * <p>
	 * This method can not automatically set a publish location.
	 */
	public PublishDescription create(Iterator ite) throws IllegalConfigurationException {
		while(ite.hasNext()){
			String line = (String)ite.next();
			if(line == null){
				continue;
			}
			int begin = line.indexOf(PUBLISHCODE_BEGIN);
			int end = line.indexOf(PUBLISHCODE_END);
			if( begin != -1 && end != -1){
				PublishDescription desc = createFromMatcher(
					line.substring(begin + PUBLISHCODE_BEGIN.length(), end));
				return desc;
			}
		}
		return null;
	}

	/**
	 * <p>
	 * Create PublishDescription (this method is define publish location from
	 * referer htFile location that process is not exec other method).
	 * 
	 * <p>
	 * Return null if do not found publish description.
	 */
	public PublishDescription create(IResource file) throws IllegalConfigurationException, CoreException, IOException {
		if( !checkPublishSource(file) || isPublishIgnore(file) ){
			return null;
		}
		
		PublishDescription desc = null;
		if( file instanceof IFile){
			desc = createFromFile((IFile)file);
		}
		if(desc == null){
			// if can not create specify html file.
			// search generator property in parent folder.
			desc = createFromParent(file);
		}
		if( desc != null){
			// if creation successfuly, initialize publish location.
			initPublishLocation(file, desc);
		}
		return desc;
	}
	

	public PublishDescription createFromFile(IFile file) throws IllegalConfigurationException, CoreException,IOException {
		WebProject wp = (WebProject)file.getProject().getNature(WebProject.ID_NATURE);
		if( !file.exists() || isPublishIgnore(file) ||
				!file.isLocal(IResource.DEPTH_ZERO) || wp == null || !wp.isHTExtension(file.getFileExtension())){
			
			return null;
		}
		TextReader reader = null;
		try{
			reader = new TextReader(file.getContents());
			return create(reader);
		}
		catch( IllegalConfigurationException e){
			e.illegalFile = file;
			throw e;
		}
		finally{
			if(reader != null){
				reader.close();
			}
		}
	}
		
	void initPublishLocation(IResource source, PublishDescription desc) throws CoreException, IllegalConfigurationException {
		if( desc.getPublishTo() != null ){
			// already initialized.
			return;
		}
		Pattern wildCard = Pattern.compile("(.+)/\\*\\.(.+?)$");
		
		IProject proj = source.getProject();
		WebProject webProj = (WebProject)proj.getNature(WebProject.ID_NATURE);
		if(webProj == null){
			// illegal placed publish description file (.publish)
			return;
		}
		IContainer publishFolder = webProj.getFolder(WebProject.KEY_PUBLISH_FOLDER);
		
		String publishPath = desc.getArgument("publish_to");
		if( publishPath != null){
			// first char is '/', remove this.
			if (publishPath.charAt(0) == '/'){
				publishPath = publishPath.substring(1);
			}
			
			// attribute is setted, referer this values.
			IResource res;
			Matcher m;
			if(publishPath.charAt(publishPath.length()-1) == '/'){
				// specified direcotry
				res = publishFolder.getFolder(new Path(publishPath));
			}else if ( (m = wildCard.matcher(publishPath)).matches() ){
				// use wild card, provisional division.
				IFolder folder = publishFolder.getFolder(new Path(m.group(1)));
				String ext = source.getFileExtension();
				String publishFileName = source.getName().replaceFirst(ext + "$",  m.group(2));
				res = folder.getFile(publishFileName);
			}else{
				// specify file.
				res = publishFolder.getFile(new Path(publishPath));
			}
			
			desc.setPublishTo(res);
			desc.setArgument("publish_to", null);
		}else{
			IResource publishLocation = htSourceLocationToPublishLocation(source);
			if( publishLocation != null){
				desc.setPublishTo( publishLocation );
			} else{
				throw new IllegalConfigurationException("can not define publish location:" + desc);
			}
		}
		
	}
	
	/**
	 * Returns copy targets (publish location) from specify resource location.
	 * If specify location is not able to define a publish location, return null.
	 */
	IResource htSourceLocationToPublishLocation(IResource res) throws CoreException{
		WebProject webProj = (WebProject)res.getProject().getNature(WebProject.ID_NATURE);
		
		String htSourceFolder = webProj.getFolder(WebProject.KEY_HTSOURCES_FOLDER).getFullPath().toString();
		String resPath = res.getFullPath().toString();
		
		if (resPath.indexOf(htSourceFolder) != -1){
			String relativePath = resPath.substring(
				htSourceFolder.toString().length()+1, resPath.length() );
			return webProj.getFolder(WebProject.KEY_PUBLISH_FOLDER).getFile( new Path(relativePath) );
		}
		
		return null;

	}
	
	/**
	 * Creates publish description from parent folder on reflexive.
	 * If none looking publish description in html source folder, return null.
	 */
	PublishDescription createFromParent(IResource targetFile) throws IllegalConfigurationException, CoreException {
		PublishDescription desc = null;
		
		IFile publishDescriptionFile = findPublishDescriptionFile(targetFile);
		if(publishDescriptionFile != null){
			desc = createFromPropertyFile(targetFile, publishDescriptionFile);
		}
		if( desc == null){
			desc = createDefaultPublishDescription(targetFile);
		}
		return desc;
	}
	
	/**
	 * Return null if can not publish location as specify file.
	 */
	public IFile findPublishDescriptionFile(IResource target) throws CoreException {
		if( checkPublishSource(target) ){
			for(IResource res = target; (res = res.getParent()) instanceof IFolder; ){
				IResource propFile = ((IFolder)res).findMember(PUBLISH_DESCRIPTION_FILENAME);
				if(propFile instanceof IFile && propFile.exists()){
					return (IFile)propFile;
				}
			}
		}
		return null;
	}


	/**
	 * Return null if can not publish location as specify file.
	 */
	PublishDescription createDefaultPublishDescription(IResource targetFile) throws CoreException{
		if( targetFile.exists() ){
			WebProject webProj = (WebProject)targetFile.getProject().getNature(WebProject.ID_NATURE);
			if( webProj != null){
				IContainer htSourceFolder = webProj.getFolder(WebProject.KEY_HTSOURCES_FOLDER);
				for(IResource res = targetFile; (res = res.getParent()) instanceof IFolder; ){
					if (res.equals(htSourceFolder) ){
						IResource copyTarget = htSourceLocationToPublishLocation(targetFile);
						if( copyTarget != null ){
							PublishDescription desc = new PublishDescription(PublishDescription.BY_COPY);
							desc.setPublishTo(copyTarget);
							return desc;
						}
					}
				}
			}
		}
		
		return null;
	}

	/**
	 * Create from properties file.
	 */
	PublishDescription createFromPropertyFile( IResource source, IFile propFile) throws IllegalConfigurationException {
		InputStream stream = null;
		Exception error = null;
		try {
			stream = propFile.getContents();
			
			// parse the ".publish" file.
			Document doc = docBuilderFac.newDocumentBuilder().parse(stream);
			Element rootElem = doc.getDocumentElement();
			NodeList mappingNodes = rootElem.getElementsByTagName("mapping");
			
			// to relative path
			String relativePath =
				source.getFullPath().removeFirstSegments(
					propFile.getParent().getFullPath().segmentCount() ).toString();
			
			for(int i=0; i<mappingNodes.getLength(); i++){
				Element elem = (Element)mappingNodes.item(i);
				if( isMatch(elem.getAttribute("pattern"), relativePath)){
					Element genElem = (Element)elem.getElementsByTagName("publish").item(0);
					if(genElem.getAttribute("by") != null){
						 return createAsElem(genElem);
					}
				}
			}
		} catch (SAXException e) {
			error = e;
		} catch (IOException e) {
			error = e;
		} catch (ParserConfigurationException e) {
			error = e;
		} catch(CoreException e){
			error = e;
		} catch (IllegalConfigurationException e) {
			e.illegalFile = propFile;
			throw e;
		} finally{
			if(stream != null){
				try {
					stream.close();
				} catch (IOException e) {
					error = e;
				}
			}
			if( error != null){
				throw new IllegalConfigurationException(error);
			}
		}
		return null;
	}
		
	PublishDescription createAsElem(Element genElem) throws IllegalConfigurationException{
		HashMap map = new HashMap();
		NamedNodeMap atts = genElem.getAttributes();
		for(int i=0; i<atts.getLength(); i++){
			Node node = atts.item(i);
			map.put(node.getNodeName(), node.getNodeValue());
		}
		return createFromMap(map);
	}
	
	/**
	 * Parse as follow style line, and return that created PublishDescriptor
	 * form line.
	 * <pre>
	 *   file="home3.rb" class="Home" method="home" publish_to="foo/home.html
	 * </pre>
	 */
	PublishDescription createFromMatcher(String code) throws IllegalConfigurationException{
		return createFromMap( createMapFromLine(code) );
	}
	
	/**
	 * Create PublishDescriptor from specify Map represents properties.
	 */
	PublishDescription createFromMap(Map map) throws IllegalConfigurationException{
		String publishBy = (String)map.remove("by");
		PublishDescription desc = new PublishDescription(publishBy);
		
		Object[] keys = map.keySet().toArray();
		for (int i = 0; i < keys.length; i++) {
			String k = (String)keys[i];
			desc.setArgument( k, (String)map.get(k) );
		}
		
		return desc;
	}
	
	Map createMapFromLine(String code){
		StringTokenizer tokenizer = new StringTokenizer(code);	
		Map map = new HashMap();
		
		while (tokenizer.hasMoreTokens()) {
			String keyValue = tokenizer.nextToken();
			int divideIndex = keyValue.indexOf('=');
			if( divideIndex == -1){
				throw new RuntimeException("invalid token:" + keyValue);
			}
				
			String key = keyValue.substring(0, divideIndex).trim();
			String value = toAttributeValidate(keyValue.substring(divideIndex+1));
			if( key == null || value == null){
				throw new RuntimeException("invalid token:" + keyValue);
			}
			map.put(key, value);
		}
		
		return map;
	}
	
	String toAttributeValidate(String v){
		v = v.trim();
		if( v.charAt(0) == '"' && v.charAt(v.length()-1) == '"'){
			return v.substring(1, v.length()-1);
		}
		else{
			return null;
		}
	}

	public String[] getIgnorePatterns() {
		return (String[])ignorePatterns.clone();
	}

	public void setIgnorePatterns(String[] strings) {
		ignorePatterns = (String[])strings.clone();
	}
	
	/**
	 * Return true if publish source file location is valid.
	 * (file exist and not underly public folder.)
	 */
	boolean checkPublishSource(IResource res) throws CoreException {
		// checks file exists.
		if(!res.exists()){
			return false;
		}
		
		// checks, publish folder under file is not publish.
		WebProject wp = (WebProject)res.getProject().getNature(WebProject.ID_NATURE);
		if( wp != null){
			IContainer pubFolder = wp.getFolder(WebProject.KEY_PUBLISH_FOLDER);
			IResource r = res;
			while( ((r = r.getParent()) instanceof IFolder) ){
				if( pubFolder.equals(r) ){
					return false;
				}
			} 
		}
		
		return true;
	}

	/**
	 * checks publish pattern is match.
	 */
	boolean isPublishIgnore(IResource resource) throws CoreException {
		String name = resource.getName();
		for(int i=0; i<ignorePatterns.length; i++){
			String pattern = ignorePatterns[i];
			if( isMatch(pattern, name) ){
				return true;
			}
		}
		
		return false;
	}
	
	static boolean isMatch(String pattern, String relativePath){
		if( pattern.equalsIgnoreCase(relativePath) ){
			return true;
		} else if( isRegexpPattern(pattern) ){
			// if enclose '/' chars use regexp
			String regexp = pattern.substring(1, pattern.length()-1);
			return Pattern.matches(regexp, relativePath);
		}else{
			// if not cnclose '/', use wild card.
			Wildcard wildcard = new Wildcard(pattern);
			return wildcard.match(relativePath);
		}
	}
	
	static void p(Object o) {
		System.out.println(":" + o);
	}

	static boolean isRegexpPattern(String pattern){
		return (pattern.charAt(0) == '/' && pattern.charAt(pattern.length()-1) == '/');
	}

}

