/*
 * $Id: WebpubPlugin.java,v 1.1 2004/01/17 15:51:53 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core;

import java.io.File;
import java.io.IOException;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.toc.*;

/**
 * WebpubPlugin has base data of the plugin.
 */
public class WebpubPlugin extends Plugin {

	final public static String
		ID_PLUGIN = "com.narucy.webpub.core";

	final static String[] defaultFindPath = {
		System.getProperty("user.rubypath"),
		"/usr/local/bin",
		"/usr/bin",
		"/bin",
		"c:/cygwin/usr/local/bin",
		"c:/cygwin/usr/bin",
		"c:/cygwin/bin",
		"c:/ruby/bin",
	};
	
	final static String[] supportCharsets = {
		"UTF-8",
		"Shift_JIS",
		"EUC-JP",
	};

	final static String[] interpreterFileNames = {
		"ruby",
		"ruby.exe",
	};

	static WebpubPlugin plugin;
	static ResourceBundle resourceBundle;
	
	IPath rubyCommandPath;
	WebProjectAdapterFactory webProjectAdapterFactory = new WebProjectAdapterFactory();
	
	Object[] adapterFactories = new Object[]{
		new TocAdapterFactory(), Topic.class,
		new FileTocAdapterFactory(), IFile.class,
		new WebProjectAdapterFactory(), IProject.class,
	};

	public WebpubPlugin(IPluginDescriptor descriptor) {
		super(descriptor);
		plugin = this;
		try {
			resourceBundle= ResourceBundle.getBundle("webpub");
		} catch (MissingResourceException x) {
			resourceBundle = null;
		}
		rubyCommandPath = findInitialInterpreter();
	}

	public static String[] getSupportCharsets() {
		return (String[])supportCharsets.clone();
	}

	public static IPath findInitialInterpreter(){
		for(int i=0; i<defaultFindPath.length; i++){
			for(int j=0; j<interpreterFileNames.length; j++){
				File file = new File(defaultFindPath[i], interpreterFileNames[j]);
				if(file.exists()){
					return new Path(file.toString());
				}
			}
		}
		return null;
	}
	
	public static WebpubPlugin getDefault(){
		return plugin;
	}
	
	public static ResourceBundle getResourceBundle(){
		return resourceBundle;
	}
		
	public IPath getRubyCommandPath(){
		return rubyCommandPath;
	}
	
	public void setRubyCommandPath(IPath rubyCommandPath){
		this.rubyCommandPath = rubyCommandPath;
	}

	public void startup() {
		IAdapterManager manager= Platform.getAdapterManager();
		for(int i=0; i<adapterFactories.length; i+=2){
			manager.registerAdapters( (IAdapterFactory)adapterFactories[i], (Class)adapterFactories[i+1] );
		}
	}

	public void shutdown() {
		IAdapterManager manager= Platform.getAdapterManager();
		for(int i=0; i<adapterFactories.length; i+=2){
			manager.unregisterAdapters( (IAdapterFactory)adapterFactories[i]);
		}
	}
	
	public Process rubyExec(String arguments, IPath workingDirectory) throws IOException {
		return WebpubPlugin.rubyExec(rubyCommandPath.toString(), arguments, workingDirectory);
	}
	
	public static Process rubyExec(String rubyCommandPath, String arguments, IPath workingDirectory) throws IOException {
		String cmd = rubyCommandPath + ' ' +  arguments;
		return Runtime.getRuntime().exec(cmd, null, workingDirectory != null ? workingDirectory.toFile() : null);
	}

	public static void handleException(Exception e) {
		e.printStackTrace();
		plugin.getLog().log(new Status(IStatus.ERROR, ID_PLUGIN, IStatus.OK, e.getMessage(), e));
	}		

}
