/*
 * $Id: WebProject.java,v 1.1 2004/01/17 15:51:53 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core;

import java.io.FileOutputStream;
import java.io.IOException;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.publish.PublisherBuilder;
import com.narucy.webpub.core.template.TemplateSynchronizeBuilder;
import com.narucy.webpub.core.toc.Toc;
import com.narucy.webpub.core.toc.TocFactory;

/**
 * WebProject represents web resources keeps web project.
 */
public class WebProject extends Preferences implements IProjectNature {

	/**
	 * File name of web project nature configuration file.
	 */
	final public static String DESCRIPTION_FILENAME = ".webproject";
	
	/**
	 * The web project nature id.
	 */
	final public static String ID_NATURE = WebpubPlugin.ID_PLUGIN + ".publishNature";
	
	/**
	 * Reference key for html source folder.
	 */
	final public static String KEY_HTSOURCES_FOLDER = "htsources_folder";
	
	/**
	 * Reference key for scripts folder.
	 */
	final public static String KEY_SCRIPTS_FOLDER = "scripts_folder";
	
	/**
	 * Reference key for publish folder.
	 */
	final public static String KEY_PUBLISH_FOLDER = "publish_folder";
	
	/**
	 * Referencekey for template synchronization extensions, and
	 * file specify publish definication featture (&gt;?publish by="foo"?&lt;) enabled.
	 */
	final public static String KEY_HT_EXTENSIONS = "ht_extensions";
	
	/**
	 * Reference key for HTML publish and template synchironiation process using
	 * encoding.
	 */
	final public static String KEY_HT_CHARSET = "ht_charset";
	
	/**
	 * Refernce key for toc file pathes.
	 */
	final public static String KEY_TOC_PATHES = "toc_pathes";
	
	/**
	 * Key for list of web contents mapping url.
	 */
	final public static String KEY_MAPPED_URL = "mapped_url";

	IProject project = null;
	HashMap tocCache = new HashMap();

	public WebProject(){
		setDefault(KEY_HTSOURCES_FOLDER, "");
		setDefault(KEY_SCRIPTS_FOLDER, "");
		setDefault(KEY_PUBLISH_FOLDER, "");
		
		setDefault(KEY_HT_EXTENSIONS, new String[]{"html", "htm", "erb"} );
		setDefault(KEY_HT_CHARSET, "UTF-8");
		setDefault(KEY_TOC_PATHES, "");
		setDefault(KEY_MAPPED_URL, "");
	}

	IResourceChangeListener resourceChangeListener = new IResourceChangeListener() {
		public void resourceChanged(IResourceChangeEvent event) {
			try {
				handleResourceChange(event);
			} catch (CoreException e) {
				WebpubPlugin.handleException(e);
			}
		}
	};
	
	void handleResourceChange(IResourceChangeEvent event) throws CoreException{
		IResourceDelta delta = event.getDelta();
		if(delta == null){
			return;
		}
		delta.accept(new IResourceDeltaVisitor() {
			public boolean visit(IResourceDelta delta) throws CoreException {
				IResource res = delta.getResource();
				if(res instanceof IProject && delta.getKind() == IResourceDelta.REMOVED && res.equals(project)){
					res.getWorkspace().removeResourceChangeListener(resourceChangeListener);
				}
				return true;
			}
		});
	}
	
	/**
	 * Project owner explicit specified toc files.
	 * (return files are no check: file exist and specify file is valid for toc.
	 */
	public IFile[] getTocFiles(){
		String[] pathes = getArray(KEY_TOC_PATHES);
		IFile[] files = new IFile[pathes.length];
		for (int i = 0; i < pathes.length; i++) {
			files[i] = project.getFile(pathes[i]);
		}
		return files;
	}
	
	void removeTocCache(IFile f) throws CoreException{
		tocCache.remove(f);
	}

	public void setProject(IProject p) {
		if( project != p){
			project = p;
			p.getWorkspace().addResourceChangeListener(resourceChangeListener, IResourceChangeEvent.POST_CHANGE);
			try {
				load();
			} catch (CoreException e) {
				WebpubPlugin.handleException(e);
			}
		}
	}
	
	public void setDefault(String name, String[] vals){
		setDefault(name, toLine(vals));
	}
	
	public void setValue(String name, String[] vals){
		setValue(name, toLine(vals));
	}
	
	public String[] getArray(String name){
		return toArray(getString(name));
	}
	
	public String[] getDefaultArray(String name){
		return toArray(getDefaultString(name));
	}
	
	public static String[] toArray(String s){
		return s.length() > 0 ? s.split("\\s+") : new String[0];
	}

	public static String toLine(String[] val){
		StringBuffer buff = new StringBuffer();
		for(int i=0; i<val.length; i++){
			buff.append(val[i]);
			if(i != val.length - 1){
				buff.append(' ');
			}
		}
		return buff.toString();
	}
	
	public boolean isHTExtension(String extension){
		if( extension != null){
			String[] exts = getArray(WebProject.KEY_HT_EXTENSIONS);
			Arrays.sort(exts);
			return Arrays.binarySearch(exts, extension) >= 0;
		}
		return false;
	}

	public void load() throws CoreException{
		IFile descFile = project.getFile(DESCRIPTION_FILENAME);
		if( descFile.isAccessible()){
			try {
				load(descFile.getContents());
				// compatibility for old version.
				if( getString(KEY_HTSOURCES_FOLDER).equals( getString(KEY_PUBLISH_FOLDER)) ){
					System.out.println("Convert web project nature configuration file:" + WebProjectConfigurationLoader052.reloadDescription(this) );
					storePreferences();
				}
			} catch (IOException e) {
				throw new CoreException(
					new Status(
						IStatus.ERROR,
						WebpubPlugin.ID_PLUGIN,
						IStatus.OK,
						"Can not read web project configuration file:" + descFile,
						e));
			}
		}
	}

	public void configure() throws CoreException {
		IProjectDescription desc = project.getDescription();
		ICommand cmds[] = desc.getBuildSpec();
		String[] cmdIds = new String[cmds.length];
		
		ArrayList newCmds = new ArrayList();
		for (int i = 0; i < cmds.length; i++) {
			ICommand cmd = cmds[i];
			cmdIds[i] = cmd.getBuilderName();
			newCmds.add(cmd);
		}
		Arrays.sort(cmdIds);
		
		if(Arrays.binarySearch(cmdIds, TemplateSynchronizeBuilder.ID_BUILDER) < 0){
			ICommand tmplCmd = desc.newCommand();
			tmplCmd.setBuilderName(TemplateSynchronizeBuilder.ID_BUILDER);
			newCmds.add(tmplCmd);
		}
		if(Arrays.binarySearch(cmdIds, PublisherBuilder.ID_BUILDER) < 0){
			ICommand publishCmd = desc.newCommand();
			publishCmd.setBuilderName(PublisherBuilder.ID_BUILDER);
			newCmds.add(publishCmd);
		}
		desc.setBuildSpec((ICommand[])newCmds.toArray(new ICommand[newCmds.size()]));
		project.setDescription(desc, new NullProgressMonitor());
	}
	
	public void deconfigure() throws CoreException {
	}
	
	/**
	 * <p>
	 * Returns web project specify resources (these are base html document
	 * folder, for html generation scripts folder, public document folder)
	 * path.
	 * 
	 * <p>
	 * @param key specify HTSOURCES_FOLDER, SCRIPTS_FOLDER, PUBLIC_FOLDER
	 */
	public IContainer getFolder(String key){
		return pathToContainer(getString(key));
	}

	/**
	 * <p>
	 * Argument string folderPath represetnts path of web base container.
	 *
	 * Return values:
	 * <pre>
	 *   /base     : IProject /base
	 *   foo       : IFolder  /(this project)/foo
	 *   /base/foo : IFolder /base/foo
	 * </pre>
	 * 
	 * <p>
	 * Does not check exist, sync, open container.
	 */
	public IContainer pathToContainer(String folderPath){
		if( folderPath.length() == 0){
			return project;
		}
		if( folderPath.charAt(0) == '/'){
			Path path = new Path(folderPath);
			IWorkspaceRoot root = project.getWorkspace().getRoot();
			IProject proj = root.getProject(path.segment(0));
			
			if(path.segmentCount() <= 1){
				return proj;
			}else{
				return proj.getFolder(path.removeFirstSegments(1));
			}
		}else{
			return project.getFolder(folderPath);
		}
	}

	public IProject getProject() {
		return project;
	}
	
	/**
	 * Return null load failed.
	 */
	public Toc getToc(IFile f){
		Toc toc = (Toc)tocCache.get(f);
		if(toc == null){
			try{
				toc = TocFactory.createToc(f);
				tocCache.put(f, toc);
			} catch (CoreException e){
				WebpubPlugin.handleException(e);
			}
		}
		return toc;
	}

	public void storePreferences() throws CoreException {
		try{
			IFile descFile = project.getFile(DESCRIPTION_FILENAME);
			FileOutputStream out = null;
			try{
				out = new FileOutputStream( descFile.getLocation().toFile() );
				store(out, null);
			}finally{
				if(out != null){
					out.close();
				}
			}
			descFile.refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor() );
		}catch(IOException e){
			WebpubPlugin.handleException(e);
		}
	}

	void distributeFolders(IProgressMonitor monitor) throws CoreException {
		try{
			monitor.beginTask("Create Web Project Folders", 3);
			// create folders
			String[] keys = {
				KEY_HTSOURCES_FOLDER,
				KEY_SCRIPTS_FOLDER,
				KEY_PUBLISH_FOLDER
			};
			for (int i = 0; i < keys.length; i++) {
				IContainer folder = getFolder(keys[i]);
				if(!folder.exists()){
					if(folder instanceof IFolder){
						((IFolder)folder).create(true, true, new SubProgressMonitor(monitor, 1));
					}else{
						throw new CoreException(new Status(IStatus.ERROR, WebpubPlugin.ID_PLUGIN, IStatus.OK, "Specify container is not exist:" + folder, null));
					}
				}
			}
		}
		finally{
			monitor.done();
		}
	}

	/**
	 * Search file members from specify folder key name.
	 * If specify key is null, returns all of project files.
	 */
	public IFile[] findFileMembers(String key) throws CoreException {
		return findFileMembers( getFolder(key) );
	}
	
	static IFile[] findFileMembers(IContainer folder) throws CoreException {
		ArrayList dist = new ArrayList();
		findFileMembers(folder, dist);
		return (IFile[])dist.toArray(new IFile[dist.size()]);
	}
	
	static void findFileMembers(IContainer folder, ArrayList distlist) throws CoreException{
		IResource[] files = folder.members();
		for (int i = 0; i < files.length; i++) {
			IResource res = files[i];
			if(res instanceof IFolder){
				findFileMembers( (IFolder)res, distlist);
			} else if(res instanceof IFile){
				distlist.add( res );
			}
		}
	}

	/**
	 * Distrubte new project.
	 */
	public static void storeNewProject(IProject project, String htSourceFolder, String scriptsFolder, String publicFolder, IProgressMonitor monitor) throws CoreException {
		try {
			monitor.beginTask("Store new web project", 4);
			
			// sets nature
			IProjectDescription desc =
				project.getWorkspace().newProjectDescription(project.getName());
			desc.setNatureIds( new String[]{ID_NATURE} );
			
			// project initialize
			boolean created = project.exists();
			if(!created){
				project.create(desc, new SubProgressMonitor(monitor,1));
			}
			if(!project.isOpen()){
				project.open(new SubProgressMonitor(monitor, 1));
			}
			if(created){
				project.setDescription(
					desc, IResource.FORCE, new SubProgressMonitor(monitor,1) );
			}
			
			// store configuration file
			WebProject wp = (WebProject)project.getNature(ID_NATURE);
			wp.setProject(project);
			wp.setValue(KEY_HTSOURCES_FOLDER, htSourceFolder);
			wp.setValue(KEY_SCRIPTS_FOLDER, scriptsFolder);
			wp.setValue(KEY_PUBLISH_FOLDER, publicFolder);
			wp.storePreferences();

			// create folders.
			wp.distributeFolders(new SubProgressMonitor(monitor,1));
			
			// TODO: why call manually configure() method?
			wp.configure();
			
		} finally{
			monitor.done();
		}
	}

}
