/*
 * $Id: FileUtilities.java,v 1.1 2004/01/17 15:51:54 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core;

import java.io.*;
import java.util.regex.Pattern;

/**
 * FileUtilities represents refrexible a file.
 */
public class FileUtilities {

	public static File copyWithDir(String fromDir, String to, Pattern[] ignoreFilePatterns) throws IOException {
		return copyWithDir(new File(fromDir), new File(to), ignoreFilePatterns);
	}

	public static File copyWithDir(File fromDir, File to, Pattern[] ignoreFilePatterns) throws IOException {
		if (!fromDir.isDirectory()) {
			throw new IllegalArgumentException("specify copy target directory is not");
		}
		File toDir = new File(to, fromDir.getName());
		if (!toDir.exists()) {
			toDir.mkdirs();
		}

		copyWithDirContainFiles(fromDir, toDir, ignoreFilePatterns);

		return toDir;
	}

	public static void copyWithDirContainFiles(File fromDir, File to, Pattern[] ignoreFilePatterns) throws IOException {
		File[] files = fromDir.listFiles();
		for (int i = 0; i < files.length; i++) {
			File f = files[i];
			if (isMatch(ignoreFilePatterns, f)) {
				continue;
			}
			if (f.isDirectory()) {
				File d = new File(to, f.getName());
				d.mkdir();
				copyWithDirContainFiles(f, d, ignoreFilePatterns);
			} else if (f.isFile()) {
				copy(f, to);
			}
		}
	}

	static boolean isMatch(Pattern[] patterns, File f) {
		if (patterns == null) {
			return true;
		}
		for (int i = 0; i < patterns.length; i++) {
			Pattern pattern = patterns[i];
			if (pattern.matcher(f.getName()).matches()) {
				return true;
			}
		}
		return false;
	}

	public static void copy(String from, String to) throws IOException {
		copy(new File(from), new File(to));
	}

	public static File copy(File from, File to) throws IOException {
		if (!from.isFile()) {
			throw new IllegalArgumentException("specify copy target file path is not file: " + from);
		}
		if (to.isDirectory()) {
			to = new File(to, from.getName());
		}
		FileInputStream in = null;
		FileOutputStream out = null;
		try {
			in = new FileInputStream(from);
			out = new FileOutputStream(to);
			byte[] byteRead = new byte[1024];
			int read;
			while ((read = in.read(byteRead)) != -1) {
				out.write(byteRead, 0, read);
			}
		} finally {
			in.close();
			out.close();
		}
		return to;
	}

	public static void deleteDirectory(File dir) {
		if (!dir.isDirectory()) {
			throw new IllegalArgumentException("Specify path is not directory:" + dir);
		}
		File[] files = dir.listFiles();
		for (int i = 0; i < files.length; i++) {
			File f = files[i];
			if (f.isDirectory()) {
				deleteDirectory(f);
			} else if (f.isFile()) {
				f.delete();
			}
		}
		dir.delete();
	}

}
