/*  NDKmol - Molecular Viewer on Android NDK

     (C) Copyright 2011, biochem_fan

     This file is part of NDKmol.

     NDKmol is free software: you can redistribute it and/or modify
     it under the terms of the GNU Lesser General Public License as published by
     the Free Software Foundation, either version 3 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU Lesser General Public License for more details.

     You should have received a copy of the GNU Lesser General Public License
     along with this program.  If not, see <http://www.gnu.org/licenses/>. */

package jp.sfjp.webglmol.NDKmol;

import java.io.File;
import java.io.FileWriter;
import java.io.InputStream;

import jp.xii.relog.customlibrary.app.FileListDialog;
import jp.xii.relog.customlibrary.app.FileListDialog.onFileListDialogListener;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.res.Resources;
import android.net.Uri;
import android.opengl.GLSurfaceView;
import android.os.Build;
import android.os.Bundle;
import android.os.Environment;
import android.text.InputFilter;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.widget.EditText;

public class NDKmolActivity extends Activity {
	private float startX, startY, startDistance;
	private float currentX, currentY, currentZ;
	private float currentCameraZ;
	private boolean isDragging;
	private Quaternion currentQ;
	private int mode = 2;
	private int prevPointerCount = 0;
	private boolean multiTouchEnabled = false;
	
	static {
		System.loadLibrary("Ndkmol");
	}

	public GLSurfaceView glSV;
	public NdkView view; 

	void initializeResource() {
		String targetPath = getDataDir() + "/sample.pdb";
		Log.d("NDKmol", "Initializing sample data " + targetPath);
		File target = new File(targetPath);
		if (target.exists()) return;

		try {
			FileWriter out= new FileWriter(target);
			out.write(readResource(R.raw.initial));
			out.close();
		} catch (Exception e) {
			Log.d("initializeResource", "failed: " + e.toString());
		}
	}

	String readResource(int resourceId) {
		String ret = ""; 

		Resources res = this.getResources();
		InputStream st = null;
		try {
			st = res.openRawResource(resourceId);
			byte[] buffer = new byte[st.available()];
			while((st.read(buffer)) != -1) {}
			st.close();
			ret = new String(buffer);
		} catch (Exception e) {
			Log.d("ResourceOpen", e.toString());
		} finally{
		}
		return ret;
	}

	/** Called when the activity is first created. */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		
		if (Build.VERSION.SDK_INT >= 6) {
			multiTouchEnabled = true;
		}
		
		initializeResource();

		glSV = new GLSurfaceView(this);
		view = new NdkView();
		glSV.setRenderer(view);

		glSV.setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
		setContentView(glSV);
		view.loadPDB(getDataDir() + "/sample.pdb");
		glSV.requestRender();
	}

	
	public void open() {
		FileListDialog dialog = new FileListDialog(this);
		dialog.setDirectorySelect(false);
		dialog.setOnFileListDialogListener(new onFileListDialogListener() {
			public void onClickFileList(File file) {
				if(file == null){
				}else{
					try {
						readFile(file.getCanonicalPath());
					} catch (Exception e) {
						Log.d("Open", "Failed to open" + file.getAbsolutePath());
					}
				}
			}
		});

		dialog.show(getDataDir(), "Load PDB file");
	}
	
	public void readFile(String path) {
		if (path.toUpperCase().endsWith("PDB")) {
			view.loadPDB(path);
		} else if (path.toUpperCase().endsWith("SDF") || path.toUpperCase().endsWith("MOL")) {
			view.loadSDF(path);
		} else {
			alert(getString(R.string.unknownFileType));
		}
		
		view.prepareScene(true);
		view.objX = 0;
		view.objY = 0;
		view.objZ = 0;
		view.cameraZ = -300;
		glSV.requestRender();
	}
	
	public String getDataDir() { // FIXME: not working?
		String dataDir = "";
		if (!Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
			dataDir = this.getFilesDir().getAbsolutePath();
		} else {
			File sdfolder = new File(Environment.getExternalStorageDirectory().getAbsoluteFile() + "/PDB");
			sdfolder.mkdirs();
			dataDir = sdfolder.getAbsolutePath();
		}
		return dataDir;
	}
	
	public void alert(String msg) {
		AlertDialog.Builder builder = new AlertDialog.Builder(this);
		builder.setMessage(msg)
		.setCancelable(false)
		.setPositiveButton("OK", new DialogInterface.OnClickListener() {
			public void onClick(DialogInterface dialog, int id) {
			} 
		});
		AlertDialog alert = builder.create();
		alert.show();
	}
	
	public void download() {
		final EditText editView = new EditText(this);
		final NDKmolActivity parent = this;
		InputFilter[] inputFilter = new InputFilter[1];
		inputFilter[0] = new InputFilter.LengthFilter(4);
		editView.setFilters(inputFilter);
		new AlertDialog.Builder(this)
		.setIcon(android.R.drawable.ic_dialog_info)
		.setTitle("Input PDB ID to download")
		.setView(editView)
		.setPositiveButton("Download", new DialogInterface.OnClickListener() {
			public void onClick(DialogInterface dialog, int whichButton) {
				String urlPrefix = "http://www.rcsb.org/pdb/files/";
				String urlSuffix = ".pdb";
				String id = editView.getText().toString().toUpperCase();
				if (id.matches("^[a-zA-Z0-9]{4}$")) {
					new Downloader(parent, urlPrefix + id + urlSuffix, getDataDir() + "/" +  id + urlSuffix);
				} else {
					alert("PDB ID is not correct.");
				}
			}
		})
		.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
			public void onClick(DialogInterface dialog, int whichButton) {
			}
		})
		.show();
	}

	public void preferences() {
		Intent intent = new Intent(this, MyPreferenceActivity.class);
		startActivity(intent);
	}
	
	public void about() {
		AlertDialog.Builder builder = new AlertDialog.Builder(this);
		builder.setMessage(getString(R.string.about))
		.setCancelable(false)
		.setPositiveButton("OK", new DialogInterface.OnClickListener() {
			public void onClick(DialogInterface dialog, int id) {
			}
		})
		.setNeutralButton("Go to website", new DialogInterface.OnClickListener() {
			public void onClick(DialogInterface dialog, int id) {
				Uri uri = Uri.parse("http://webglmol.sfjp.jp/");
				Intent i = new Intent(Intent.ACTION_VIEW,uri);
				startActivity(i); 
			}
		});
		AlertDialog alert = builder.create();
		alert.show();
	}

	public boolean onCreateOptionsMenu(Menu menu){
		super.onCreateOptionsMenu(menu);
		getMenuInflater().inflate(R.menu.mainmenu, menu);

		return true;
	}

	public boolean onPrepareOptionsMenu(Menu menu) {
		if (view != null) {
			menu.findItem(R.id.Sidechain).setChecked(view.showSidechain);
			menu.findItem(R.id.Cell).setChecked(view.showUnitcell);
			menu.findItem(R.id.Solvent).setChecked(view.showSolvents);

			switch (view.proteinMode) {
			case 0:
				menu.findItem(R.id.Ribbon).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.CA_trace).setChecked(true);
			case 2:
				menu.findItem(R.id.Strand).setChecked(true);
				break;
			case 3:
				menu.findItem(R.id.Tube).setChecked(true);
				break;
			}
			
			switch (view.nucleicAcidMode) {
			case 0:
				menu.findItem(R.id.baseLine).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.basePolygon).setChecked(true);
				break;
			}

			switch (view.hetatmMode) {
			case 0:
				menu.findItem(R.id.Sphere).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.Stick).setChecked(true);
				break;
			case 2:
				menu.findItem(R.id.Line).setChecked(true);;
			}

			switch (view.symmetryMode) {
			case 0:
				menu.findItem(R.id.Monomer).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.Biomt).setChecked(true);
				break;
			case 2:
				menu.findItem(R.id.Packing).setChecked(true);
			}

			switch (view.colorMode) {
			case 0:
				menu.findItem(R.id.Chainbow).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.Chain).setChecked(true);
				break;
			case 2:
				menu.findItem(R.id.Structure).setChecked(true);
			case 3:
				menu.findItem(R.id.Polarity).setChecked(true);
				break;
			case 4:
				menu.findItem(R.id.BFactor).setChecked(true);
				break;
			}

		}

		return super.onPrepareOptionsMenu(menu);
	}

	public boolean onOptionsItemSelected(MenuItem item) {
		boolean ret = true;
		switch (item.getItemId()) {
		default:
			ret = super.onOptionsItemSelected(item);
			break;
		case R.id.move:
			mode = 0;
			break;
		case R.id.zoom:
			mode = 1;
			break;
		case R.id.rotate:
			mode = 2;
			break;
		case R.id.open:
			open();
			break;
		case R.id.downloadPDB:
			download();
			break;
		case R.id.preferences:
			preferences();
			break;
		case R.id.about:
			about();
			break;
			
		case R.id.CA_trace:
			view.proteinMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Ribbon:
			view.proteinMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Strand:
			view.proteinMode = 2;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Tube:
			view.proteinMode = 3;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.baseLine:
			view.nucleicAcidMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.basePolygon:
			view.nucleicAcidMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Monomer:
			view.symmetryMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Biomt:
			view.symmetryMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Packing:
			view.symmetryMode = 2; 
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Chainbow:
			view.colorMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Chain:
			view.colorMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Structure:
			view.colorMode = 2;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Polarity:
			view.colorMode = 3;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.BFactor:
			view.colorMode = 4;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Sphere:
			view.hetatmMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Stick:
			view.hetatmMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Line:
			view.hetatmMode = 2;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Sidechain:
			item.setChecked(!item.isChecked());
			view.showSidechain = !view.showSidechain;
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Cell:
			item.setChecked(!item.isChecked());
			view.showUnitcell = !view.showUnitcell;
			view.prepareScene(false);
			glSV.requestRender();
			break;
			
		case R.id.Solvent:
			item.setChecked(!item.isChecked());
			view.showSolvents = !view.showSolvents;
			view.prepareScene(false);
			glSV.requestRender();
			break;
		}
		return ret;
	}
//
//	public boolean onTouchEvent(MotionEvent e) {
//		float x = e.getX();
//		float y = e.getY();
//		int pointerCount = 1;
//		float distance = -1; 
//
//		switch (e.getAction()) {
//		case MotionEvent.ACTION_DOWN:
//			//        	Log.d("event", "down");
//			isDragging = true;
//			view.isMoving = true;
//			startX = x;
//			startY = y;
//
//			currentX = view.objX;
//			currentY = view.objY;
//			currentZ = view.objZ;
//			currentCameraZ = view.cameraZ;
//			currentQ = view.rotationQ;
//			startDistance = distance;
//
//			break;
//		case MotionEvent.ACTION_UP:
//			view.isMoving = false;
//			isDragging = false;
//			glSV.requestRender();
//			break; 
//		case MotionEvent.ACTION_MOVE:
//			if (isDragging) {
//				if (pointerCount > 1 || mode == 0) { // translation
//					Vector3 vec = new Vector3(- (startX - x) * 0.15f, (startY - y) * 0.15f, 0);
//					Vector3 translation = view.rotationQ.rotateVector(vec);
//					view.objX = currentX + translation.x;
//					view.objY = currentY + translation.y;
//					view.objZ = currentZ + translation.z;
//				}
//				if (mode == 1) { // zoom
//					view.cameraZ = currentCameraZ + (startY - y) * 0.5f;
//				}
//				if (pointerCount > 1 && startDistance > 50) {
//					//				view.cameraZ = currentCameraZ * distance / startDistance;
//				}
//				if (mode == 2) { // rotate
//					float dx = (x - startX) / (float)view.width, dy = (y - startY) / (float)view.height;
//					float r = (float)Math.sqrt(dx * dx + dy * dy);
//					if (r == 0) return true;
//
//					float rs = (float)Math.sin(r * Math.PI) / r;
//					Quaternion dq = new Quaternion(rs * dy, rs * dx, 0,  (float)Math.cos(r * Math.PI)); 
//					view.rotationQ = Quaternion.multiply(dq, currentQ);
//				}
//				glSV.requestRender();
//			}
//			break;
//		}
//
//		return true;
//	}
	
	public boolean onTouchEvent(MotionEvent e) {
		float x = e.getX();
		float y = e.getY();
		int pointerCount = 1;
		float distance = -1; 
		if (multiTouchEnabled) {
			pointerCount = MultitouchWrapper.getPointerCount(e);
		}
		if (pointerCount > 1) {
			float x1 = MultitouchWrapper.getX(e, 0);
			float x2 = MultitouchWrapper.getX(e, 1);
			float y1 = MultitouchWrapper.getY(e, 0);
			float y2 = MultitouchWrapper.getY(e, 1);
			x = (x1 + x2) / 2; y = (y1 + y2) / 2;
			distance = (float)Math.sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));
			if (prevPointerCount <= 1) {
				startDistance = distance;
			}
			Log.d("ESmol", "multitouched dist = " + distance);
		} else if (prevPointerCount > 1) { // 2 to 1
			startX = x;
			startY = y;
		}

		switch (e.getAction()) {
		case MotionEvent.ACTION_DOWN: 
//			Log.d("event", "down");
			isDragging = true;
			view.isMoving = true;
			startX = x;
			startY = y;

			currentX = view.objX;
			currentY = view.objY;
			currentZ = view.objZ;
			currentCameraZ = view.cameraZ;
			currentQ = view.rotationQ;
			startDistance = -1;
			break;
//		case MotionEvent.ACTION_POINTER_DOWN: //second finger down FIXME: avoid hard-coding
//			Log.d("event", "pointer down");
//			startDistance = distance;
//			break;
//		case MotionEvent.ACTION_POINTER_UP: // second finger up
//			Log.d("event", "pointer up");
//			break;
		case MotionEvent.ACTION_UP:
//			Log.d("event", "up");
			view.isMoving = false;
			isDragging = false;
			glSV.requestRender();
			break; 
		case MotionEvent.ACTION_MOVE:
//			Log.d("event", "move");
			if (isDragging) {
				if (pointerCount > 1) { 
					if (startDistance > 150) {
						if (distance > 100) {
							view.cameraZ = currentCameraZ * startDistance / distance;
							Log.d("ESmol", "distance = " + distance + " start distance = " + startDistance + " CameraZ = " + view.cameraZ);
						}
					} else {
						float scaleFactor = 0.13f;
						if (view.cameraZ > -150) scaleFactor = 0.035f; // FIXME: improve
						if (view.cameraZ > -50) scaleFactor = 0.02f;
						Vector3 vec = new Vector3(- (startX - x) * scaleFactor, (startY - y) * scaleFactor, 0);
						Vector3 translation = view.rotationQ.rotateVector(vec);
						view.objX = currentX + translation.x;
						view.objY = currentY + translation.y;
						view.objZ = currentZ + translation.z;
					}
					glSV.requestRender();
				} else {
					if (mode == 0) { // translation
						Log.d("ESmol", "cameraZ: " + view.cameraZ);
						float scaleFactor = 0.13f;
						if (view.cameraZ > -150) scaleFactor = 0.035f; // FIXME: improve
						if (view.cameraZ > -50) scaleFactor = 0.02f;
						Vector3 vec = new Vector3(- (startX - x) * scaleFactor, (startY - y) * scaleFactor, 0);
						Vector3 translation = view.rotationQ.rotateVector(vec);
						view.objX = currentX + translation.x;
						view.objY = currentY + translation.y;
						view.objZ = currentZ + translation.z;
					} else if (mode == 1) { // zoom
						view.cameraZ = currentCameraZ + (startY - y) * 0.5f;
					} else	if (mode == 2) { // rotate
						float dx = (x - startX) / (float)view.width, dy = (y - startY) / (float)view.height;
						float r = (float)Math.sqrt(dx * dx + dy * dy);
						if (r == 0) return true;

						float rs = (float)Math.sin(r * Math.PI) / r;
						Quaternion dq = new Quaternion(rs * dy, rs * dx, 0,  (float)Math.cos(r * Math.PI)); 
						view.rotationQ = Quaternion.multiply(dq, currentQ);
					}
					glSV.requestRender();
				}
			}
			break;
		}

		prevPointerCount = pointerCount;
		return true;
	}
}