/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2006,2007 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.web.filter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang.LocaleUtils;
import org.apache.struts.Globals;

/**
 * LocaleFilter
 *
 * @author Yomei Komiya
 *
 * $Id: LocaleFilter.java 6 2007-06-27 13:49:06Z whitestar $
 */
public class LocaleFilter implements Filter {

	/**
	 * Locale Query Parameter
	 * injected by Spring
	 */
	private String localeParameter = "lang";

	/**
	 * Acceptable Locale Strings (e.g. "en_UK", "ja", "ja_JP", "zh_CN", "zh_TW" ...)
	 * injected by Spring
	 */
	private String[] acceptableLocaleStrings = new String[] {};
	
	/**
	 * Acceptable Locales
	 */
	private List acceptableLocales = null;
	
	/**
	 * Default Locale String
	 * injected by Spring
	 */
	private String defaultLocaleString = null;
	
	/**
	 * Default Locale
	 */
	private Locale defaultLocale = null;
	
	/**
	 * Default ContentType
	 */
	private String defaultContentType = "text/html;charset=UTF-8";

	
	protected void initLocales() {
		this.acceptableLocales = new ArrayList();
		for (int i = 0; i < this.acceptableLocaleStrings.length; i++) {
			this.acceptableLocales.add(
					LocaleUtils.toLocale(this.acceptableLocaleStrings[i]));
		}
		
		this.defaultLocale = LocaleUtils.toLocale(this.defaultLocaleString);
	}
	
	
	protected Locale resolveAvailableLocale(String userLocaleString) {
		Locale userLocale = null;
		try {
			userLocale = LocaleUtils.toLocale(userLocaleString);
			return this.resolveAvailableLocale(userLocale);
		}
		catch (Exception e) {
			return this.defaultLocale;
		}
	}
	
	
	protected Locale resolveAvailableLocale(Locale userLocale) {
		if (userLocale != null) {
			String userlanguage = userLocale.getLanguage();
			String userCountry = userLocale.getCountry();
			List languageMatchedLocales = new ArrayList();
			
			for (Iterator ite = this.acceptableLocales.iterator(); ite.hasNext();) {
				Locale acceptableLocale = (Locale)ite.next();
				if (acceptableLocale.getLanguage().equals(userlanguage)) {
					if (acceptableLocale.getCountry().equals(userCountry)) {
						// both language and country matched.
						return acceptableLocale;
					}
					else {
						languageMatchedLocales.add(acceptableLocale);
					}
				}
			}
			
			if (!languageMatchedLocales.isEmpty()) {
				// language matched (1st) locale
				return (Locale)languageMatchedLocales.get(0);
			}
		}
		
		return this.defaultLocale;
	}
	
	
	/*
	 * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
	 */
	public void init(FilterConfig config) throws ServletException {
		// do nothing.
	}
	

	/*
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	public void doFilter(
					ServletRequest request,
					ServletResponse response,
					FilterChain chain)
		throws IOException, ServletException 
	{
		if (this.acceptableLocales == null
			|| this.defaultLocale == null) {
			this.initLocales();
		}
		
		// Locale from request
		HttpSession session = ((HttpServletRequest)request).getSession();
		Locale headerLocale = null;
		
		Locale appLocale = null;
		// 1. request parameter
		if (request.getParameterMap().containsKey(this.localeParameter)) {
			appLocale = this.resolveAvailableLocale(request.getParameter(this.localeParameter));
			session.setAttribute(this.localeParameter, appLocale);
		}
		// 2. session attribute
		else if (session.getAttribute(this.localeParameter) != null) {
			appLocale = (Locale)session.getAttribute(this.localeParameter);
			// no check acceptable languages.
		}
		// 3. request header
		else if ((headerLocale = request.getLocale()) != null) {
			appLocale = this.resolveAvailableLocale(headerLocale);
		}
		// 4. default
		else {
			appLocale = this.defaultLocale;
		}

		response.setLocale(appLocale);
		response.setContentType(this.defaultContentType);
		session.setAttribute(Globals.LOCALE_KEY, appLocale);
		
		chain.doFilter(request, response);
	}

	
	/*
	 * @see javax.servlet.Filter#destroy()
	 */
	public void destroy() {
		// do nothing.
	}

	
	public void setLocaleParameter(String localeParameter) {
		this.localeParameter = localeParameter;
	}


	/**
	 * @param langParameter
	 * @deprecated use setLocaleParameter()
	 */
	public final void setLangParameter(String langParameter) {
		this.localeParameter = langParameter;
	}


	public void setAcceptableLocales(String[] acceptableLocaleStrings) {
		this.acceptableLocaleStrings = acceptableLocaleStrings;
	}


	/**
	 * @param acceptableLanguages
	 * @deprecated use setAcceptableLocales()
	 */
	public final void setAcceptableLanguages(String[] acceptableLanguages) {
		this.acceptableLocaleStrings = acceptableLanguages;
	}


	public void setDefaultLocale(String defaultLocaleString) {
		this.defaultLocaleString = defaultLocaleString;
	}


	/**
	 * @param defaultLanguage
	 * @deprecated use setDefaultLocale()
	 */
	public final void setDefaultLanguage(String defaultLanguage) {
		this.defaultLocaleString = defaultLanguage;
	}


	public final void setDefaultContentType(String defaultContentType) {
		this.defaultContentType = defaultContentType;
	}
	
}
