/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.util;

/**
 * Paginator
 *
 * @author Yomei Komiya
 *
 * $Id: Paginator.java 34 2009-05-06 04:25:19Z whitestar $
 */
public class Paginator {

	/**
	 * Action name (URL id, URL path, Struts action name and so on.)
	 */
	private String action = null;

	/**
	 * Display record count per page
	 */
	private int pageRecordCount = 100;

	private int visiblePageIndexMargin = 10;

	/**
	 * Current page index (1 start)
	 */
	private int currentPageIndex = 1;

	private int recordTotal = 0;


	public String getAction() {
		return action;
	}


	public void setAction(String action) {
		this.action = action;
	}


	/**
	 * Gets the start record offset (0 start) of the current page index.
	 *
	 * @return offset
	 */
	public int getRecordOffset() {
		return getRecordOffset(currentPageIndex);
	}


	/**
	 * Gets the start record offset of the given page index.
	 *
	 * @param pageIndex 1 start
	 * @return
	 */
	public int getRecordOffset(int pageIndex) {
		return (pageIndex - 1) * pageRecordCount;
	}


	/**
	 * Sets the start record offset (0 start) of the current page.
	 *
	 * @param recordOffset
	 */
	public void setRecordOffset(int recordOffset) {
		setRecordOffset(recordOffset, getPageRecordCount());
	}


	/**
	 * Sets the start record offset (0 start) of the current page.
	 *
	 * @param recordOffset
	 * @param pageRecordCount
	 */
	public void setRecordOffset(int recordOffset, int pageRecordCount) {
		setPageRecordCount(pageRecordCount);
		setCurrentPageIndex((recordOffset / pageRecordCount) + 1);
	}


	public int getFirstRecordIndexOfPage() {
		return getFirstRecordIndexOfPage(currentPageIndex);
	}


	public int getFirstRecordIndexOfPage(int pageIndex) {
		return getRecordOffset(pageIndex) + 1;
	}


	public int getLastRecordIndexOfPage() {
		return getLastRecordIndexOfPage(currentPageIndex);
	}


	public int getLastRecordIndexOfPage(int pageIndex) {
		int lastIndex = getRecordOffset(pageIndex) + pageRecordCount;
		if (recordTotal < lastIndex) {
			lastIndex = recordTotal;
		}

		return lastIndex;
	}


	public int getPageTotal() {
		if (recordTotal == 0) {
			return 0;
		}
		else {
			int quotient = recordTotal / pageRecordCount;
			int residue = recordTotal % pageRecordCount;
			return (residue > 0 ? quotient + 1 : quotient);
		}
	}


	public int getPageRecordCount() {
		return pageRecordCount;
	}


	public void setPageRecordCount(int recordCount) {
		this.pageRecordCount = recordCount;
	}


	/**
	 * Alias of the getPageRecordCount() method
	 *
	 * @return record count
	 */
	public int getPerPage() {
		return this.getPageRecordCount();
	}


	/**
	 * Alias of the setPageRecordCount() method
	 *
	 * @param recordCount
	 */
	public void setPerPage(int recordCount) {
		this.setPageRecordCount(recordCount);
	}


	public int getPreviousPageIndex() {
		return (1 < currentPageIndex ? currentPageIndex - 1 : currentPageIndex);
	}


	public int getRecordOffsetOfPreviousPage() {
		return getRecordOffset(getPreviousPageIndex());
	}


	public int getCurrentPageIndex() {
		return currentPageIndex;
	}


	public void setCurrentPageIndex(int pageIndex) {
		this.currentPageIndex = pageIndex;
	}


	public int getNextPageIndex() {
		return (currentPageIndex < getPageTotal() ?
				currentPageIndex + 1 : currentPageIndex);
	}


	public int getRecordOffsetOfNextPage() {
		return getRecordOffset(getNextPageIndex());
	}


	public boolean hasPreviousPage() {
		return (1 < currentPageIndex ? true : false);
	}


	public boolean hasNextPage() {
		return (currentPageIndex < getPageTotal() ? true : false);
	}


	public boolean isVisiblePageIndex(int pageIndex) {
		if (currentPageIndex - visiblePageIndexMargin <= pageIndex
			&& pageIndex < currentPageIndex + visiblePageIndexMargin) {
			return true;
		}
		else {
			return false;
		}
	}


	public int getRecordTotal() {
		return recordTotal;
	}


	public void setRecordTotal(int recordTotal) {
		this.recordTotal = recordTotal;
	}


	public int getVisiblePageIndexMargin() {
		return visiblePageIndexMargin;
	}


	public void setVisiblePageIndexMargin(int visiblePageIndexMargin) {
		this.visiblePageIndexMargin = visiblePageIndexMargin;
	}

}
