/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2007 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.springframework.util;

import javax.servlet.ServletContext;

import org.apache.commons.collections.BidiMap;
import org.apache.commons.collections.bidimap.DualHashBidiMap;
import org.apache.struts.action.Action;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.springframework.web.struts.ContextLoaderPlugIn;

/**
 * SpringUtils
 *
 * @author Yomei Komiya
 * @since 0.9.2 
 *
 * @version $Id$
 */
public class SpringUtils {
	
	/**
	 * Get WebApplicationContext object
	 * 
	 * @param action Action object
	 * @return WebApplicationContext object
	 * @throws IllegalStateException
	 */
	static public WebApplicationContext getWebApplicationContext(Action action) 
		throws IllegalStateException {

		return getWebApplicationContext(action);
	}
	
	
	/**
	 * Get WebApplicationContext object
	 * 
	 * @param servletContext ServletContext object
	 * @return WebApplicationContext object
	 * @throws IllegalStateException
	 */
	static public WebApplicationContext getWebApplicationContext(
			ServletContext servletContext)
		throws IllegalStateException {

		WebApplicationContext wac = (WebApplicationContext)
		servletContext.getAttribute(ContextLoaderPlugIn.SERVLET_CONTEXT_PREFIX);
		if (wac == null) {
			wac = WebApplicationContextUtils
				.getRequiredWebApplicationContext(servletContext);
		}
		
		return wac;
	}
	
	
	/**
	 * Get SpringContextHolder object
	 * 
	 * @param action Action object
	 * @return SpringContextHolder object
	 * @throws IllegalStateException
	 */
	static public SpringContextHolder getSpringContextHolder(Action action)
		throws IllegalStateException {

		return SpringUtils.getSpringContextHolder(
				action.getServlet().getServletContext());
	}
	
	
	/**
	 * Get SpringContextHolder object
	 * 
	 * @param servletContext ServletContext object
	 * @return SpringContextHolder object
	 * @throws IllegalStateException
	 */
	static public SpringContextHolder getSpringContextHolder(
			ServletContext servletContext) throws IllegalStateException {

		ConfigurableApplicationContext applicationContext
			= (ConfigurableApplicationContext)SpringUtils.getWebApplicationContext(
					servletContext);
		SpringContextHolder contextHolder
			= new SpringContextHolder(applicationContext);
		
		return contextHolder;
	}

	
	/**
	 * Get all SpringBeanHolders
	 * 
	 * @param applicationContext ConfigurableApplicationContext object
	 * @return SpringContextHolder array
	 */
	static public SpringBeanHolder[] getSpringBeanHolders(
			ConfigurableApplicationContext applicationContext) {
		if (applicationContext != null) {
	
			String[] beanNames = applicationContext.getBeanDefinitionNames();
			return SpringUtils.getSpringBeanHoldersForNames(
					applicationContext, beanNames);
		}
		else {
			throw new IllegalArgumentException(
					"applicationContext is null.");
		}
	}

	
	/**
	 * Get all SpringBeanHolders by same type
	 * 
	 * @param applicationContext ConfigurableApplicationContext object
	 * @param type Class type
	 * @return SpringBeanHolder array
	 */
	static public SpringBeanHolder[] getSpringBeanHoldersForType(
			ConfigurableApplicationContext applicationContext,
			Class type) {
		if (applicationContext != null) {
			
			String[] beanNames = applicationContext.getBeanNamesForType(type);
			return SpringUtils.getSpringBeanHoldersForNames(
					applicationContext, beanNames);
		}
		else {
			throw new IllegalArgumentException(
					"applicationContext is null.");
		}
	}
	
	
	/**
	 * Get SpringBeanHolders by bean names
	 * 
	 * @param applicationContext ConfigurableApplicationContext object
	 * @param beanNames Bean name array
	 * @return SpringBeanHolder array
	 */
	static public SpringBeanHolder[] getSpringBeanHoldersForNames(
			ConfigurableApplicationContext applicationContext,
			String[] beanNames) {
		if (applicationContext != null) {
			ConfigurableListableBeanFactory beanFactory
				= applicationContext.getBeanFactory();
	
			SpringBeanHolder[] beanHolders
				= new SpringBeanHolder[beanNames.length];
			for (int i = 0; i < beanNames.length; i++) {
				beanHolders[i] = new SpringBeanHolder(beanFactory, beanNames[i]);
			}
		
			return beanHolders;
		}
		else {
			throw new IllegalArgumentException(
					"applicationContext is null.");
		}
	}
	
	
	/**
	 * Get SpringBeanHolder by bean name
	 * 
	 * @param applicationContext ConfigurableApplicationContext object
	 * @param beanName Bean name
	 * @return SpringBeanHolder object
	 * @throws NoSuchBeanDefinitionException
	 */
	static public SpringBeanHolder getSpringBeanHolder(
			ConfigurableApplicationContext applicationContext,
			String beanName)
		throws NoSuchBeanDefinitionException {

		if (applicationContext != null) {
			ConfigurableListableBeanFactory beanFactory
				= applicationContext.getBeanFactory();
	
			return new SpringBeanHolder(beanFactory, beanName);
		}
		else {
			throw new IllegalArgumentException(
					"applicationContext is null.");
		}
	}

	
	/**
	 * Get bean name <-> singleton object ID bi-directional map
	 * 
	 * @param beanHolders BeanHolder array
	 * @return bi-directional map
	 */
	static public BidiMap getSpringBeanSingletonNameObjectIdMap(
			SpringBeanHolder[] beanHolders) {
		
		BidiMap singletonNameObjectIdMap = new DualHashBidiMap();
		
		for (int i =0; i < beanHolders.length; i++) {
			if (beanHolders[i].isSingleton()) {
				singletonNameObjectIdMap.put(
						beanHolders[i].getBeanName(),
						beanHolders[i].getActualSingletonObjectId());
			}
		}
		
		return singletonNameObjectIdMap;
	}
	
}
