/* vim: set tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * 	http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.sourceforge.webframe.web.filter;

import java.io.IOException;
import java.util.Arrays;
import java.util.Locale;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.struts.Globals;

/**
 * LocaleFilter
 *
 * @author Yomei Komiya
 *
 * $Id$
 */
public class LocaleFilter implements Filter {

	/**
	 * Language Query Parameter
	 * injected by Spring
	 */
	private String langParameter = "lang";

	/**
	 * Acceptable Languages
	 * injected by Spring
	 */
	private String[] acceptableLanguages = new String[] {"en", "ja"};
	
	/**
	 * Default Language
	 * injected by Spring
	 */
	private String defaultLanguage = "en";
	
	private String defaultContentType = "text/html;charset=UTF-8";

	
	public final void setLangParameter(String langParameter) {
		this.langParameter = langParameter;
	}


	public final void setAcceptableLanguages(String[] acceptableLanguages) {
		this.acceptableLanguages = acceptableLanguages;
	}


	public final void setDefaultLanguage(String defaultLanguage) {
		this.defaultLanguage = defaultLanguage;
	}


	public final void setDefaultContentType(String defaultContentType) {
		this.defaultContentType = defaultContentType;
	}


	/*
	 * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
	 */
	public void init(FilterConfig config) throws ServletException {
		// do nothing.
	}

	/*
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	public void doFilter(
					ServletRequest request,
					ServletResponse response,
					FilterChain chain)
		throws IOException, ServletException 
	{
		// Locale from request
		HttpSession session = ((HttpServletRequest)request).getSession();
		Locale headerLocale = request.getLocale();
		
		String appLang = this.defaultLanguage;
		// 1. request parameter
		if (request.getParameterMap().containsKey(this.langParameter)) {
			appLang = request.getParameter(this.langParameter);
			
			if (Arrays.asList(acceptableLanguages).contains(appLang)) {
				session.setAttribute(this.langParameter, appLang);
			}
			else {
				appLang = this.defaultLanguage;
				session.setAttribute(this.langParameter, appLang);
			}
		}
		// 2. session attribute
		else if (session.getAttribute(this.langParameter) != null) {
			appLang = (String)session.getAttribute(this.langParameter);
		}
		// 3. request header
		else if (headerLocale != null) {
			appLang = headerLocale.getLanguage();
			if (!Arrays.asList(acceptableLanguages).contains(appLang)) {
				appLang = this.defaultLanguage;
			}
		}
		// 4. default
		else {
			appLang = this.defaultLanguage;
		}

		Locale appLocale = new Locale(appLang);

		response.setLocale(appLocale);
		response.setContentType(this.defaultContentType);
		session.setAttribute(Globals.LOCALE_KEY, appLocale);
		
		chain.doFilter(request, response);
	}

	/*
	 * @see javax.servlet.Filter#destroy()
	 */
	public void destroy() {
		// do nothing.
	}

}
