/***************************************************************************
                          wvoxstorm.cpp  -  description
                             -------------------
    begin                : 2004ǯ  5  4  22:42:12 JST
    copyright            : (C) 2004 by Tomoaki Murakami
    email                : solarisphere@yahoo.co.jp
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <unistd.h>
#include <sys/wait.h>
#include <signal.h>
#include <stdlib.h>
#include <qfileinfo.h>
#include <qstrlist.h>
#include <qstringlist.h>
#include <kaccel.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <qpopupmenu.h>
#include "wvoxstorm.h"
#include "cvoxstormglobal.h"

/////////////////////////////////////////////////////////////////////////////

WVoxStorm::WVoxStorm(QWidget *WidgetParent, const char *cName)
  : KMainWindow(WidgetParent, cName)
{
  setupActions();
  setupStatusBar();

  readConfigurationFile();

  VoxBase = new WVoxBase(this, "VoxBase");
  connect(VoxBase, SIGNAL(playbackDeviceChanged(const QString&, const QString&)),
          this, SLOT(slotChangePlaybackDevice(const QString&, const QString&)));
  connect(VoxBase, SIGNAL(recordingDeviceChanged(const QString&, const QString&)),
          this, SLOT(slotChangeRecordingDevice(const QString&, const QString&)));
  connect(VoxBase, SIGNAL(trackRemoved(const QString&)),
          this, SLOT(slotRemoveTrack(const QString&)));
  connect(VoxBase, SIGNAL(fileNameChanged(const QString&, const QString&)),
          this, SLOT(slotChangeFileName(const QString&, const QString&)));
  connect(VoxBase, SIGNAL(levelChanged(const QString&, int)),
          this, SLOT(slotChangeLevel(const QString&, int)));
  connect(VoxBase, SIGNAL(panChanged(const QString&, int)),
          this, SLOT(slotChangePan(const QString&, int)));
  connect(VoxBase, SIGNAL(muteToggled(const QString&, bool)),
          this, SLOT(slotToggleMute(const QString&, bool)));
  connect(VoxBase, SIGNAL(recToggled(const QString&, bool)),
          this, SLOT(slotToggleRec(const QString&, bool)));
  connect(VoxBase, SIGNAL(startPositionChanged(const QString&, const QTime&)),
          this, SLOT(slotChangeStartPosition(const QString&, const QTime&)));
  connect(VoxBase, SIGNAL(offsetChanged(const QString&, const QTime&)),
          this, SLOT(slotChangeOffset(const QString&, const QTime&)));
  connect(VoxBase, SIGNAL(lengthChanged(const QString&, const QTime&)),
          this, SLOT(slotChangeLength(const QString&, const QTime&)));

  setCentralWidget(VoxBase);

  connect(VoxBase, SIGNAL(masterPositionChanged(const QTime&)),
          &EcaIAManager, SLOT(slotSetMasterPosition(const QTime&)));
  connect(&EcaIAManager, SIGNAL(masterPositionChanged(const QTime&)),
          VoxBase, SLOT(slotChangePosition(const QTime&)));
  connect(&EcaIAManager, SIGNAL(finished()), this, SLOT(slotFinish()));
  connect(&EcaIAManager, SIGNAL(messageGenerated(const QString&)),
          VoxBase, SLOT(slotWriteConsoleMessage(const QString&)));

  bDirty = false;
  
  slotNew();

  bDirty = true;

  pDocVox = NULL;
}

/////////////////////////////////////////////////////////////////////////////

WVoxStorm::~WVoxStorm()
{
  writeConfigurationFile();

  if ( pDocVox != NULL ) {
    delete pDocVox;
    pDocVox = new QDomDocument;
  }

  delete ToggleActionRec;
  delete ToggleActionInteractive;
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::readConfigurationFile()
{
  KConfig  *Config = KGlobal::config();
  QString   strTmp;
  QStrList  strListTmp;

  // ---->>>> setting for Preferences member
  Config->setGroup("Devices");
  Preferences.strPlayDevice = Config->readEntry("PlaybackDevice", "/dev/dsp");
  Preferences.strRecDevice  = Config->readEntry("RecDevice", "/dev/dsp");

  Config->setGroup("File");
  strTmp = Config->readEntry("FileType", "WAV");
  if ( strTmp == "WAV" ) {
    Preferences.FileType = CVoxStormGlobal::ecaWAV;
  } else {
    Preferences.FileType = CVoxStormGlobal::ecaMP3;
  }

  Config->setGroup("Project");
  Preferences.strProjectPath = Config->readEntry("ProjectPath", "~/voxstorm");

  Config->setGroup("Sound Spec");
  Preferences.uiSampleRate = Config->readNumEntry("SamplingRate", 44100);
  Preferences.uiBuffer     = Config->readNumEntry("BufferSize", 256);

  Config->setGroup("Track");
  Config->readListEntry("DefaultTracks", strListTmp, ',');
  Preferences.strlDefaultTracks.clear();
  Preferences.strlDefaultTracks = QStringList::fromStrList(strListTmp);

  Config->setGroup("Generic");
  Preferences.iFrameSeconds = Config->readNumEntry("FrameSeconds", 5);
  // <<<<---- setting for Preferences member

  // ---->>>> setting for Mainwindow
  Config->setGroup("Mainwindow");
  setGeometry(Config->readNumEntry("Left", 0),
              Config->readNumEntry("Top", 0),
              1, 1);  // Width and Height will be read after soon
  applyMainWindowSettings(Config);
  restoreWindowSize(Config);
  ToggleActionInteractive->setChecked(true);
  setWidgetsByIAState();
  // <<<<---- setting for Mainwindow
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::writeConfigurationFile()
{
  KConfig  *Config = KGlobal::config();

  // ---->>>> setting for Mainwindow
  Config->setGroup("Devices");
  Config->writeEntry("PlayDevice", Preferences.strPlayDevice);
  Config->writeEntry("RecDevice", Preferences.strRecDevice);

  Config->setGroup("File");
  if ( Preferences.FileType == CVoxStormGlobal::ecaWAV ) {
    Config->writeEntry("FileType", "WAV");
  } else {
    Config->writeEntry("FileType", "MP3");
  }

  Config->setGroup("Project");
  Config->writeEntry("ProjectPath", Preferences.strProjectPath);

  Config->setGroup("Sound Spec");
  Config->writeEntry("SamplingRate", Preferences.uiSampleRate);
  Config->writeEntry("BufferSize", Preferences.uiBuffer);

  QStrList  strListTmp;
  int       i, nTrack;
  Config->setGroup("Track");
  nTrack = Preferences.strlDefaultTracks.count();
  for ( i = 0; i < nTrack; i++ ) {
    strListTmp.append(Preferences.strlDefaultTracks[i]);
  }
  Config->writeEntry("DefaultTracks", strListTmp, ',');

  Config->setGroup("Generic");
  Config->writeEntry("FrameSeconds", Preferences.iFrameSeconds);
  // <<<<---- setting for Mainwindow

  // ---->>>> setting for Mainwindow
  Config->setGroup("Mainwindow");
  Config->writeEntry("Left", geometry().x());
  Config->writeEntry("Top", geometry().y());
  saveMainWindowSettings(Config);
  saveWindowSize(Config);
  Config->writeEntry("Interactive", ToggleActionInteractive->isChecked());
  // <<<<---- setting for Mainwindow

  Config->sync();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::setupActions()
{
  QPopupMenu* PopupMenu = new QPopupMenu;
  KAction* Action;

  Action = KStdAction::openNew(this, SLOT(slotNew()), actionCollection());
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = KStdAction::open(this, SLOT(slotOpen()), actionCollection());
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = KStdAction::save(this, SLOT(slotSave()), actionCollection());
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = KStdAction::saveAs(this, SLOT(slotSaveAs()), actionCollection());
  Action->plug(PopupMenu);
      
  Action = KStdAction::quit(this, SLOT(close()), actionCollection());
  Action->plug(PopupMenu);

  menuBar()->insertItem(i18n("&File"), PopupMenu);
  toolBar()->insertLineSeparator();

  PopupMenu = new QPopupMenu;

  Action = new KAction(i18n("&Play"),
                       "voxstorm_play",
                       KShortcut(CTRL+Key_P),
                       this,
                       SLOT(slotPlay()),
                       actionCollection(),
                       "Play");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  ToggleActionRec = new KToggleAction(
                       i18n("&Record"),
                       "voxstorm_record",
                       KShortcut(CTRL+Key_R),
                       this,
                       SLOT(slotRecord()),
                       this,
                       "Record");
  ToggleActionRec->plug(PopupMenu);
  ToggleActionRec->plug(toolBar());

  Action = new KAction(i18n("P&ause"),
                       "voxstorm_pause",
                       KShortcut(CTRL+Key_A),
                       this,
                       SLOT(slotPause()),
                       actionCollection(),
                       "Pause");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = new KAction(i18n("&Stop"),
                       "voxstorm_stop",
                       KShortcut(CTRL+Key_D),
                       this,
                       SLOT(slotStop()),
                       actionCollection(),
                       "Stop");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = new KAction(i18n("&Jump to Start"),
                       "voxstorm_start",
                       0,
                       this,
                       SLOT(slotJumpToStart()),
                       actionCollection(),
                       "Jump To Start");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = new KAction(i18n("Re&wind"),
                       "voxstorm_rewind",
                       0,
                       this,
                       SLOT(slotRewind()),
                       actionCollection(),
                       "Rewind");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  Action = new KAction(i18n("&Forward"),
                       "voxstorm_forward",
                       0,
                       this,
                       SLOT(slotForward()),
                       actionCollection(),
                       "Forward");
  Action->plug(PopupMenu);
  Action->plug(toolBar());

  menuBar()->insertItem(i18n("&Operation"), PopupMenu);

  PopupMenu = new QPopupMenu;

  Action = new KAction(i18n("&Add Track..."),
                       NULL,
                       0,
                       this,
                       SLOT(slotAddTrack()),
                       actionCollection(),
                       "Add Track");
  Action->plug(PopupMenu);

  Action = new KAction(i18n("&Track Order"),
                       NULL,
                       0,
                       this,
                       SLOT(slotTrackOrder()),
                       actionCollection(),
                       "Track Order");
  Action->plug(PopupMenu);

  menuBar()->insertItem(i18n("&Track"), PopupMenu);

  PopupMenu = new QPopupMenu;

  //  toolBar()->insertLineSeparator();

  ToggleActionInteractive = new KToggleAction(
                       i18n("&Interactive Mode"),
                       "voxstorm_interactive",
                       0,
                       this,
                       SLOT(slotInteractive()),
                       this,
                       "Interactive");
  ToggleActionInteractive->plug(PopupMenu);
  ToggleActionInteractive->plug(toolBar());
  ToggleActionInteractive->setEnabled(false);

  Action = new KAction(i18n("&Configure VoxStorm..."),
                       NULL,
                       "configure",
                       this,
                       SLOT(slotPreferences()),
                       actionCollection(),
                       "Configure");
  Action->plug(PopupMenu);

  menuBar()->insertItem(i18n("&Settings"), PopupMenu);

  QString strHelp =
            i18n("VoxStorm 0.1\n\n"
                 "(C) 2003-2004 Tomoaki Murakami\n"
                 "solarisphere@yahoo.co.jp\n"
                 "Simple MTR using Ecasound on KDE\n");

  PopupMenu = (QPopupMenu *)helpMenu(strHelp);

  menuBar()->insertItem(i18n("&Help"), PopupMenu);

  menuBar()->show();
  toolBar()->show();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::setupStatusBar()
{
  statusBar()->insertItem("", 0, 1, false);
  statusBar()->setSizeGripEnabled(true);
  statusBar()->show();
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxStorm::queryClose()
{
  if ( maybeSave() ) {
    QApplication::setOverrideCursor(waitCursor);
    slotStop();
    if ( ToggleActionInteractive->isChecked() ) {
      EcaIAManager.slotReleaseCurrentChainsetup();
    }
    QApplication::restoreOverrideCursor();

    return true;
  }

  return false;
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxStorm::maybeSave()
{
  if ( isChanged() ) {
    int iResult;

    iResult = KMessageBox::questionYesNoCancel(
                this,
                i18n("The session has been modified.\n"
                     "Do you want to save your change?"));

    switch ( iResult ) {
    case KMessageBox::Yes:
      return slotSave();
    case KMessageBox::No:
      break;
    case KMessageBox::Cancel:
      return false;
    }
  }

  return true;
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxStorm::sessionName() const
{
  if ( strSessionFile == "" ) {
    return kapp->caption();
  }

  QFileInfo FileInfo(strSessionFile);
  return FileInfo.fileName();
}

/////////////////////////////////////////////////////////////////////////////

bool 
WVoxStorm::isChanged() const
{
  return bDirty;
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxStorm::isPlaying()
{
  return EcaIAManager.isPlaying();
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxStorm::createNewUntitledName() const
{
  static unsigned int unID;
  QString strNewUntitledName;

  unID++;
  strNewUntitledName.sprintf("%s%d", "Untitled", unID);

  return strNewUntitledName;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::loadFile(const QString& strFileName)
{
  QDomDocument doc;
  QFile File(strFileName);
  if ( !File.open(IO_ReadOnly) ) {
    return;
  }
  if ( !doc.setContent(&File) ) {
    File.close();
    return;
  }
  File.close();

  VoxBase->clearTracks();
  strSessionFile = strFileName;
  QFileInfo FileInfo(strFileName);
  setCaption(FileInfo.fileName());

  QDomElement eleRoot = doc.documentElement();
  VoxBase->addTracks(eleRoot.firstChild());

  enterIAMode(ToggleActionInteractive->isChecked());
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::saveFile(const QString& strFileName)
{
  if ( pDocVox != NULL ) {
    delete pDocVox;
  }
  pDocVox = new QDomDocument;

  // root node
  {
    QDomNode nodRoot(stateInDOM());
    pDocVox->appendChild(nodRoot);
  }

  // processing instruction
  {
    QDomNode nodPI = pDocVox->createProcessingInstruction(
                       "xml", 
                       "version=\"1.0\"");
    pDocVox->insertBefore(nodPI, pDocVox->firstChild());
  }

  QFile File(strFileName);
  if ( !File.open(IO_WriteOnly) ) {
    delete pDocVox;
    pDocVox = NULL;
    return;
  }
  QTextStream out(&File);
  pDocVox->save(out, 4);
  delete pDocVox;
  pDocVox = NULL;

  bDirty = false;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::enterIAMode(bool bEnter)
{
  if ( bEnter ) {
    QString strChainsetup = sessionName();
    EcaIAManager.slotCreateNewChainsetup(strChainsetup);
  } else {
    // clear chainsetup
    EcaIAManager.slotReleaseCurrentChainsetup();
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::setWidgetsByIAState()
{
  if ( ToggleActionInteractive->isChecked() ) {
  } else {
  }
}

/////////////////////////////////////////////////////////////////////////////

QDomNode 
WVoxStorm::stateInDOM()
{
  QDomElement eleSession = pDocVox->createElement("Session");

  // child node
  {
    QDomNode nodChild = VoxBase->stateInDOM(pDocVox);
    eleSession.appendChild(nodChild);
  }

  return eleSession;
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::slotNew()
{
  if ( maybeSave() ) {
    setCaption(createNewUntitledName(), false);
    strSessionFile = "";

    VoxBase->clearTracks();

    int     i, nTrack;
    QString strNewTrack;
    nTrack = Preferences.strlDefaultTracks.count();
    for ( i = 0; i < nTrack; i++ ) {
      strNewTrack = Preferences.strlDefaultTracks[i];
      if ( !VoxBase->queryExistingTrackName(strNewTrack) ) {
	VoxBase->addTrack(Preferences.strlDefaultTracks[i],
			  Preferences.strPlayDevice, 
			  Preferences.strRecDevice);
      }
    }

    if ( ToggleActionInteractive->isChecked() ) {
      enterIAMode(true);
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotOpen()
{
  if ( maybeSave() ) {
    KURL url = KFileDialog::getOpenURL(
                 QString::null, 
                 "*.voxs|VoxStorm session files\n*|All files");
    if ( !url.isEmpty() ) {
      if ( url.isLocalFile() ) {
	loadFile(url.path());	
      } else {
	KMessageBox::sorry(this, i18n("VoxStorm cannot handle remote files yet."));
      }
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

bool 
WVoxStorm::slotSave()
{
  if ( strSessionFile == "" ) {
    return slotSaveAs();
  } else {
    saveFile(strSessionFile);

    return true;
  }
}

/////////////////////////////////////////////////////////////////////////////

bool 
WVoxStorm::slotSaveAs()
{
  KFileDialog Dialog(QString::null,
                     "*.voxs|VoxStorm session files", 
                     NULL,
                     "Save File", 
                     true);
  Dialog.exec();
  QString strFileName = Dialog.selectedFile();
  if ( strFileName == "" ) {
    return false;
  }

  QFileInfo FileInfo(strFileName);
  // modify extension if needed
  if ( FileInfo.extension(false) != "voxs" ) {
    FileInfo.setFile(strFileName + ".voxs");
  }

  if ( FileInfo.exists() ) {
    if ( KMessageBox::warningYesNo(
           this, 
           QString("A file named \"%1\" already exists. "
                "Are you sure you want to overwrite it?").arg(FileInfo.filePath()), 
           "Overwrite File?") != KMessageBox::Yes ) {
      return false;
    }
  }

  saveFile(FileInfo.filePath());
  strSessionFile = FileInfo.filePath();
  setCaption(FileInfo.fileName());

  return true;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotPlay()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    EcaIAManager.slotRemoveAllChains();
    int nCount = VoxBase->getTrackCount();
    int i;
    for ( i = 0; i < nCount; i++ ) {
      EcaIAManager.slotAddChain(VoxBase->getTrackInf(i));
    }

    EcaIAManager.slotPlay(ToggleActionRec->isChecked());
    if ( EcaIAManager.isPlaying() ) { // success playing
      ToggleActionRec->setEnabled(false);
      VoxBase->slotSetEnabledWidgets(WVoxBase::Play);
      VoxBase->slotChangeMessageTab(WVoxBase::Console);
    }
  } else {
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotRecord()
{
  ;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotPause()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    EcaIAManager.slotPause();
    VoxBase->slotSetEnabledWidgets(WVoxBase::Pause);
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotStop()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    EcaIAManager.slotStop();
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotJumpToStart()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    EcaIAManager.slotSetMasterPosition(0);
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotRewind()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {    
    EcaIAManager.slotRewind(Preferences.iFrameSeconds);
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotForward()
{
  QApplication::setOverrideCursor(waitCursor);
  if ( ToggleActionInteractive->isChecked() ) {
    EcaIAManager.slotForward(Preferences.iFrameSeconds);
  }
  QApplication::restoreOverrideCursor();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotPreferences()
{
  WPrefDialog *PrefDialog = new WPrefDialog(this);
  PrefDialog->setFields(Preferences);
  if ( PrefDialog->exec() == QDialog::Accepted ) {
    Preferences = PrefDialog->fields();
  }

  delete PrefDialog;

  writeConfigurationFile();
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotInteractive()
{
  if ( ToggleActionInteractive->isChecked() ) {
    setWidgetsByIAState();
    // enter intaractive mode
    enterIAMode(true);
  } else {
    setWidgetsByIAState();
    // exit interactive mode
    enterIAMode(false);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotAddTrack()
{
  VoxBase->addTrack(Preferences.strPlayDevice, Preferences.strRecDevice);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotTrackOrder()
{
  ;
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::slotChangePlaybackDevice(const QString& strTrackName,
                                    const QString& strPlaybackDevice)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotChangePlaybackDevice(strTrackName, strPlaybackDevice);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeRecordingDevice(const QString& strTrackName, 
                                     const QString& strRecordingDevice)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotChangeRecordingDevice(strTrackName, strRecordingDevice);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotRemoveTrack(const QString& strTrackName)
{
  if ( ToggleActionInteractive->isChecked() ) {
    EcaIAManager.slotRemoveChain(strTrackName);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeFileName(const QString& strTrackName, const QString& strFileName)
{
  if ( ToggleActionInteractive->isChecked() ) {
    if ( !EcaIAManager.isExisting(strTrackName) ) {
      ;
    } else {
      ;
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeLevel(const QString& strTrackName, int nLevel)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotChangeLevel(strTrackName, nLevel);
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxStorm::slotChangePan(const QString& strTrackName, int nPan)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotChangePan(strTrackName, nPan);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotToggleMute(const QString& strTrackName, bool bMute)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotToggleMute(strTrackName, bMute);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotToggleRec(const QString& strTrackName, bool bRec)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotToggleRec(strTrackName, bRec);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeStartPosition(
             const QString& strTrackName,
             const QTime& TimeStartPosition)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotChangeStartPosition(strTrackName, TimeStartPosition);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeOffset(
             const QString& strTrackName,
             const QTime& TimeOffset)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotChangeOffset(strTrackName, TimeOffset);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotChangeLength(
             const QString& strTrackName,
             const QTime& TimeLength)
{
  if ( ToggleActionInteractive->isChecked() ) {
    // NOTE :
    // the track that has name as strTrackName is surely existing
    // so we not check existence of the track

    EcaIAManager.slotChangeLength(strTrackName, TimeLength);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxStorm::slotFinish()
{
  if ( ToggleActionRec->isChecked() ) {
    VoxBase->slotNotifySucceedInRecording();
  }
  VoxBase->slotSetEnabledWidgets(WVoxBase::Stop);
  ToggleActionRec->setEnabled(true);
}

/////////////////////////////////////////////////////////////////////////////

#include "wvoxstorm.moc"
