/***************************************************************************
                          wvoxtrack.cpp  -  description
                             -------------------
    begin                : 2004ǯ  5  4  17:01:12 JST
    copyright            : (C) 2003 by Black Sabbath
    email                : solarisphere@yahoo.co.jp
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qcursor.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <klineeditdlg.h>
#include "wvoxtrack.h"
#include "wnamedialog.h"
#include "wtimedialog.h"
#include "cvoxstormglobal.h"

/////////////////////////////////////////////////////////////////////////////

unsigned int WVoxTrack::Mute(0x1);
unsigned int WVoxTrack::Record(0x2);
unsigned int WVoxTrack::File(0x4);
unsigned int WVoxTrack::Remove(0x8);
unsigned int WVoxTrack::Device(0x10);
unsigned int WVoxTrack::StartPosition(0x20);
unsigned int WVoxTrack::Offset(0x40);
unsigned int WVoxTrack::Length(0x80);

/////////////////////////////////////////////////////////////////////////////

WVoxTrack::WVoxTrack(const QString& strName,
                     const QString& strDeviceIn,
                     const QString& strDeviceOut,
                     QWidget*       WidgetParent,
                     const char*    cName)
  : QHBox(WidgetParent, cName)
{
  strInputDevice = strDeviceIn;
  strOutputDevice = strDeviceOut;
  setupWidgets();
  LabelName->setText(strName);
}

/////////////////////////////////////////////////////////////////////////////

WVoxTrack::WVoxTrack(const CTrackInf& TrackInf,
                     QWidget*         WidgetParent, 
                     const char*      cName)
  : QHBox(WidgetParent, cName)
{
  setupWidgets();
  setStateByTrackInf(TrackInf);
}

/////////////////////////////////////////////////////////////////////////////

WVoxTrack::WVoxTrack(const QDomNode& nodTrack,
                     QWidget*        WidgetParent, 
                     const char*     cName)
  : QHBox(WidgetParent, cName)
{
  setupWidgets();
  setStateByDom(nodTrack);
}

/////////////////////////////////////////////////////////////////////////////

WVoxTrack::~WVoxTrack()
{
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::setupWidgets()
{
  // this has VBoxController
  setupPopupMenu();
  setMinimumWidth(400);
  setFrameStyle(QFrame::WinPanel | QFrame::Plain);

  VBoxController = new QVBox(this);

  // VBoxController has HBoxStandard, HBoxButtonBase
  HBoxStandard = new QHBox(VBoxController);

  // HBoxStandard has LabelName, VBoxValue
  LabelName = new QLabel(HBoxStandard);
  LabelName->setFrameStyle(QFrame::WinPanel | QFrame::Sunken);

  VBoxValue = new QVBox(HBoxStandard);
  VBoxValue->setMaximumWidth(200);
  VBoxValue->setMinimumWidth(200);

  // VBoxValue has LabelLevel, IntNumInputLevel, LabelPan, IntNumInputPan
  LabelLevel = new QLabel("Level (%)", VBoxValue);

  IntNumInputLevel = new KIntNumInput(VBoxValue);
  IntNumInputLevel->setRange(0, 600);
  IntNumInputLevel->setValue(100);  // 100 %

  LabelPan = new QLabel("Pan", VBoxValue);
  
  IntNumInputPan = new KIntNumInput(VBoxValue);
  IntNumInputPan->setRange(0, 100);
  IntNumInputPan->setValue(50);     // just center

  HBoxButtonBase = new QHBox(VBoxController);

  // HBoxButtonBase has FileChooserTrack, WidgetSpacer, ColorButtonMute, ColorButtonRec
  FileChooserTrack = new WFileChooser("", HBoxButtonBase);
  FileChooserTrack->setFileType("*.wav *.mp3|Sound files (*.wav; *.mp3)\n*"
                                "|All files (*)");

  WidgetSpacer = new QHBox(HBoxButtonBase);
  WidgetSpacer->setMinimumWidth(10);
  WidgetSpacer->setMaximumWidth(10);

  // mute button
  PushButtonMute = new QPushButton(HBoxButtonBase);
  PushButtonMute->setText("M");
  PushButtonMute->setFont(QFont("Gothic", 8, QFont::Bold));
  PushButtonMute->setToggleButton(true);
  PushButtonMute->setOn(false);
  PushButtonMute->setMaximumWidth(20);

  // record button
  PushButtonRec = new QPushButton(HBoxButtonBase);
  PushButtonRec->setText("R");
  PushButtonRec->setFont(QFont("Gothic", 8, QFont::Bold));
  PushButtonRec->setToggleButton(true);
  PushButtonRec->setOn(false);
  PushButtonRec->setMaximumWidth(20);

  // setup signals and slots
  connect(FileChooserTrack, SIGNAL(fileNameChanged(const QString&)),
	  this, SIGNAL(fileNameChanged(const QString&)));
  connect(IntNumInputLevel, SIGNAL(valueChanged(int)), this, SIGNAL(levelChanged(int)));
  connect(IntNumInputPan, SIGNAL(valueChanged(int)), this, SIGNAL(panChanged(int)));
  connect(PushButtonMute, SIGNAL(toggled(bool)), this, SIGNAL(muteToggled(bool)));
  connect(PushButtonRec, SIGNAL(toggled(bool)), this, SIGNAL(recToggled(bool)));
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setupPopupMenu()
{
  PopupMenu = new QPopupMenu();

  nMenuIDPlaybackDevice = PopupMenu->insertItem(i18n("Change &Playback Device..."),
                        this, SLOT(slotChangePlaybackDevice()));
  nMenuIDRecordingDevice = PopupMenu->insertItem(i18n("Change &Recording Device..."), 
                        this, SLOT(slotChangeRecordingDevice()));

  PopupMenu->insertSeparator();

  nMenuIDRename = PopupMenu->insertItem(i18n("Re&name..."), this, SLOT(slotRename()));
  nMenuIDRemove = PopupMenu->insertItem(i18n("&Remove"), this, SLOT(slotRemove()));

  PopupMenu->insertSeparator();

  nMenuIDStartPosition = PopupMenu->insertItem(i18n("Change Start P&osition..."),
                        this, SLOT(slotChangeStartPosition()));
  nMenuIDOffset = PopupMenu->insertItem(i18n("Change Offset from Origin..."),
                        this, SLOT(slotChangeOffset()));
  nMenuIDLength = PopupMenu->insertItem(i18n("Change Playing Time Length..."),
                        this, SLOT(slotChangeLength()));
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxTrack::name() const
{
  return LabelName->text();
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxTrack::playbackDevice() const
{
  return strOutputDevice;
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxTrack::recordingDevice() const
{
  return strInputDevice;
}

/////////////////////////////////////////////////////////////////////////////

int
WVoxTrack::level() const
{
  return IntNumInputLevel->value();
}

/////////////////////////////////////////////////////////////////////////////

int 
WVoxTrack::pan() const
{
  return IntNumInputPan->value();
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxTrack::fileName() const
{
  return FileChooserTrack->fileName();
}

/////////////////////////////////////////////////////////////////////////////

QTime
WVoxTrack::startPosition() const
{
  return TimeStartPosition;
}

/////////////////////////////////////////////////////////////////////////////

QTime
WVoxTrack::offset() const
{
  return TimeOffset;
}

/////////////////////////////////////////////////////////////////////////////

QTime
WVoxTrack::length() const
{
  return TimeLength;
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxTrack::isMuted() const
{
  return PushButtonMute->isOn();
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxTrack::isSetRec() const
{
  return PushButtonRec->isOn();
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxTrack::isBindedToFile() const
{
  return !(FileChooserTrack->fileName()).isEmpty();
}

/////////////////////////////////////////////////////////////////////////////

CTrackInf 
WVoxTrack::stateInTrackInf() const
{
  CTrackInf TrackInfReturn;

  TrackInfReturn.strTrackName    = LabelName->text();
  TrackInfReturn.strFileName     = FileChooserTrack->fileName();
  TrackInfReturn.strInputDevice  = strInputDevice;
  TrackInfReturn.strOutputDevice = strOutputDevice;
  TrackInfReturn.nLevel          = IntNumInputLevel->value();
  TrackInfReturn.nPan            = IntNumInputPan->value();
  TrackInfReturn.TimeStartPosition  = TimeStartPosition;
  TrackInfReturn.TimeOffset      = TimeOffset;
  TrackInfReturn.TimeLength      = TimeLength;
  TrackInfReturn.bMute           = PushButtonMute->isOn();
  TrackInfReturn.bRecPort        = PushButtonRec->isOn();

  return TrackInfReturn;
}

/////////////////////////////////////////////////////////////////////////////

QDomNode
WVoxTrack::stateInDOM(QDomDocument* pDoc) const
{
  QDomElement eleTrack = pDoc->createElement("Track");
  eleTrack.setAttribute("name", LabelName->text());

  // file name
  {
    QDomElement eleFileName = pDoc->createElement("File");
    eleFileName.setAttribute("name", FileChooserTrack->fileName());
    eleTrack.appendChild(eleFileName);
  }

  // I/O device
  {
    QDomElement eleDevice = pDoc->createElement("Device");
    eleDevice.setAttribute("in", strInputDevice);
    eleDevice.setAttribute("out", strOutputDevice);
    eleTrack.appendChild(eleDevice);
  }

  // level
  {
    QDomElement eleLevel = pDoc->createElement("Level");
    eleLevel.setAttribute("value", level());
    eleTrack.appendChild(eleLevel);
  }

  // pan
  {
    QDomElement elePan = pDoc->createElement("Pan");
    elePan.setAttribute("value", pan());
    eleTrack.appendChild(elePan);
  }

  // mute state
  {
    QDomElement eleMute = pDoc->createElement("Mute");
    QString strState;
    if ( isMuted() ) {
      strState = "true";
    } else {
      strState = "false";
    }
    eleMute.setAttribute("state", strState);
  }

  // recording state
  {
    QDomElement eleRec = pDoc->createElement("Record");
    QString strState;
    if ( isSetRec() ) {
      strState = "true";
    } else {
      strState = "false";
    }
    eleRec.setAttribute("state", strState);
  }

  // start position
  {
    QDomElement eleStartPosition = pDoc->createElement("StartPosition");
    QTime TimeTmp;
    eleStartPosition.setAttribute("value", TimeTmp.msecsTo(TimeStartPosition));
    eleTrack.appendChild(eleStartPosition);
  }

  // offset
  {
    QDomElement eleOffset = pDoc->createElement("Offset");
    QTime TimeTmp;
    eleOffset.setAttribute("value", TimeTmp.msecsTo(TimeOffset));
    eleTrack.appendChild(eleOffset);
  }

  // length
  {
    QDomElement eleLength = pDoc->createElement("Length");
    QTime TimeTmp;
    eleLength.setAttribute("value", TimeTmp.msecsTo(TimeLength));
    eleTrack.appendChild(eleLength);
  }

  return eleTrack;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setName(const QString& strName)
{
  LabelName->setText(strName);
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::setLevel(int nLevel)
{
  IntNumInputLevel->setValue(nLevel);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setPan(int nPan)
{
  IntNumInputPan->setValue(nPan);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setFileName(const QString& strFileName)
{
  FileChooserTrack->setFileName(strFileName);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setStartPosition(const QTime& TimeStartPositionArg)
{
  TimeStartPosition = TimeStartPositionArg;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setOffset(const QTime& TimeOffsetArg)
{
  TimeOffset = TimeOffsetArg;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setLength(const QTime& TimeLengthArg)
{
  TimeLength = TimeLengthArg;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setMute(bool bMute)
{
  PushButtonMute->setOn(bMute);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setRec(bool bRecPort)
{
  PushButtonRec->setOn(bRecPort);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setStateByTrackInf(const CTrackInf& TrackInfSet)
{
  LabelName->setText(TrackInfSet.strTrackName);
  FileChooserTrack->setFileName(TrackInfSet.strFileName);
  strInputDevice  = TrackInfSet.strInputDevice;
  strOutputDevice = TrackInfSet.strOutputDevice;
  IntNumInputLevel->setValue(TrackInfSet.nLevel);
  IntNumInputPan->setValue(TrackInfSet.nPan);
  TimeStartPosition = TrackInfSet.TimeStartPosition;
  TimeOffset        = TrackInfSet.TimeOffset;
  TimeLength        = TrackInfSet.TimeLength;
  PushButtonMute->setOn(TrackInfSet.bMute);
  PushButtonRec->setOn(TrackInfSet.bRecPort);
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::setStateByDom(const QDomNode& nodTrack)
{
  // name
  {
    QDomNamedNodeMap nnmAttributes = nodTrack.attributes();
    if ( nnmAttributes.count() == 0 ) {
      LabelName->setText("");
    } else {
      QDomNode nodName = nnmAttributes.namedItem("name");
      LabelName->setText(nodName.nodeValue());
    }
  }

  QDomNodeList nolChildren = nodTrack.childNodes();

  // file name
  {
    QDomNode nodFile = nolChildren.item(0);
    if ( nodFile.isNull() ){
      FileChooserTrack->setFileName("");
    } else {
      QDomNamedNodeMap nnmAttributes = nodFile.attributes();
      if ( nnmAttributes.count() == 0 ) {
        FileChooserTrack->setFileName("");
      } else {
        QDomNode nodName = nnmAttributes.namedItem("name");
        FileChooserTrack->setFileName(nodName.nodeValue());
      }
    }
  }

  // device
  {
    QDomNode nodDevice = nolChildren.item(1);
    if ( nodDevice.isNull() ){
      strInputDevice = "/dev/dsp";
      strOutputDevice = "/dev/dsp";
    } else {
      QDomNamedNodeMap nnmAttributes = nodDevice.attributes();
      if ( nnmAttributes.count() == 0 ) {
        strInputDevice = "/dev/dsp";
        strOutputDevice = "/dev/dsp";
      } else {
        QDomNode nodIn = nnmAttributes.namedItem("in");
        strInputDevice = nodIn.nodeValue();
        QDomNode nodOut = nnmAttributes.namedItem("out");
        strOutputDevice = nodOut.nodeValue();
      }
    }
  }

  // Level
  {
    QDomNode nodLevel = nolChildren.item(2);
    if ( nodLevel.isNull() ){
      IntNumInputLevel->setValue(100);
    } else {
      QDomNamedNodeMap nnmAttributes = nodLevel.attributes();
      if ( nnmAttributes.count() == 0 ) {
        IntNumInputLevel->setValue(100);
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        IntNumInputLevel->setValue(nodValue.nodeValue().toInt());
      }
    }
  }

  // Pan
  {
    QDomNode nodPan = nolChildren.item(3);
    if ( nodPan.isNull() ){
      IntNumInputPan->setValue(50);
    } else {
      QDomNamedNodeMap nnmAttributes = nodPan.attributes();
      if ( nnmAttributes.count() == 0 ) {
        IntNumInputPan->setValue(50);
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        IntNumInputPan->setValue(nodValue.nodeValue().toInt());
      }
    }
  }

  // start position
  {
    QDomNode nodStartPosition = nolChildren.item(4);
    TimeStartPosition = QTime();
    if ( nodStartPosition.isNull() ){
      ;
    } else {
      QDomNamedNodeMap nnmAttributes = nodStartPosition.attributes();
      if ( nnmAttributes.count() == 0 ) {
        ;
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        TimeStartPosition.addMSecs(nodValue.nodeValue().toInt());
      }
    }
  }

  // offset
  {
    QDomNode nodOffset = nolChildren.item(5);
    TimeOffset = QTime();
    if ( nodOffset.isNull() ){
      ;
    } else {
      QDomNamedNodeMap nnmAttributes = nodOffset.attributes();
      if ( nnmAttributes.count() == 0 ) {
        ;
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        TimeOffset.addMSecs(nodValue.nodeValue().toInt());
      }
    }
  }

  // length
  {
    QDomNode nodLength = nolChildren.item(6);
    TimeLength = QTime();
    if ( nodLength.isNull() ){
      ;
    } else {
      QDomNamedNodeMap nnmAttributes = nodLength.attributes();
      if ( nnmAttributes.count() == 0 ) {
        ;
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        TimeLength.addMSecs(nodValue.nodeValue().toInt());
      }
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::mousePressEvent(QMouseEvent* MouseEvent)
{
  if ( MouseEvent->button() == Qt::RightButton ) {
    PopupMenu->exec(QCursor::pos());
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotSetEnabledWidgets(unsigned int unWidgets, bool bEnabled)
{
  if ( unWidgets & Mute ) {
    PushButtonMute->setEnabled(bEnabled);
  }
  if ( unWidgets & Record ) {
    PushButtonRec->setEnabled(bEnabled);
  }
  if ( unWidgets & File ) {
    FileChooserTrack->setEnabled(bEnabled);
  }
  if ( unWidgets & Remove ) {
    PopupMenu->setItemEnabled(nMenuIDRemove, bEnabled);
  }
  if ( unWidgets & Device ) {
    PopupMenu->setItemEnabled(nMenuIDPlaybackDevice, bEnabled);
    PopupMenu->setItemEnabled(nMenuIDRecordingDevice, bEnabled);
  }
  if ( unWidgets & StartPosition ) {
    PopupMenu->setItemEnabled(nMenuIDStartPosition, bEnabled);
  }
  if ( unWidgets & Offset ) {
    PopupMenu->setItemEnabled(nMenuIDOffset, bEnabled);
  }
  if ( unWidgets & Length ) {
    PopupMenu->setItemEnabled(nMenuIDLength, bEnabled);
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotChangePlaybackDevice()
{
  KLineEditDlg LineEditDlg("Playback device of this track",
                           strOutputDevice, this);
  if ( LineEditDlg.exec() == QDialog::Accepted ) {
    strOutputDevice = LineEditDlg.text();
    emit playbackDeviceChanged(strOutputDevice);
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotChangeRecordingDevice()
{
  KLineEditDlg LineEditDlg("Recording device of this track",
                           strInputDevice, this);
  if ( LineEditDlg.exec() == QDialog::Accepted ) {
    strInputDevice = LineEditDlg.text();
    emit recordingDeviceChanged(strInputDevice);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotRemove()
{
  QString strMessage = "Are you sure to remove this track?";
  if ( KMessageBox::questionYesNo(this, strMessage) == KMessageBox::Yes ) {
    emit removed();
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotRename()
{

}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotChangeStartPosition()
{
  WTimeDialog TimeDialog(this);

  TimeDialog.setCaption("Set Start Position");
  TimeDialog.setTime(TimeStartPosition);
  TimeDialog.setDisplay(
      WTimeDialog::Minutes | WTimeDialog::Seconds | WTimeDialog::MilliSecs);

  if ( TimeDialog.exec() == QDialog::Accepted ) {
    TimeStartPosition = TimeDialog.time();
    emit startPositionChanged(TimeStartPosition);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotChangeOffset()
{
  WTimeDialog TimeDialog(this);

  TimeDialog.setCaption("Set Offset from Origin");
  TimeDialog.setTime(TimeOffset);
  TimeDialog.setDisplay(
      WTimeDialog::Minutes | WTimeDialog::Seconds | WTimeDialog::MilliSecs);

  if ( TimeDialog.exec() == QDialog::Accepted ) {
    TimeOffset = TimeDialog.time();
    emit offsetChanged(TimeOffset);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotChangeLength()
{
  WTimeDialog TimeDialog(this);

  TimeDialog.setCaption("Set Length for Playing Time");
  TimeDialog.setTime(TimeLength);
  TimeDialog.setDisplay(
      WTimeDialog::Minutes | WTimeDialog::Seconds | WTimeDialog::MilliSecs);

  if ( TimeDialog.exec() == QDialog::Accepted ) {
    TimeLength = TimeDialog.time();
    emit lengthChanged(TimeLength);
  }
}

/////////////////////////////////////////////////////////////////////////////

#include "wvoxtrack.moc"
