/***************************************************************************
                          wladspadialog.cpp  -  description
                             -------------------
    begin                : 2004ǯ  7 18  02:26:12 JST
    copyright            : (C) 2004 by Tomoaki Murakami
    email                : solarisphere@yahoo.co.jp
 ***************************************************************************/

/***************************************************************************
 *   Copyright (C) 2004 by Tomoaki Murakami                                *
 *   solarisphrere@yahoo.co.jp                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdlib.h>
#include <ladspa.h>
#include <qcursor.h>
#include <qdir.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <klocale.h>
#include <kmessagebox.h>
#include "cvoxstormglobal.h"
#include "wladspadialog.h"
#include "cladspafactory.h"

/////////////////////////////////////////////////////////////////////////////

WLadspaDialog::WLadspaDialog(
  const QString& strLadspaPluginPath, 
  QWidget*       WidgetParent,
  const char*    cName)
  : KDialogBase(KDialogBase::Plain,
                "Set Effects",
                Ok | Cancel,
                Ok,
                WidgetParent,
                cName), 
    effectsApply(CVoxStormGlobal::MaxEffectCounts),
    nEffectCounts(0), 
    PortEditor(NULL)
{
  slotSetLadspaPath(strLadspaPluginPath);

  setupWidgets();
  setupPopupMenus();
  slotLoadAvailableEffects();
}

/////////////////////////////////////////////////////////////////////////////

WLadspaDialog::~WLadspaDialog()
{
}

/////////////////////////////////////////////////////////////////////////////

void
WLadspaDialog::setupWidgets()
{
  QFrame *FramePage = plainPage();

  QVBoxLayout* LayoutTop = new QVBoxLayout(FramePage);
  QLabel*      Label;

  // display LADSPA path
  {
    QHBoxLayout* Layout = new QHBoxLayout(FramePage);

    Label = new QLabel(FramePage);
    LabelLadspaPath = new QLabel(FramePage);
    LabelLadspaPath->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    Layout->addWidget(Label);
    Layout->addWidget(LabelLadspaPath);
    Layout->addStretch();

    if ( !this->strLadspaPath.isEmpty() ) {
      Label->setText(i18n("Searched plugin from "));
      Label->setAlignment(AlignRight);
      LabelLadspaPath->setText(this->strLadspaPath);
    }

    LayoutTop->addLayout(Layout, 0);
  }

  LayoutTop->insertSpacing(1, 20);

  // display effect lists (available and applied)
  {
    QHBoxLayout* LayoutEffect = new QHBoxLayout(FramePage);

    // display available effect list
    {
      QVBoxLayout* Layout = new QVBoxLayout(FramePage);
      Label = new QLabel(i18n("Available effects list"), FramePage);
      Layout->addWidget(Label);

      ListViewAvailable = new KListView(FramePage);
      ListViewAvailable->addColumn(i18n("Effect"));
      ListViewAvailable->addColumn(i18n("File"));
      ListViewAvailable->addColumn(i18n("ID"));
      ListViewAvailable->setColumnWidthMode(0, QListView::Manual);
      ListViewAvailable->setColumnWidthMode(2, QListView::Manual);
      ListViewAvailable->setColumnWidth(0, 100);
      ListViewAvailable->hideColumn(2);
      connect(ListViewAvailable, 
              SIGNAL(doubleClicked(QListViewItem*, const QPoint&, int)), 
              this, 
              SLOT(slotAdd()));
      Layout->addWidget(ListViewAvailable);

      LayoutEffect->addLayout(Layout);
    }

    // display command buttons
    {
      QVBoxLayout* Layout = new QVBoxLayout(FramePage);
      QPushButton* Button;

      Layout->addStretch();
      Button = new QPushButton(i18n("Add"), FramePage);
      connect(Button, SIGNAL(clicked()), this, SLOT(slotAdd()));
      Layout->addWidget(Button);

      Button = new QPushButton(i18n("Remove"), FramePage);
      connect(Button, SIGNAL(clicked()), this, SLOT(slotRemove()));
      Layout->addWidget(Button);

      Layout->addStretch();
      Layout->addStretch();

      LayoutEffect->addLayout(Layout);
    }

    // display applied effect list
    {
      QVBoxLayout* Layout = new QVBoxLayout(FramePage);
      Label = new QLabel(i18n("Applied effects list"), FramePage);
      Layout->addWidget(Label);

      ListViewApply = new KListView(FramePage);
      ListViewApply->addColumn(i18n("Effect"));
      ListViewApply->addColumn(i18n("File"));
      ListViewApply->addColumn(i18n("ID"));
      ListViewApply->setColumnWidthMode(0, QListView::Manual);
      ListViewApply->setColumnWidthMode(1, QListView::Manual);
      ListViewApply->setColumnWidthMode(2, QListView::Manual);
      ListViewApply->setColumnWidth(0, 100);
      ListViewApply->hideColumn(2);
      connect(ListViewApply, SIGNAL(selectionChanged()), 
              this, SLOT(slotShowParamEditor()));
      Layout->addWidget(ListViewApply);

      LayoutEffect->addLayout(Layout);
    }

    LayoutTop->addLayout(LayoutEffect, 10);
  }

  LayoutTop->insertSpacing(3, 10);

  // display port editor panel
  {
    WidgetEditor = new QWidget(FramePage);
    WidgetEditor->setMinimumHeight(220);

    QVBoxLayout* Layout = new QVBoxLayout(WidgetEditor);
    Label = new QLabel(i18n("Selected effect's parameter"), WidgetEditor);
    Layout->addWidget(Label, 0);

    LayoutEditor = new QHBoxLayout(WidgetEditor);
    Layout->addLayout(LayoutEditor);

    LabelEffectName = new QLabel("Not selected", WidgetEditor);
    LayoutEditor->addWidget(LabelEffectName, 10);

    LayoutEditor->addStretch();

    LayoutTop->addWidget(WidgetEditor, 2);
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::setupPopupMenus()
{
  // available-list's popup
  {
    PopupOnAvailable = new QPopupMenu(ListViewAvailable);
    PopupOnAvailable->insertItem(i18n("Add"),
                      this, 
                      SLOT(slotAdd()));
    connect(ListViewAvailable, 
            SIGNAL(rightButtonPressed(QListViewItem*, const QPoint&, int)), 
            this, 
            SLOT(slotPopupOnAvailable()));
  }

  // applied-list's popup
  {
    PopupOnApplied = new QPopupMenu(ListViewApply);
    PopupOnApplied->insertItem(i18n("Remove"),
                      this, 
                      SLOT(slotRemove()));
    connect(ListViewApply, 
            SIGNAL(rightButtonPressed(QListViewItem*, const QPoint&, int)), 
            this, 
            SLOT(slotPopupOnApplied()));
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::setEffects(const QValueVector<CLadspaEffect>& effects)
{
  slotClearAppliedEffects();

  int i = 0;
  QValueVector<CLadspaEffect>::const_iterator it = effects.begin();
  while ( it != effects.end() ) {
    if ( i > CVoxStormGlobal::MaxEffectCounts ) {
      break;
    }

    setTo(i, *it);
    i++;
    it++;
  }

  if ( effects.size() > 0 ) {
    ListViewApply->setSelected(ListViewApply->firstChild(), true);
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::getEffects(QValueVector<CLadspaEffect>& effectsDest)
{
  effectsDest.clear();

  slotShowParamEditor(); // to apply current effect information to effectsApply
  QValueVector<CLadspaEffect>::const_iterator it = effectsApply.begin();
  while ( it != effectsApply.end() ) {
    if ( !(*it).libFileName().isEmpty() ) {
      effectsDest.push_back(*it);
    }

    it++;
  }
}

/////////////////////////////////////////////////////////////////////////////

QString 
WLadspaDialog::ladspaPath() const
{
  return strLadspaPath;
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::slotSetLadspaPath(const QString& strPath)
{
  if ( strPath.isEmpty() ) {
    strLadspaPath = getenv("LADSPA_PATH");
  } else {
    strLadspaPath = strPath;
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::slotClearAppliedEffects()
{
  QValueVector<CLadspaEffect>::iterator it = effectsApply.begin();
  while ( it != effectsApply.end() ) {
    *it = CLadspaEffect();
    it++;
  }

  if ( PortEditor != NULL ) {
    delete PortEditor;
    PortEditor = NULL;
  }

  nEffectCounts = 0;
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::slotLoadAvailableEffects()
{
  CLadspaFactory* ladspaLoader = CLadspaFactory::instance();
  if ( ladspaLoader == NULL ) {
    return;
  }

  int i;
  int nEffects = ladspaLoader->effectCount();
  for ( i = 0; i < nEffects; i++ ) {
    QListViewItem* Item = new QListViewItem(ListViewAvailable);
    CLadspaEffect effect(ladspaLoader->effect(i));
    Item->setText(0, effect.name());
    Item->setText(1, effect.libFileName());
    Item->setText(2, QString("%1").arg(i));
    ListViewAvailable->insertItem(Item);
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WLadspaDialog::slotAdd()
{
  const QListViewItem* Item = ListViewAvailable->selectedItem();
  if ( Item == NULL ) {  // not selected
    return;
  }
  
  int nEmptyIndex;
  {
    QString strMessage = i18n("You cannot add effect.\n");
    strMessage += i18n("max effect counts : ");
    strMessage += QString("%1").arg(CVoxStormGlobal::MaxEffectCounts);

    if ( nEffectCounts >= CVoxStormGlobal::MaxEffectCounts ) {
      KMessageBox::sorry(this, strMessage);
      return;
    }
    
    if ( (nEmptyIndex = firstNotUsedSlotID()) == -1 ) {
      KMessageBox::sorry(this, strMessage);
      return;
    }
  }

  int nID = Item->text(2).toInt();
  QListViewItem* ItemAdd = 
    setTo(nEmptyIndex, CLadspaFactory::instance()->effect(nID));

  if ( ItemAdd != NULL ) {
    ListViewApply->setSelected(ItemAdd, true);
  }
}

/////////////////////////////////////////////////////////////////////////////

QListViewItem*
WLadspaDialog::setTo(int nSlotIndex, const CLadspaEffect& effect)
{
  if ( effect.libFileName().isEmpty() ) {
    return NULL;
  }

  effectsApply[nSlotIndex] = effect;

  QListViewItem* ItemAdd = new QListViewItem(ListViewApply);
  ItemAdd->setText(0, effect.name());
  ItemAdd->setText(1, effect.libFileName());
  // ΤˡeffectsApplyźݻ롣
  ItemAdd->setText(2, QString("%1").arg(nSlotIndex));
  ListViewApply->insertItem(ItemAdd);

  nEffectCounts++;

  return ItemAdd;
}

/////////////////////////////////////////////////////////////////////////////

void
WLadspaDialog::slotRemove()
{
  if ( nEffectCounts == 0 ) { // no applied effect
    return;
  }

  QListViewItem* Item = ListViewApply->selectedItem();
  if ( Item == NULL ) {  // not selected
    return;
  }

  int nRemoveIndex = Item->text(2).toInt();
  int i = 0;
  QValueVector<CLadspaEffect>::iterator it = effectsApply.begin();
  while ( it != effectsApply.end() ) {
    if ( i == nRemoveIndex ) {
      *it = CLadspaEffect(); // default is unavailable
      break;
    }

    it++;
    i++;
  }

  ListViewApply->takeItem(Item);
  delete Item;

  if ( PortEditor != NULL ) {
    delete PortEditor;
    PortEditor = NULL;
  }

  nEffectCounts--;
}

/////////////////////////////////////////////////////////////////////////////

int 
WLadspaDialog::firstNotUsedSlotID() const
{
  int i;
  for ( i = 0; i < CVoxStormGlobal::MaxEffectCounts; i++ ) {
    if ( effectsApply[i].label().isEmpty() ) {
      return i;
    }
  }

  return -1;
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::slotShowParamEditor()
{
  static int nCurrentIndex = -1;

  if ( PortEditor != NULL ) {
    effectsApply[nCurrentIndex] = PortEditor->effectSetting();
    delete PortEditor;
    PortEditor = NULL;
  }

  QListViewItem* Item = ListViewApply->selectedItem();
  if ( Item == NULL ) {  // not selected
    LabelEffectName->setText("Not selected");
    nCurrentIndex = -1;
    return;
  }

  LabelEffectName->setText(Item->text(0));
  PortEditor = new WEffectPortEditor(
                     effectsApply[Item->text(2).toInt()], 
                     WidgetEditor);
  LayoutEditor->addWidget(PortEditor, 50);
  nCurrentIndex = Item->text(2).toInt();
  PortEditor->show();
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::slotPopupOnAvailable()
{
  PopupOnAvailable->exec(QCursor::pos());
}

/////////////////////////////////////////////////////////////////////////////

void 
WLadspaDialog::slotPopupOnApplied()
{
  PopupOnApplied->exec(QCursor::pos());
}

/////////////////////////////////////////////////////////////////////////////

#include "wladspadialog.moc"
