/***************************************************************************
                          wvoxtrack.cpp  -  description
                             -------------------
    begin                : 2004ǯ  5  4  17:01:12 JST
    copyright            : (C) 2003 by Tomoaki Murakami
    email                : solarisphere@yahoo.co.jp
 ***************************************************************************/

/***************************************************************************
 *   Copyright (C) 2004 by Tomoaki Murakami                                *
 *   solarisphrere@yahoo.co.jp                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdlib.h>
#include <qcursor.h>
#include <qlayout.h>
#include <qscrollview.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <klineeditdlg.h>
#include "wvoxtrack.h"
#include "wnamedialog.h"
#include "wtimedialog.h"
#include "wladspadialog.h"
#include "cvoxstormglobal.h"

/////////////////////////////////////////////////////////////////////////////

unsigned int WVoxTrack::Mute(0x1);
unsigned int WVoxTrack::Record(0x2);
unsigned int WVoxTrack::File(0x4);
unsigned int WVoxTrack::Remove(0x8);
unsigned int WVoxTrack::Device(0x10);
unsigned int WVoxTrack::StartPosition(0x20);
unsigned int WVoxTrack::Offset(0x40);
unsigned int WVoxTrack::Length(0x80);

/////////////////////////////////////////////////////////////////////////////

WVoxTrack::WVoxTrack(const QString& strName,
                     const QString& strDeviceIn,
                     const QString& strDeviceOut,
                     const QTime&   timeMaxSession, 
                     QWidget*       WidgetParent,
                     const char*    cName)
  : QWidget(WidgetParent, cName)
{
  strInputDevice = strDeviceIn;
  strOutputDevice = strDeviceOut;
  setupWidgets(timeMaxSession);
  LabelName->setText(strName);
}

/////////////////////////////////////////////////////////////////////////////

WVoxTrack::WVoxTrack(const CTrackInf& trackInf,
                     const QTime&     timeMaxSession, 
                     QWidget*         WidgetParent, 
                     const char*      cName)
  : QWidget(WidgetParent, cName)
{
  setupWidgets(timeMaxSession);
  setStateByTrackInf(trackInf);
}

/////////////////////////////////////////////////////////////////////////////

WVoxTrack::WVoxTrack(const QDomNode& nodTrack,
                     const QTime&    timeMaxSession, 
                     QWidget*        WidgetParent, 
                     const char*     cName)
  : QWidget(WidgetParent, cName)
{
  setupWidgets(timeMaxSession);
  setStateByDom(nodTrack);
}

/////////////////////////////////////////////////////////////////////////////

WVoxTrack::~WVoxTrack()
{
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setupWidgets(const QTime& timeMaxSession)
{
  // this has VBoxController
  QHBoxLayout* HBoxLayout = new QHBoxLayout(this);
  
  setupPopupMenu();
  QHBox* HBoxTop = new QHBox(this);
  HBoxTop->setFrameStyle(QFrame::WinPanel | QFrame::Plain);
  HBoxLayout->addWidget(HBoxTop);

  VBoxController = new QVBox(HBoxTop);

  // VBoxController has HBoxStandard, HBoxButtonBase
  HBoxStandard = new QHBox(VBoxController);

  // HBoxStandard has LabelName, VBoxValue
  LabelName = new QLabel(HBoxStandard);
  LabelName->setFrameStyle(QFrame::WinPanel | QFrame::Sunken);

  VBoxValue = new QVBox(HBoxStandard);
  VBoxValue->setMaximumWidth(200);
  VBoxValue->setMinimumWidth(200);

  // VBoxValue has LabelLevel, IntNumInputLevel, LabelPan, IntNumInputPan
  LabelLevel = new QLabel("Level (%)", VBoxValue);

  IntNumInputLevel = new KIntNumInput(VBoxValue);
  IntNumInputLevel->setRange(0, 1600);
  IntNumInputLevel->setValue(100);  // 100 %

  LabelPan = new QLabel("Pan (center : 50)", VBoxValue);
  
  IntNumInputPan = new KIntNumInput(VBoxValue);
  IntNumInputPan->setRange(0, 100);
  IntNumInputPan->setValue(50);     // just center

  HBoxButtonBase = new QHBox(VBoxController);

  // HBoxButtonBase has FileChooserTrack, WidgetSpacer, ColorButtonMute, ColorButtonRec
  FileChooserTrack = new WFileChooser("", i18n("Detect sound source file"), HBoxButtonBase);
  FileChooserTrack->setFileType(i18n("*.wav *.mp3|Sound files (*.wav; *.mp3)\n*"
                                     "|All files (*)"));

  WidgetSpacer = new QHBox(HBoxButtonBase);
  WidgetSpacer->setMinimumWidth(10);
  WidgetSpacer->setMaximumWidth(10);

  // mute button
  PushButtonMute = new QPushButton(HBoxButtonBase);
  QFont font = PushButtonMute->font();
  font.setPointSize(9);
  PushButtonMute->setFont(font);
  PushButtonMute->setPaletteForegroundColor(QColor(0, 255, 0));
  PushButtonMute->setText("M");
  PushButtonMute->setToggleButton(true);
  PushButtonMute->setOn(false);
  PushButtonMute->setMaximumWidth(PushButtonMute->sizeHint().width());

  // record button
  PushButtonRec = new QPushButton(HBoxButtonBase);
  font = PushButtonRec->font();
  font.setPointSize(9);
  PushButtonRec->setFont(font);
  PushButtonRec->setPaletteForegroundColor(QColor(255, 0, 0));
  PushButtonRec->setText("R");
  PushButtonRec->setToggleButton(true);
  PushButtonRec->setOn(false);
  PushButtonRec->setMaximumWidth(PushButtonRec->sizeHint().width());

  //!!
#if 0
  {
    QPushButton* Button = new QPushButton(i18n("Edit Effects..."), HBoxTop);
    connect(Button, SIGNAL(clicked()), this, SLOT(slotEditEffects()));
  }
#endif
  
  HBoxTop->setMaximumWidth(400);
  HBoxTop->setMinimumWidth(400);

  Locator = new WSoundLocator(timeMaxSession, this);
  HBoxLayout->addWidget(Locator, 100);
  Locator->setBaseColor(QColor(0, 0, 0));
  Locator->setItemFaceColor(QColor(111, 109, 158));
  Locator->setItemEdgeColor(QColor(138, 136, 197));

  // setup signals and slots
  connect(FileChooserTrack, SIGNAL(fileNameChanged(const QString&)),
          this, SLOT(slotChangedFileName(const QString&)));
  connect(FileChooserTrack, SIGNAL(fileNameChanged(const QString&)),
          this, SIGNAL(fileNameChanged(const QString&)));
  connect(IntNumInputLevel, SIGNAL(valueChanged(int)), this, SIGNAL(levelChanged(int)));
  connect(IntNumInputPan, SIGNAL(valueChanged(int)), this, SIGNAL(panChanged(int)));
  connect(PushButtonMute, SIGNAL(toggled(bool)), this, SIGNAL(muteToggled(bool)));
  connect(PushButtonRec, SIGNAL(toggled(bool)), this, SIGNAL(recToggled(bool)));
  connect(Locator, SIGNAL(startPositionChanged(const QTime&)), 
          this, SLOT(slotSetStartPosition(const QTime&)));
  connect(Locator, SIGNAL(offsetChanged(const QTime&)), 
          this, SLOT(slotSetOffset(const QTime&)));
  connect(Locator, SIGNAL(lengthChanged(const QTime&)), 
          this, SLOT(slotSetLength(const QTime&)));
  connect(Locator, SIGNAL(contentsMoving(int, int)), 
          this, SLOT(slotMovedContents(int)));
  connect(this, SIGNAL(startPositionChanged(const QTime&)), 
          Locator, SLOT(slotSetStartPosition(const QTime&)));
  connect(this, SIGNAL(offsetChanged(const QTime&)), 
          Locator, SLOT(slotSetOffset(const QTime&)));
  connect(this, SIGNAL(lengthChanged(const QTime&)), 
          Locator, SLOT(slotSetLength(const QTime&)));

  // state changing signal
  connect(FileChooserTrack, SIGNAL(fileNameChanged(const QString&)), 
          this, SIGNAL(stateChanged()));
  connect(IntNumInputLevel, SIGNAL(valueChanged(int)), 
          this, SIGNAL(stateChanged()));
  connect(IntNumInputPan, SIGNAL(valueChanged(int)), 
          this, SIGNAL(stateChanged()));
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setupPopupMenu()
{
  PopupMenu = new QPopupMenu();

  nMenuIDPlaybackDevice = PopupMenu->insertItem(i18n("Change &Playback Device..."),
                        this, SLOT(slotChangePlaybackDevice()));
  nMenuIDRecordingDevice = PopupMenu->insertItem(i18n("Change &Capture Device..."), 
                        this, SLOT(slotChangeRecordingDevice()));

  PopupMenu->insertSeparator();

  nMenuIDRename = PopupMenu->insertItem(i18n("Re&name..."), this, SLOT(slotRename()));
  nMenuIDRemove = PopupMenu->insertItem(i18n("&Remove"), this, SLOT(slotRemove()));

  PopupMenu->insertSeparator();

  nMenuIDStartPosition = PopupMenu->insertItem(i18n("Change &Start Position..."),
                        this, SLOT(slotChangeStartPosition()));
  nMenuIDOffset = PopupMenu->insertItem(i18n("Change &Offset from Origin..."),
                        this, SLOT(slotChangeOffset()));
  nMenuIDLength = PopupMenu->insertItem(i18n("Change Playing Time &Length..."),
                        this, SLOT(slotChangeLength()));

  PopupMenu->insertSeparator();

  nMenuIDEffects = PopupMenu->insertItem(i18n("Edit Effects..."), 
                        this, SLOT(slotEditEffects()));

  PopupMenu->insertSeparator();

  nMenuIDProperty = PopupMenu->insertItem(
                        i18n("&View Property of This Track"), 
                        this, 
                        SLOT(slotProperty()));
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxTrack::name() const
{
  return LabelName->text();
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxTrack::playbackDevice() const
{
  return strOutputDevice;
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxTrack::recordingDevice() const
{
  return strInputDevice;
}

/////////////////////////////////////////////////////////////////////////////

int
WVoxTrack::level() const
{
  return IntNumInputLevel->value();
}

/////////////////////////////////////////////////////////////////////////////

int 
WVoxTrack::pan() const
{
  return IntNumInputPan->value();
}

/////////////////////////////////////////////////////////////////////////////

QString
WVoxTrack::fileName() const
{
  return FileChooserTrack->fileName();
}

/////////////////////////////////////////////////////////////////////////////

QTime
WVoxTrack::startPosition() const
{
  return timeStartPosition;
}

/////////////////////////////////////////////////////////////////////////////

QTime
WVoxTrack::offset() const
{
  return timeOffset;
}

/////////////////////////////////////////////////////////////////////////////

QTime
WVoxTrack::length() const
{
  return timeLength;
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxTrack::isMuted() const
{
  return PushButtonMute->isOn();
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxTrack::isSetRec() const
{
  return PushButtonRec->isOn();
}

/////////////////////////////////////////////////////////////////////////////

bool
WVoxTrack::isBindedToFile() const
{
  return !(FileChooserTrack->fileName()).isEmpty();
}

/////////////////////////////////////////////////////////////////////////////

CTrackInf 
WVoxTrack::stateInTrackInf() const
{
  CTrackInf trackInfReturn;

  trackInfReturn.strTrackName    = LabelName->text();
  trackInfReturn.strFileName     = FileChooserTrack->fileName();
  trackInfReturn.strInputDevice  = strInputDevice;
  trackInfReturn.strOutputDevice = strOutputDevice;
  trackInfReturn.nLevel          = IntNumInputLevel->value();
  trackInfReturn.nPan            = IntNumInputPan->value();
  trackInfReturn.timeStartPosition  = timeStartPosition;
  trackInfReturn.timeOffset      = timeOffset;
  trackInfReturn.timeLength      = timeLength;
  trackInfReturn.bMute           = PushButtonMute->isOn();
  trackInfReturn.bRecPort        = PushButtonRec->isOn();

  return trackInfReturn;
}

/////////////////////////////////////////////////////////////////////////////

QDomNode
WVoxTrack::stateInDOM(QDomDocument* pDoc) const
{
  QDomElement eleTrack = pDoc->createElement("Track");
  eleTrack.setAttribute("name", LabelName->text());

  // file name
  {
    QDomElement eleFileName = pDoc->createElement("File");
    eleFileName.setAttribute("name", FileChooserTrack->fileName());
    eleTrack.appendChild(eleFileName);
  }

  // I/O device
  {
    QDomElement eleDevice = pDoc->createElement("Device");
    eleDevice.setAttribute("in", strInputDevice);
    eleDevice.setAttribute("out", strOutputDevice);
    eleTrack.appendChild(eleDevice);
  }

  // level
  {
    QDomElement eleLevel = pDoc->createElement("Level");
    eleLevel.setAttribute("value", level());
    eleTrack.appendChild(eleLevel);
  }

  // pan
  {
    QDomElement elePan = pDoc->createElement("Pan");
    elePan.setAttribute("value", pan());
    eleTrack.appendChild(elePan);
  }

  // mute state
  {
    QDomElement eleMute = pDoc->createElement("Mute");
    QString strState;
    if ( isMuted() ) {
      strState = "true";
    } else {
      strState = "false";
    }
    eleMute.setAttribute("state", strState);
  }

  // recording state
  {
    QDomElement eleRec = pDoc->createElement("Record");
    QString strState;
    if ( isSetRec() ) {
      strState = "true";
    } else {
      strState = "false";
    }
    eleRec.setAttribute("state", strState);
  }

  // start position
  {
    QDomElement eleStartPosition = pDoc->createElement("StartPosition");
    QTime timeTmp;
    eleStartPosition.setAttribute("value", timeTmp.msecsTo(timeStartPosition));
    eleTrack.appendChild(eleStartPosition);
  }

  // offset
  {
    QDomElement eleOffset = pDoc->createElement("Offset");
    QTime timeTmp;
    eleOffset.setAttribute("value", timeTmp.msecsTo(timeOffset));
    eleTrack.appendChild(eleOffset);
  }

  // length
  {
    QDomElement eleLength = pDoc->createElement("Length");
    QTime timeTmp;
    eleLength.setAttribute("value", timeTmp.msecsTo(timeLength));
    eleTrack.appendChild(eleLength);
  }

  return eleTrack;
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setName(const QString& strName)
{
  LabelName->setText(strName);
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::setLevel(int nLevel)
{
  IntNumInputLevel->setValue(nLevel);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setPan(int nPan)
{
  IntNumInputPan->setValue(nPan);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setFileName(const QString& strFileName)
{
  FileChooserTrack->setFileName(strFileName);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setMute(bool bMute)
{
  PushButtonMute->setOn(bMute);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setRec(bool bRecPort)
{
  PushButtonRec->setOn(bRecPort);
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::setStateByTrackInf(const CTrackInf& trackInfSet)
{
  LabelName->setText(trackInfSet.strTrackName);
  FileChooserTrack->setFileName(trackInfSet.strFileName);
  strInputDevice  = trackInfSet.strInputDevice;
  strOutputDevice = trackInfSet.strOutputDevice;
  IntNumInputLevel->setValue(trackInfSet.nLevel);
  IntNumInputPan->setValue(trackInfSet.nPan);
  timeStartPosition = trackInfSet.timeStartPosition;
  timeOffset        = trackInfSet.timeOffset;
  timeLength        = trackInfSet.timeLength;
  PushButtonMute->setOn(trackInfSet.bMute);
  PushButtonRec->setOn(trackInfSet.bRecPort);
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::setStateByDom(const QDomNode& nodTrack)
{
  // name
  {
    QDomNamedNodeMap nnmAttributes = nodTrack.attributes();
    if ( nnmAttributes.count() == 0 ) {
      LabelName->setText("");
    } else {
      QDomNode nodName = nnmAttributes.namedItem("name");
      LabelName->setText(nodName.nodeValue());
    }
  }

  QDomNodeList nolChildren = nodTrack.childNodes();

  // file name
  {
    QDomNode nodFile = nolChildren.item(0);
    if ( nodFile.isNull() ){
      FileChooserTrack->setFileName("");
    } else {
      QDomNamedNodeMap nnmAttributes = nodFile.attributes();
      if ( nnmAttributes.count() == 0 ) {
        FileChooserTrack->setFileName("");
      } else {
        QDomNode nodName = nnmAttributes.namedItem("name");
        FileChooserTrack->setFileName(nodName.nodeValue());
      }
    }
  }

  // device
  {
    QDomNode nodDevice = nolChildren.item(1);
    if ( nodDevice.isNull() ){
      strInputDevice = "/dev/dsp";
      strOutputDevice = "/dev/dsp";
    } else {
      QDomNamedNodeMap nnmAttributes = nodDevice.attributes();
      if ( nnmAttributes.count() == 0 ) {
        strInputDevice = "/dev/dsp";
        strOutputDevice = "/dev/dsp";
      } else {
        QDomNode nodIn = nnmAttributes.namedItem("in");
        strInputDevice = nodIn.nodeValue();
        QDomNode nodOut = nnmAttributes.namedItem("out");
        strOutputDevice = nodOut.nodeValue();
      }
    }
  }

  // Level
  {
    QDomNode nodLevel = nolChildren.item(2);
    if ( nodLevel.isNull() ){
      IntNumInputLevel->setValue(100);
    } else {
      QDomNamedNodeMap nnmAttributes = nodLevel.attributes();
      if ( nnmAttributes.count() == 0 ) {
        IntNumInputLevel->setValue(100);
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        IntNumInputLevel->setValue(nodValue.nodeValue().toInt());
      }
    }
  }

  // Pan
  {
    QDomNode nodPan = nolChildren.item(3);
    if ( nodPan.isNull() ){
      IntNumInputPan->setValue(50);
    } else {
      QDomNamedNodeMap nnmAttributes = nodPan.attributes();
      if ( nnmAttributes.count() == 0 ) {
        IntNumInputPan->setValue(50);
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        IntNumInputPan->setValue(nodValue.nodeValue().toInt());
      }
    }
  }

  // start position
  {
    QDomNode nodStartPosition = nolChildren.item(4);
    timeStartPosition = QTime();
    if ( nodStartPosition.isNull() ){
      ;
    } else {
      QDomNamedNodeMap nnmAttributes = nodStartPosition.attributes();
      if ( nnmAttributes.count() == 0 ) {
        ;
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        QTime timeTmp;
        slotSetStartPosition(timeTmp.addMSecs(nodValue.nodeValue().toInt()));
	//        timeStartPosition.addMSecs(nodValue.nodeValue().toInt());
      }
    }
  }

  // offset
  {
    QDomNode nodOffset = nolChildren.item(5);
    timeOffset = QTime();
    if ( nodOffset.isNull() ){
      ;
    } else {
      QDomNamedNodeMap nnmAttributes = nodOffset.attributes();
      if ( nnmAttributes.count() == 0 ) {
        ;
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        QTime timeTmp;
        slotSetOffset(timeTmp.addMSecs(nodValue.nodeValue().toInt()));
	//        timeOffset.addMSecs(nodValue.nodeValue().toInt());
      }
    }
  }

  // length
  {
    QDomNode nodLength = nolChildren.item(6);
    timeLength = QTime();
    if ( nodLength.isNull() ){
      ;
    } else {
      QDomNamedNodeMap nnmAttributes = nodLength.attributes();
      if ( nnmAttributes.count() == 0 ) {
        ;
      } else {
        QDomNode nodValue = nnmAttributes.namedItem("value");
        QTime timeTmp;
        slotSetLength(timeTmp.addMSecs(nodValue.nodeValue().toInt()));
	//        timeLength.addMSecs(nodValue.nodeValue().toInt());
      }
    }
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::setMaxSessionLength(const QTime& time)
{
  Locator->setMaxSessionLength(time);
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::scrollLocator(int x)
{
  disconnect(Locator, SIGNAL(contentsMoving(int, int)), 
          this, SLOT(slotMovedContents(int)));
  Locator->setContentsPos(x, Locator->contentsY());
  connect(Locator, SIGNAL(contentsMoving(int, int)), 
          this, SLOT(slotMovedContents(int)));
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::forceRepaintLocator()
{
  Locator->bind(FileChooserTrack->fileName());
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::mousePressEvent(QMouseEvent* MouseEvent)
{
  if ( MouseEvent->button() == Qt::RightButton ) {
    PopupMenu->exec(QCursor::pos());
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::mouseDoubleClickEvent(QMouseEvent* MouseEvent)
{
  if ( MouseEvent->button() == Qt::LeftButton ) {
    slotProperty();
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotSetEnabledWidgets(unsigned int unWidgets, bool bEnabled)
{
  if ( unWidgets & Mute ) {
    PushButtonMute->setEnabled(bEnabled);
  }
  if ( unWidgets & Record ) {
    PushButtonRec->setEnabled(bEnabled);
  }
  if ( unWidgets & File ) {
    FileChooserTrack->setEnabled(bEnabled);
  }
  if ( unWidgets & Remove ) {
    PopupMenu->setItemEnabled(nMenuIDRemove, bEnabled);
  }
  if ( unWidgets & Device ) {
    PopupMenu->setItemEnabled(nMenuIDPlaybackDevice, bEnabled);
    PopupMenu->setItemEnabled(nMenuIDRecordingDevice, bEnabled);
  }
  if ( unWidgets & StartPosition ) {
    PopupMenu->setItemEnabled(nMenuIDStartPosition, bEnabled);
    Locator->slotSetItemEnabled(bEnabled);
  }
  if ( unWidgets & Offset ) {
    PopupMenu->setItemEnabled(nMenuIDOffset, bEnabled);
    Locator->slotSetItemEnabled(bEnabled);
  }
  if ( unWidgets & Length ) {
    PopupMenu->setItemEnabled(nMenuIDLength, bEnabled);
    Locator->slotSetItemEnabled(bEnabled);
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotSetStartPosition(const QTime& time)
{
  if ( timeStartPosition == time ) {
    return;
  }

  timeStartPosition = time;
  emit startPositionChanged(timeStartPosition);
  emit stateChanged();
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotSetOffset(const QTime& time)
{
  if ( timeOffset == time ) {
    return;
  }

  timeOffset = time;
  emit offsetChanged(timeOffset);
  emit stateChanged();
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotSetLength(const QTime& time)
{
 if ( timeLength == time ) {
    return;
  }

  timeLength = time;
  emit lengthChanged(timeLength);
  emit stateChanged();
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotChangedFileName(const QString& strFileName)
{
  // set defaults value to locator attributes
  slotSetStartPosition(QTime());
  slotSetOffset(QTime());
  slotSetLength(QTime());

  Locator->bind(strFileName);
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotChangePlaybackDevice()
{
  KLineEditDlg lineEditDlg(i18n("Playback device of this track"),
                           strOutputDevice, this);
  if ( lineEditDlg.exec() == QDialog::Accepted ) {
    strOutputDevice = lineEditDlg.text();
    emit playbackDeviceChanged(strOutputDevice);
    emit stateChanged();
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotChangeRecordingDevice()
{
  KLineEditDlg lineEditDlg(i18n("Recording device of this track"),
                           strInputDevice, this);
  if ( lineEditDlg.exec() == QDialog::Accepted ) {
    strInputDevice = lineEditDlg.text();
    emit recordingDeviceChanged(strInputDevice);
    emit stateChanged();
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotRemove()
{
  QString strMessage = i18n("Are you sure to remove this track?");
  if ( KMessageBox::questionYesNo(this, strMessage) == KMessageBox::Yes ) {
    emit removed();
    emit stateChanged();
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotRename()
{

}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotChangeStartPosition()
{
  WTimeDialog timeDialog(this);

  timeDialog.setCaption(i18n("Set Start Position"));
  timeDialog.setTime(timeStartPosition);
  timeDialog.setDisplay(
      WTimeDialog::Minutes | WTimeDialog::Seconds | WTimeDialog::MilliSecs);

  if ( timeDialog.exec() == QDialog::Accepted ) {
    slotSetStartPosition(timeDialog.time());
    Locator->slotSetStartPosition(timeDialog.time());//!!
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotChangeOffset()
{
  WTimeDialog timeDialog(this);

  timeDialog.setCaption(i18n("Set Offset from Origin"));
  timeDialog.setTime(timeOffset);
  timeDialog.setDisplay(
      WTimeDialog::Minutes | WTimeDialog::Seconds | WTimeDialog::MilliSecs);

  if ( timeDialog.exec() == QDialog::Accepted ) {
    slotSetOffset(timeDialog.time());
    Locator->slotSetOffset(timeDialog.time());//!!
  }
}

/////////////////////////////////////////////////////////////////////////////

void
WVoxTrack::slotChangeLength()
{
  WTimeDialog timeDialog(this);

  timeDialog.setCaption(i18n("Set Length for Playing Time"));
  timeDialog.setTime(timeLength);
  timeDialog.setDisplay(
      WTimeDialog::Minutes | WTimeDialog::Seconds | WTimeDialog::MilliSecs);

  if ( timeDialog.exec() == QDialog::Accepted ) {
    slotSetLength(timeDialog.time());
    Locator->slotSetLength(timeDialog.time());//!!
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotEditEffects()
{
  WLadspaDialog ladspaDialog(getenv("LADSPA_PATH"), this);

  if ( ladspaDialog.exec() == QDialog::Accepted ) {
  }
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotProperty()
{
  emit propertyDemanded(stateInTrackInf());
}

/////////////////////////////////////////////////////////////////////////////

void 
WVoxTrack::slotMovedContents(int x)
{
  emit locatorScrolled(this, x);
}

/////////////////////////////////////////////////////////////////////////////

#include "wvoxtrack.moc"
