#ifndef VFIELD_STREAM_H__
#define VFIELD_STREAM_H__

#include "autosock.h"
#include "node.h"
#include "vfield.h"
#include "exception.h"
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <sys/types.h>
#include <vector>
#include <map>
#include <aio.h>
#include <boost/utility.hpp>
#include <boost/shared_array.hpp>
#include <boost/scoped_ptr.hpp>

namespace VFIELD {


static const uint16_t STREAM_HEADER_BYTES = 5;


////
// StreamManager
//
class StreamManagerIMPL;
class Storage;
class ThreadPool;
class DataRange;
class addr46_t;
class VTable;
class StreamManager : private boost::noncopyable {
public:
	StreamManager(	const NodeIdentity& self_identity,
			Storage& storage,		// streamGetDataを配分
			VTable& vtable,			// streamJoinを配分
			ThreadPool& threadpool,
			size_t max_connection,
			unsigned int sweep_interval );
	~StreamManager();
public:
	// クライアント機能
	AutoSock getReadyConnection(const addr46_t& to, const char* request_buf, size_t request_len, uint8_t ack_magic );
	void streamClientJoin(const addr46_t& to, VTable& vtable, uint16_t& result_image_id, uint64_t& result_image_size);
											// ブロックする
public:
	// for SearchEngine
	bool isConnected(const addr46_t& node);
	bool tryConnect(const addr46_t& ndoe) throw();
public:
	// for AutoSock
	void addNewConnection(int sock, const addr46_t& from);		// throw(StreamSocketException)
private:
	boost::scoped_ptr<StreamManagerIMPL> impl;
};


////
// StrippedNodeIdentity
//
class stripped_addr46_t;
class StrippedNodeIdentity {
public:
	const static unsigned short raw_size_part = 17;
public:
	StrippedNodeIdentity(const stripped_addr46_t& strip);	// implicit
public:
	bool isIPv6(void) const { return (m_data_part[0] & NODE_IDENTITY_FLAG_IPV6); }
	inline bool operator== (const NodeIdentity& node) const
	{
		// IPv4射影アドレスはただのIPv4アドレスと同じとみなす
		if( isIPv6() != node.isIPv6() ) {
			// 片方がIPv6で片方がIPv4
			if( *((uint16_t*)&m_data_part[11]) == 0xFFFF ||
				*((uint16_t*)&node.getNetRaw()[11]) == 0xFF ) {
				// どちらかがIPv4射影アドレス
				// IPv4アドレス部分だけ比較
				return (::memcmp(m_data_part+13, node.getNetRaw()+13, 4) == 0);
			} else {
				// どちらもIPv4射影アドレスでは無い
				return (::memcmp(m_data_part, node.getNetRaw(), raw_size_part) == 0);
			}
		} else {
			// どちらもIPv6またはどちらもIPv4
			return (::memcmp(m_data_part, node.getNetRaw(), raw_size_part) == 0);
		}
	}
private:
	char m_data_part[raw_size_part];
};


////
// Exception
//
struct StreamListenException : public SystemCallException {
	StreamListenException(int errno_, const std::string& message) :
		SystemCallException(errno_, message) {}
};
struct StreamConnectFailedException : public std::runtime_error 
{
	StreamConnectFailedException(const std::string& message) :
		std::runtime_error(
			std::string("An error occurred while making new stream connection: ")
			+ message ) {}
	StreamConnectFailedException() :
		std::runtime_error("Unknown error occurred while making new stream connection") {}
};


}  // namespace VFIELD

#endif /* stream.h */
