#include "storage.h"
#include "datarange.h"
#include "rpc/rpcclient.h"
#include "stream/stream.h"
#include "vtable/vtable.h"
#include "vfield.h"
#include "log.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <algorithm>
#include <boost/format.hpp>

namespace VFIELD {


RootStorage::RootStorage(const std::string& image_path, RPCClient& rpc) :
	m_fd(-1), m_rpc(rpc)
{
	// FIXME: O_LARGEFILEフラグが必要か？
	m_fd = ::open(image_path.c_str(), O_RDONLY);
	if( m_fd < 0 ) {
		throw FileSystemException(errno, std::string("Can't open ") + image_path );
	}

	struct stat fd_stat;
	if( ::fstat(m_fd, &fd_stat) < 0 ) {
		::close(m_fd);
		m_fd = -1;
		throw FileSystemException(errno, std::string("Failed to get the status of ") + image_path );
	}
	if( fd_stat.st_size <= 0 ) {
		::close(m_fd);
		m_fd = -1;
		throw FileSystemException(errno, image_path + " is empty file");
	}

	m_image_size = fd_stat.st_size;
	m_image_id = 0;		// XXX: image_idは？

	LogInfo( Log::format("RootStorage is initialized with %1% (%2% bytes)") % image_path % m_image_size);
}

RootStorage::~RootStorage()
{
	if( m_fd >= 0 ) {
		::close(m_fd);
	}
}


uint32_t RootStorage::readData(char* buf, uint64_t from, uint32_t length)
{
	LogDebug0( Log::format("request: %1%") % DataRange(from, from + length - 1) );

	ssize_t rl;
	if( (rl = ::pread(m_fd, buf, length, from)) <= 0 ) {
		throw FileSystemException(
				errno,
				( boost::format("Failed to read data from %1% length %2%") % from % length ).str()
				);
	}

	return rl;
}


uint32_t RootStorage::request(char* buf, uint64_t from, uint32_t length)
{
	return readData(buf, from, length);
}

void RootStorage::rpcFind(const NodeIdentity& from, const DataRange& find_range)
{
	// データを一部でも持っているならNotifySelfUpを送る
	if( find_range.start() < m_image_size ) {
		m_rpc.ucastNotifySelfUp(from);
	}
}

void RootStorage::streamGetData(AutoSock& asock, const DataRange& range)
{
	if( unlikely(range.end() >= m_image_size) ) {
		// 要求されたデータ範囲がイメージサイズより大きい
		LogDebugError("Invalid Strem Get Data (requred data range exceeds the size of the root image)");
		asock.close();	// 閉じる
		// TODO: 「間違ってるよ。再Joinせよ」と知らせる必要がある
		// TODO: ユニキャストで送りたいが、相手のRPCのポート番号が分からない
		// TODO: 今のところはNotifySelfUpをブロードキャスト
		m_rpc.bcastNotifySelfUp();
		return;
	}

	// ブロッキングモードにする
	asock.setBlock();

	char buf[STREAM_BUFFER_SIZE];

	// 最初のデータはSTREAM_GET_DATA_ACK
	buf[0] = STREAM_GET_DATA_ACK_MAGIC;
	asock.write(buf, STREAM_HEADER_SIZE);		// 例外はstream.ccがキャッチ

	uint64_t pos = range.start();
	uint64_t pos_end = range.end();

	// TODO: 最初のデータをまとめて送る？　（このデータが一発で送れなかった場合は？）
	//size_t len = std::min(size_t(10), static_cast<size_t>(pos_end - pos + 1));
	//size_t len = std::min(STREAM_BUFFER_SIZE-1, static_cast<size_t>(pos_end - pos + 1));
	//request(&buf[1], pos, len);
	//pos += (asock.write(buf, len+1) - 1);		// +1はヘッダ分の長さ（lenはボディの長さ）

	{
		uint32_t len;
		uint32_t rl;
		do {
			len = std::min(STREAM_BUFFER_SIZE, static_cast<size_t>(pos_end - pos + 1));
			rl = readData(buf, pos, len);
			pos += asock.write(buf, rl);	// 例外はstream.ccがキャッチ
			// TODO: asock.writeでrl全部書き込めなかったとき、同じデータを2回readDataすることになる
		} while(pos <= pos_end);
	}
}

void RootStorage::initialize(const NodeIdentity& self, VTable& vtable)
{
	DataRange self_range( DataRange(0, m_image_size - 1) );

	// RPC Clientにself rangeをセット
	m_rpc.setSelfRange(self_range);

	// VTableを初期化
	vtable.setImageSize(m_image_size);

	// rpcNotifyUpをブロードキャスト
	m_rpc.bcastNotifySelfUp();

	// VTableに自分を追加
	vtable.rpcNotifyUp(self, self_range);
}


}  // namespace VFIELD
