#include "rpcserver.h"
#include "ip46.h"
#include "datarange.h"
#include "vtable/vtable.h"
#include "stream/stream.h"
#include "search/controller.h"
#include "storage/storage.h"
#include "threadpool.h"
#include "vfield.h"
#include "log.h"
#include <unistd.h>
#include <fcntl.h>
#include <boost/preprocessor.hpp>

namespace VFIELD {


////
// RPCCommand
//
class RPCCommand {
public:
	RPCCommand() {}
	~RPCCommand() {}
public:
	static const ssize_t header_size = RPC_COMMON_HEADER_SIZE;
	static const size_t buf_size = RPC_COMMON_HEADER_SIZE + RPC_MAX_BODY_SIZE;
public:
	char* setbuf(void) throw() { return m_buf; }
	void setlen(unsigned short len) throw() { m_len = len; }
	uint8_t getMagic(void) throw() { return uint8_t(m_buf[0]); }
	uint16_t getImageID(void) throw() { return htons(m_buf[1]); }
	const char* getBody(void) throw() { return &m_buf[header_size]; }
	unsigned short getBodyLen(void) throw() { return m_len - header_size; }
private:
	char m_buf[buf_size];
	unsigned short m_len;
};


////
// RPCServerIMPL
//
class RPCServerIMPL {
public:
	RPCServerIMPL(	const NodeIdentity& self_identity,
			StreamManager& smgr,
			VTable& vtable,
			Storage& storage,
			SearchController& engine,
			ThreadPool& threadpool	);
	~RPCServerIMPL() throw();
public:
	const NodeIdentity& getSelfIdentity(void) const;
private:
	NodeIdentity m_self_identity;
	StreamManager& m_smgr;
	VTable& m_vtable;
	Storage& m_storage;
	SearchController& m_engine;

	int m_listen_sock;
	char m_listen_port[7];

	void RPCWorker(RPCCommand header);
	ThreadPool& m_threadpool;

	void ListenThread(void);
	void stopListenThread(void);
	volatile bool m_end_flag;
	boost::thread m_listen_thread;		// スレッドは必ず最後に初期化
};



RPCServer::RPCServer(	const NodeIdentity& self_identity,
			StreamManager& smgr,
			VTable& vtable,
			Storage& storage,
			SearchController& engine,
			ThreadPool& threadpool	)
	: impl( new RPCServerIMPL(self_identity, smgr, vtable, storage, engine, threadpool) ) {}

RPCServerIMPL::RPCServerIMPL(	const NodeIdentity& self_identity,
				StreamManager& smgr,
				VTable& vtable,
				Storage& storage,
				SearchController& engine,
				ThreadPool& threadpool	) :
	m_self_identity(self_identity),
	m_smgr(smgr),
	m_vtable(vtable),
	m_storage(storage),
	m_engine(engine),
	m_listen_sock(-1),
	m_threadpool(threadpool),
	m_end_flag(false),
	m_listen_thread( boost::bind(&RPCServerIMPL::ListenThread,this) )
{
	if( self_identity.isIPv6() ) {
		// FIXME: ブロードキャストはすべてのインターフェースで待ち受けないと受け取れない？
		//m_listen_sock = listenUDP6(self_identity.getIPv6Address(), self_identity.getPortHostByteOrder());
		m_listen_sock = listenUDP6(&in6addr_any, self_identity.getPortHostByteOrder());
	} else {
		// FIXME: ブロードキャストはすべてのインターフェースで待ち受けないと受け取れない？
		//m_listen_sock = listenUDP4(self_identity.getIPv4Address(), self_identity.getPortHostByteOrder());
		struct in_addr in_any;
		in_any.s_addr = INADDR_ANY;
		m_listen_sock = listenUDP4(in_any, self_identity.getPortHostByteOrder());
	}
	if( m_listen_sock < 0 ) {  // O_NONBLOCKはセットしない
		int err = errno;
		m_end_flag = true;
		m_listen_thread.join();
		throw RPCListenException(err, "Can't listen RPC");	// XXX: ポート番号を出力
	}
	{
		// SO_BROADCASTをセット
		// FIXME: 必要ない？
		int on = 1;
		if( setsockopt(m_listen_sock, SOL_SOCKET, SO_BROADCAST, &on, sizeof(on)) < 0 ) {
			int err = errno;
			m_end_flag = true;
			m_listen_thread.join();
			throw RPCListenException(err, "Can't enable broadcast for listening RPC");
		}
	}
	//LogInfo( Log::format("RPC is listened on %1%:%2%") % addr46_t(self_identity) % self_identity.getPortHostByteOrder() );
	LogInfo( Log::format("RPC is listened on any:%1%") % self_identity.getPortHostByteOrder() );
	LogDebug( Log::format("RPC listen socket: %1%") % m_listen_sock );
}

RPCServer::~RPCServer() {}
RPCServerIMPL::~RPCServerIMPL() throw()
{
	try {
		stopListenThread();

		// すべてjoinしないと、子スレッドが持っているリファレンスが無効になって、
		// セグメンテーションフォールトする
		m_threadpool.joinThreads();

	} catch (const std::runtime_error& ex) {
		LogError( Log::format("Can't stop RPC managing thread regularly: %1%") % ex.what() );
	} catch (...) {
		LogError("Can't stop RPC managing thread regularly");
	}
}

const NodeIdentity& RPCServer::getSelfIdentity(void) const { return impl->getSelfIdentity(); }
const NodeIdentity& RPCServerIMPL::getSelfIdentity(void) const
{
	return m_self_identity;
}

void RPCServerIMPL::RPCWorker(RPCCommand command)
{
	switch( command.getMagic() ) {
	////
	// RPC Notify Up
	//
	case RPC_NOTIFY_UP_MAGIC:
		if( command.getBodyLen() < RPC_NOTIFY_UP_SIZE ) {
			LogDebugError(
				Log::format("Receive invalid RPC Notify Up (too small body size, %1% bytes)")
				% command.getBodyLen() );
		} else {
			try {
				NodeIdentity up_node(&command.getBody()[0]);
				DataRange up_range(&command.getBody()[19]);

				LogDebug0(
					Log::format("Receive RPC Notify Up (%1%:%2%  %3%)")
					% up_node.addressString()
					% up_node.getPortHostByteOrder()
					% up_range );


				// 配分
				m_engine.rpcNotifyUp(up_node, up_range);
				m_vtable.rpcNotifyUp(up_node, up_range);

			} catch (const std::runtime_error& ex) {
				LogWarn(
					Log::format("An error occurred while receiving RPC Notify Up: %1%")
					% ex.what() );
			} catch (...) {
				LogWarn("Unknown error occurred while receiving RPC Notify Up");
			}
		}
		break;


	////
	// RPC Notify Down
	//
	case RPC_NOTIFY_DOWN_MAGIC:
		if( command.getBodyLen() < RPC_NOTIFY_DOWN_SIZE ) {
			LogDebugError(
				Log::format("Receive invalid RPC Notify Down (too small body size, %1% bytes)")
				% command.getBodyLen() );
		} else {
			try {
				NodeIdentity down_node(&command.getBody()[0]);
				LogDebug0(
					Log::format("Receive RPC Notify Down (%1%:%2%)")
					% down_node.addressString()
					% down_node.getPortHostByteOrder() );


				// 配分
				m_vtable.rpcNotifyDown(down_node, m_smgr);

			} catch (const std::runtime_error& ex) {
				LogWarn(
					Log::format("An error occurred while receiving RPC Notify Down: %1%")
					% ex.what() );
			} catch (...) {
				LogWarn("Unknown error occurred while receiving RPC Notify Down");
			}
		}
		break;


	////
	// RPC Find
	//
	case RPC_FIND_MAGIC:
		if( command.getBodyLen() < RPC_FIND_SIZE ) {
			LogDebugError(
				Log::format("Receive invalid RPC Find (too small body size, %1% bytes)")
				% command.getBodyLen() );
		} else {
			try {
				DataRange range(&command.getBody()[0]);
				NodeIdentity up_node(&command.getBody()[16]);
				DataRange up_range(&command.getBody()[37]);

				LogDebug0(
					Log::format("Receive RPC Find (%1%) (notify: %2%:%3%  %4%)")
					% range
					% up_node.addressString()
					% up_node.getPortHostByteOrder()
					% up_range );


				// 配分
				m_storage.rpcFind(up_node, range);
				m_vtable.rpcNotifyUp(up_node, up_range);

			} catch (const std::runtime_error& ex) {
				LogWarn(
					Log::format("An error occurred while receiving RPC Find: %1%")
					% ex.what() );
			} catch (...) {
				LogWarn("Unknown error occurred while receiving RPC Find");
			}
		}
		break;

	////
	// RPC PING
	//
	case RPC_PING_MAGIC:
		if( command.getBodyLen() < RPC_PING_SIZE ) {
			LogDebugError(
				Log::format("Receive invalid RPC Ping (too small body size, %1% bytes)")
				% command.getBodyLen() );
		} else {
			try {
				NodeIdentity up_node(&command.getBody()[0]);
				DataRange up_range(&command.getBody()[19]);

				LogDebug0(
					Log::format("Receive RPC Ping (%1%:%2%  %3%)")
					% up_node.addressString()
					% up_node.getPortHostByteOrder()
					% up_range );

				// 配分
				m_vtable.rpcNotifyUp(up_node, up_range);
				m_vtable.rpcPing(up_node);

			} catch (const std::runtime_error& ex) {
				LogWarn(
					Log::format("An error occurred while receiving RPC Ping: %1%")
					% ex.what() );
			} catch (...) {
				LogWarn("Unknown error occurred while receiving RPC Ping");
			}
		}
		break;


	default:
		LogDebugError(	Log::format("Receive unknown RPC (number 0x%x)")
				% (short)command.getMagic() );
	}
}


namespace {
struct RPCListeningErrorException : public SystemCallException {
	RPCListeningErrorException(int errno_, const std::string& message) :
		SystemCallException(errno_, message) {}
};
}  // noname namespace

void RPCServerIMPL::ListenThread(void)
{
	// コンストラクタでm_listen_sockが初期化されるまでend_flagを確認しながら待つ
	while( m_listen_sock < 0 ) {
		::usleep(100*1000);
		if( m_end_flag ) return;
	}
	LogDebug("RPC listen thread start");

	unsigned short retry = RPC_SERVER_RETRY_LIMIT;
	while(1) {  // retry loop
	try {

	while(!m_end_flag) {
		RPCCommand command;
		LogDebug0("Waiting for RPC");
		ssize_t len = recvfrom46(m_listen_sock, command.setbuf(), RPCCommand::buf_size, 0, NULL);
		if( likely(len >= RPCCommand::header_size) ) {
			LogDebug0( Log::format("Receive RPC command (%1% bytes)") % len );
			command.setlen(len);
			m_threadpool.submitTask(
					boost::bind(&RPCServerIMPL::RPCWorker, this, command) );
		} else if( len > 0 ) {
			LogDebug0( Log::format("Receive RPC command is too small (%1% bytes)") % len );
		} else if( len < 0 && (errno == EAGAIN || errno == EINTR) ) {
			// 何もしない、単に無視
		} else if( len < 0 && errno != EAGAIN && errno != EINTR ) {
			throw RPCListeningErrorException(errno, "Failed to receive RPC packet");
		} else { // len == 0
			if( m_end_flag ) break;
			// end_flagがtrueでないのにソケットがクローズされた
			throw RPCListeningErrorException(errno, "RPC listen socket is closed unexpectedly");
		}
	}
	break;	// retry loopを抜ける

	} //try
	catch (const std::runtime_error& ex) {
		LogWarn( Log::format("An error is occurred while proceeding RPC socket: %1%")
			% ex.what() );
	} catch (const std::bad_alloc& ex) {
		LogWarn("Memory error is occurred while proceeding RPC socket");
	} catch (...) {
		LogWarn("Unknown error is occurred while proceeding RPC socket");
	}

	// FIXME: listen_sockも再初期化が必要

	--retry;

	if( retry <= 0 ) {
		LogWarn("Can't recover RPC server thread");
		// XXX: main()スレッドにエラーを知らせる方法が無い
		exit(91);
	}
	LogWarn("Restarting RPC server thread");

	}  // retry loop
}

void RPCServerIMPL::stopListenThread(void)
{
	m_end_flag = true;

	LogDebug("Stopping RPC listen thread");
	// select()を再開させるために、ダミーパケットを自分に送る
	int trasher;
	if( m_self_identity.isIPv6() ) {
		trasher = socketUDP6();
	} else {
		trasher = socketUDP4();
	}
	if( trasher > 0 && sendto46(trasher, "stop", sizeof("stop"), 0, m_self_identity) >= 0 ) {
		m_listen_thread.join();
	} else {
		throw SystemCallException(errno, "Failed to stop RPC thread");
	}

	if( ::close(m_listen_sock) < 0 ) {
		throw SystemCallException(errno, "An error is occurred while closing RPC listen socket: %1%");
	}

	LogInfo("RPC listen socket is closed");
}


}  // namespace VFIELD
