#include "threadpool.h"
#include "log.h"
#include <functional>
#include <boost/bind.hpp>

namespace VFIELD {


ThreadPool::ThreadPool(unsigned short initial_threads) :
	m_end_flag(false), ready_threads(0)
{
	addThreads(initial_threads);
}

ThreadPool::~ThreadPool()
{
	joinThreads();
}
void ThreadPool::joinThreads(void)
{
	// stopThreadsが返ったときにはすべてのスレッドが終了していることを保証する
	boost::mutex::scoped_lock lk_pool(pool_mutex);
	boost::mutex::scoped_lock lk_ring(ring_mutex);
	if( !m_end_flag ) {	// ring_mutexとpool_mutexで保護
		LogDebug( Log::format("Stopping ThreadPool, number of threads is %1%") % threads_pool.size() );

		// end_flagをtrueにして、スレッドを再開させる
		m_end_flag = true;	// ring_mutexとpool_mutexで保護
		ring_cond.notify_all();	// ring_mutexで保護
		lk_ring.unlock();	// アンロックしないと子スレッドが動かない

		// joinする
		std::for_each(	threads_pool.begin(),	// pool_mutexで保護
				threads_pool.end(),
				std::mem_fun_ref(&boost::thread::join) );
		// threadsを削除
		threads_pool.clear();	// pool_mutexで保護

		LogDebug("ThreadPool is successfully stopped");
	}
}


void ThreadPool::addThreads(unsigned short num)
{
	LogDebug("Adding threads");
	boost::mutex::scoped_lock lk_pool(pool_mutex);
	for( unsigned short i=0; i < num; i++ ) {
		threads_pool.push_back(
				new boost::thread(
					boost::bind(&ThreadPool::WorkerThread, this)
					)
				);
	}
	m_end_flag = false;
	LogDebug( Log::format("Number of threads is %1%") % threads_pool.size());
}

void ThreadPool::addThread(void)
{
	// XXX: スレッド数に制限が必要？
	LogDebug("Adding thread");
	boost::mutex::scoped_lock lk_pool(pool_mutex);
	threads_pool.push_back(
			new boost::thread(
				boost::bind(&ThreadPool::WorkerThread, this)
				)
			);
	m_end_flag = false;
	LogDebug( Log::format("Number of threads is %1%") % threads_pool.size());
}

void ThreadPool::WorkerThread(void)
{
	while(1) {
		// forの中でtry/catchすると、task_queue.pop_back()時に例外が発生するとデッドロックする
		try {
			// キューが空になるまでタスクを実行
			for( boost::mutex::scoped_lock lk_queue(queue_mutex);	// キューをロック
					! task_queue.empty();
					lk_queue.lock() )			// キューをアンロック
			{
					// タスクを取り出す
					boost::ptr_deque<task_type>::auto_type next_task( task_queue.pop_back() );
					lk_queue.unlock();

					// タスクを実行
					(*next_task)();
			}

			// タスクが追加されるまで待つ
			{
				boost::mutex::scoped_lock lk_ring(ring_mutex);

				// m_end_flagはring_mutexのロック内でチェック
				// 必ずキューを空にしてからbreak
				if(m_end_flag) break;

				ready_threads++;
				ring_cond.wait(lk_ring);
				ready_threads--;
			}

		} catch (const std::runtime_error& ex) {
			LogError(ex.what());
		} catch (...) {
			LogError("Catch unknown error while proceeding thread");
		}
	}
}


}  // namespace VFIELD
