#include "usbids.h"
#include <fstream>
#include <sstream>
#include <cstring>
#include <stdexcept>

namespace SimpleDetect {


using namespace USBIDTypes;

namespace {
bool isHex(char c) {
	return ('0'<=c && c<='9') || ('a'<=c && c<='f' );
}
bool isLargeChar(char c) {
	return ('A'<=c && c<='Z');
}
}  // noname namespace


// コンストラクタ
USBIDs::USBIDs(const std::string& ids_path) :
	m_ids_path(ids_path)
{}


// デストラクタ
USBIDs::~USBIDs()
{}


// public:
void USBIDs::load(void)
{
	std::ifstream file(m_ids_path.c_str());
	if( ! file.is_open() ) {
		std::string msg( std::string("Can't open ") + m_ids_path );
		throw std::runtime_error( msg );
	}

	char buf[256];

	// Vendor and Device
	Vendor* current_vendor = NULL;
	while( ! file.eof() ) {
		file.getline(buf, sizeof(buf));
		if( isLargeChar(buf[0]) ) {
			// Classの定義に突入
			// ファイルポインタを一行戻してbreak
			// DeviceとSubClassの書式が同じなので分ける必要がある
			file.seekg( -1 + file.tellg() - strlen(buf) );
			break;
		}

		if( isHex(buf[0]) ) {
			// Vendor
			// 0000  Gammagraphx, Inc.
			if( ::strlen(buf) <= 6 ) continue;  // 不正な行

			Vendor::id_type id;
			buf[4] = '\0';
			std::istringstream id_stream(buf);
			id_stream >> std::hex >> id;
			if( id_stream.fail() ) continue;

			std::string name(&buf[6]);

			Vendor* next_vendor( m_idmap.addVendor(id, name) );
			if( next_vendor == NULL ) continue;

			current_vendor = next_vendor;

		} else if( buf[0] == '\t' && isHex(buf[1])) {
			// Device
			// 	0680 Ultra ATA/133 IDE RAID
			if( ::strlen(buf) <= 7 ) continue;  // 不正な行

			if( current_vendor == NULL) continue;

			Device::id_type id;
			buf[5] = '\0';
			std::istringstream id_stream(buf);
			id_stream >> std::hex >> id;
			if( id_stream.fail() ) continue;

			std::string name(&buf[7]);

			current_vendor->addDevice(id, name);
		}
	}

	// BaseClass and SubClass
	BaseClass* current_baseclass = NULL;
	while( ! file.eof() ) {
		file.getline(buf, sizeof(buf));

		if( buf[0] == 'C' && isHex(buf[2]) ) {
			// BaseClass
			// C 00  Unclassified device
			if( ::strlen(buf) <= 6 ) continue;  // 不正な行

			BaseClass::id_type id;
			buf[4] = '\0';
			std::istringstream id_stream( &buf[2] );
			id_stream >> std::hex >> id;
			if( id_stream.fail() ) continue;

			std::string name(&buf[6]);

			BaseClass* next_baseclass( m_classmap.addBaseClass(id, name) );
			if( next_baseclass == NULL ) continue;

			current_baseclass = next_baseclass;
		} else if( buf[0] == '\t' && isHex(buf[1]) ) {
			// SubClass
			// 	00  Non-VGA unclassified device
			if( ::strlen(buf) <= 5 ) continue;  // 不正な行

			if( current_baseclass == NULL ) continue;

			SubClass::id_type id;
			buf[3] = '\0';
			std::istringstream id_stream( &buf[1] );
			id_stream >> std::hex >> id;
			if( id_stream.fail() ) continue;

			std::string name( &buf[5] );

			current_baseclass->addSubClass(id, name);
		}
	}

	file.close();
}


const std::string* USBIDs::getVendorName(Vendor::id_type id) const
{
	return m_idmap.getVendorName(id);
}

const std::string* USBIDs::getDeviceName(Vendor::id_type vendor_id, Device::id_type device_id) const
{
	return m_idmap.getDeviceName(vendor_id, device_id);
}
const std::string* USBIDs::getBaseClassName(BaseClass::id_type id) const
{
	return m_classmap.getBaseClassName(id);
}
const std::string* USBIDs::getSubClassName(BaseClass::id_type base_id, SubClass::id_type sub_id) const
{
	return m_classmap.getSubClassName(base_id, sub_id);
}


}  // namespace SimpleDetect
