#include "sharedblock.h"
#include "constant.h"
#include "process.h"
#include "filesystem.h"
#include <math.h>
#include <boost/lexical_cast.hpp>

namespace VIVER {


// 定数
const std::string SharedBlockDevice::KERNEL_MODULE("nbd");


// コンストラクタ
SharedBlockDevice::SharedBlockDevice(const Parameters& _parameters, DeviceManager& _devmanager) :
	parameters(_parameters),
	devmanager(_devmanager),
	m_dn_nbd(NULL)
{}


// デストラクタ
SharedBlockDevice::~SharedBlockDevice()
{}


// public:
void SharedBlockDevice::configure(void)
{
	// TODO: IPv4依存
	const char* const pa = parameters.getValueString("bootstrap", NULL);
	const std::string param( (pa == NULL ? "" : pa) );

	std::string::size_type pos_colon;
	if( (pos_colon = param.find(':')) != std::string::npos ) {
		m_bootstrap_ip = param.substr(0, pos_colon);
		m_bootstrap_port = param.substr(pos_colon+1);
	} else {
		m_bootstrap_ip = param;
		m_bootstrap_port = DefaultInfo::SharedBlock::PORT;
	}
}

namespace {
std::string determineDefaultStoreSize(void)
{
	FILE* f_pm;
	std::string ret;
	f_pm = ::fopen("/proc/meminfo", "r");

	if( f_pm == NULL ) {
		ret = "32";
	} else {
		char label_trash[128];
		int mem_total;
		char unit_trash[32];
		fscanf(f_pm, "%s %d %s", label_trash, &mem_total, unit_trash);

		//double smart_size = ::log( (double)mem_total / 1024 / 800 + 1) * 200;
		/*
		double smart_size = ::pow((double)mem_total/1024/20, 0.7) * 18;
		double max_size = (double)mem_total * 0.8 / 1024;
		if( smart_size > max_size ) {
			ret = boost::lexical_cast<std::string>( uint64_t(max_size) );
		} else if( smart_size < 10 ) {
			ret = "0";
		} else {
			ret = boost::lexical_cast<std::string>( uint64_t(smart_size) );
		}
		*/
		double store_size = (double)mem_total/5.5/1024;
		ret = boost::lexical_cast<std::string>( static_cast<uint64_t>(store_size) );
	}
	::fclose(f_pm);
	return ret;
}
}  // noname namespace

DeviceNode* SharedBlockDevice::connect(const std::string& fstype,
		unsigned long mount_flags, const std::string& data_string)
{
	// TODO: この関数のコードはNBDに依存

	std::string chroot_to( MP::ARK.str() );


	// vfieldをArkにコピー
	Path vfield(MP::ARK + "bin/vfield");
	Path back_exec(MP::ARK + "bin/back-exec");

	FileSystem::copyFile(Component::Program::VFIELD, vfield.str());	// throw(FileSystemError)
	FileSystem::copyFile(Component::Program::BACK_EXEC, back_exec.str());	// throw(FileSystemError)


	Path daemon(MP::ARK + "bin/nbd-client");

	// クライアントデーモンをArkにコピー
	FileSystem::copyFile(Component::Program::NBD_CLIENT, daemon.str());	// throw(FileSystemError)

	// devicemanagerにデバイスを要求
	m_dn_nbd = devmanager.requestNode(43, 0, "nbd0", fstype, mount_flags, data_string);
	if( m_dn_nbd->makeNode() < 0 ) {
		throw NoSuchFileError(CANNOT + "make device node " + m_dn_nbd->getNodePath().str() + errorDescribe());
	}

	// chrootしてvfieldとクライアントデーモンを実行
	Process ps_vfield_back( back_exec.chroot(chroot_to).c_str() );
	ps_vfield_back % vfield.chroot(chroot_to).c_str()
		     % "-bs"
		     % m_bootstrap_ip.c_str()
		     % "-port"
		     % m_bootstrap_port.c_str()
		     % "-nbd_port"
		     % "8990";
	if( parameters.getValueString("store", NULL) != NULL ) {
		ps_vfield_back % "-store"
			       % parameters.getValueString("store", NULL);
	} else {
		try {
			std::string intelligent = determineDefaultStoreSize();
			ps_vfield_back % "-store"
				       % intelligent.c_str();
		} catch (...) {
		}
	}
	if( parameters.getValueString("join_delay", NULL) != NULL ) {
		ps_vfield_back % "-join_delay"
			       % parameters.getValueString("delay", NULL);
	} else {
		ps_vfield_back % "-join_delay" % "120";
	}
	if( parameters.getValueString("max_connection", NULL) != NULL ) {
		ps_vfield_back % "-max_connection"
			       % parameters.getValueString("max_connection", NULL);
	}
	if( parameters.getValueString("init_thread", NULL) != NULL ) {
		ps_vfield_back % "-init_thread"
			       % parameters.getValueString("init_thread", NULL);
	}
	if( parameters.getValueString("sweep", NULL) != NULL ) {
		ps_vfield_back % "-sweep"
			       % parameters.getValueString("sweep", NULL);
	}

	Process ps_nbdclient( daemon.chroot(chroot_to).c_str() );
	//ps_nbdclient % "127.0.0.1"
	//	     % m_bootstrap_port.c_str()
	//	     % m_dn_nbd->getNodePath().chroot(chroot_to).c_str();
	ps_nbdclient % "127.0.0.1"
		     % "8990"
		     % m_dn_nbd->getNodePath().chroot(chroot_to).c_str();

	ps_vfield_back.chroot_exec( chroot_to );
	sleep(4);	// XXX:

	ps_nbdclient.chroot_exec( chroot_to );
	if( ps_vfield_back.wait() != 0 ) {
		throw SBDConnectFailedError(CANNOT + "execute vfield " + m_bootstrap_ip + " port " + m_bootstrap_port);
	}
	if( ps_nbdclient.wait() != 0 ) {
		throw SBDConnectFailedError(CANNOT + "connect to " + m_bootstrap_ip + " port " + m_bootstrap_port );
	}
	/* // squashfsが1024に戻してしまう
	sleep(1);	// XXX:
	{  // エラー無視
		Process ps_blockdev(Component::Program::BLOCKDEV);
		ps_blockdev % "--setbsz" % "4096" % m_dn_nbd->getNodePath().c_str();
		ps_blockdev.exec();
	}
	*/

	return m_dn_nbd;
}

void SharedBlockDevice::disconnect(void)
{
	// TODO: この関数のコードはNBDに依存

	std::string chroot_to( MP::ARK.str() );
	Path daemon(MP::ARK + "bin/nbd-client");

	// デーモンをkill
	{  // nbd-client -d
		Process ps_nbdclient( daemon.chroot(chroot_to).c_str() );
		ps_nbdclient % "-d" % m_dn_nbd->getNodePath().chroot(chroot_to).c_str();
		ps_nbdclient.chroot_exec( chroot_to );
		ps_nbdclient.wait();
	}
	{  // killall (SIGTERM)
		Process ps_killall(Component::Program::KILLALL);
		ps_killall % "nbd-client";
		ps_killall.exec();
		ps_killall.wait();
	}
	{  // killall (SIGKILL)
		Process ps_killall(Component::Program::KILLALL);
		ps_killall % "-KILL" % "nbd-client";
		ps_killall.exec();
		ps_killall.wait();
	}

	// devicemanagerにデバイスを返却
	devmanager.returnNode(m_dn_nbd);
	m_dn_nbd = NULL;

	// クライアントデーモンをArkから削除
	FileSystem::remove(daemon.str());
}

}  // namespace VIVER
