#include "sdpci_impl.h"
#include "sdpci_known.h"
#include <fstream>
#include <algorithm>
#include <functional>
#include <cstring>
#include <cstdio>
#include <ostream>
#include <iomanip>

namespace SimpleDetect {


// コンストラクタ
sdPCIIMPL::sdPCIIMPL(const std::string& ids_path, const std::string map_path) :
	m_ids(ids_path),
	m_map(map_path)
{}

// デストラクタ
sdPCIIMPL::~sdPCIIMPL()
{}


// public:
void sdPCIIMPL::initialize(void)
{
	m_ids.load();
	m_map.load();
}

void sdPCIIMPL::setBaseClassBasedLoadConfig  (const std::string& class_substr, load_config_t config)
{
	m_base_class_config.push_back( pattern_config_t(class_substr, config) );
}

void sdPCIIMPL::setSubClassBasedLoadConfig   (const std::string& class_substr, load_config_t config)
{
	m_base_class_config.push_back( pattern_config_t(class_substr, config) );
}

void sdPCIIMPL::setKnownClassBasedLoadConfig (known_class_t known_class, load_config_t config)
{
	m_known_class_config[ known_class ] = config;
}


void sdPCIIMPL::detect(ModuleLoader& loader, bool devices_loaded)
{
	using namespace PCIKnown;
	loadBusDrivers(loader);

	if( ! devices_loaded ) {
		loadDevices();
	}
	devices_t::size_type before( m_devices.size() );

	for( devices_t::const_iterator dev_pair(m_devices.begin()), end(m_devices.end());
			dev_pair != end;
			++dev_pair ) {
		loadModules(loader, dev_pair->second);
	}

	// XXX: sleepの時間を調整 または Bridgeクラスを発見したときのみsleep
	::sleep(1);
	loadDevices();
	devices_t::size_type after( m_devices.size() );

	if( before < after ) {
		// PCI-PCIブリッジのドライバがロードされ、PCIデバイスが増えた
		// 再度detect
		detect(loader, true);	// loadDevices()は更新済みなので、再度実行しない
	}
}


// private:
void sdPCIIMPL::loadDevices(void)
{
	std::ifstream proc_pci("/proc/bus/pci/devices");
	if( ! proc_pci.is_open() ) {
		// デバイス無し
		return;
	}

	char buf[512];
	char devinfo_path[512];

	uint8_t bus_id;
	uint32_t devfn;
	unsigned short vendor_id;
	unsigned short device_id;
	uint8_t irq;

	uint8_t baseclass_id;
	uint8_t subclass_id;
	unsigned short subvendor_id;
	unsigned short subdevice_id;

	while( ! proc_pci.eof() ) {

		{  // bus_id, devfn, vendor_id, device_id, irq
			proc_pci.getline(buf, sizeof(buf));
			::sscanf(buf, "%02hhx%02x %04hx%04hx %02hhx", &bus_id, &devfn, &vendor_id, &device_id, &irq);
			// TODO: Linux/drivers/pci/proc.cによると、ファイル名は %04x:%02x.%x の場合がある
			::sprintf(devinfo_path, "%s/%02hhx/%02x.%x",
					"/proc/bus/pci",
					bus_id,
					( ((devfn)>>3) & 0x1f ),
					((devfn)&0x07) );
		}

		{  // baseclass_id, subclass_id, subvendor_id, subdevice_id
			FILE* devinfo = ::fopen(devinfo_path, "r");
			if( devinfo == NULL ) continue;
			// TODO: 移植するときはバイトオーダー注意？
			::fread(buf, sizeof(char), sizeof(buf), devinfo);
			::fclose(devinfo);

			baseclass_id = buf[0x0b]&0x00ff;
			subclass_id  = buf[0x0a]&0x00ff;
			subvendor_id = (buf[0x2c]&0x00ff) | ((buf[0x2d]&0x00ff)<<8);
			subdevice_id = (buf[0x2e]&0x00ff) | ((buf[0x2f]&0x00ff)<<8);
		}

		PCIDevice device;
		device.vendor_id = vendor_id;
		device.device_id = device_id;
		// device.subvendor_id = subvendor_id;
		// device.subdevice_id = subdevice_id;
		device.baseclass_id = baseclass_id;
		device.subclass_id  = subclass_id;
		device.irq = irq;

		device.modules = m_map.getModuleName(vendor_id, device_id);
		device.loaded = false;

		m_devices[
			static_cast<uint64_t>(bus_id) << 32 |
			static_cast<uint64_t>(devfn)
			] = device;
	}
}


sdPCIIMPL::load_config_t sdPCIIMPL::detLoadConfig(
		const configured_list_t& configured_list,
		const std::string* p_pattern
		) const
{
	if( p_pattern != NULL ) {
		configured_list_t::const_iterator configured(
				std::find_if(
					configured_list.begin(),
					configured_list.end(),
					isMatchPattern(*p_pattern)
					)
				);
		if( configured != configured_list.end() ) {
			return configured->second;
		}
	}
	return sdPCI::AUTO;
}


void sdPCIIMPL::loadModules(ModuleLoader& loader, const PCIDevice& device) const
{
	using namespace PCIKnown;

	if( device.loaded ) return;

	known_class_t known_class;
	if( isKnown(device.baseclass_id, device.subclass_id, &known_class) ) {
		known_class_config_map_t::const_iterator configured_class(
				m_known_class_config.find(known_class)
				);
		if( configured_class != m_known_class_config.end() ) {
			// 設定値あり
			switch(configured_class->second) {
			case sdPCI::NEVER:
				return;		// ロードしない
			case sdPCI::FORCE:
				loadKnownDriversForce(loader, known_class);
				// force load後、auto load
			case sdPCI::AUTO:
				loadKnownDriversAuto(loader, known_class);
				break;
			}
		} else {
			// 設定値無し
			loadKnownDriversAuto(loader, known_class);
		}
	}

	{  // Base Class"名"ベースのコンフィグレーション
		const std::string* baseclass_name( m_ids.getBaseClassName(device.baseclass_id) );
		switch( detLoadConfig(m_base_class_config, baseclass_name) ) {
		case sdPCI::NEVER:
			return;		// ロードしない
		case sdPCI::FORCE:
			// pass through
		case sdPCI::AUTO:
			break;
		}
	}

	{  // Sub Class"名"ベースのコンフィグレーション
		const std::string* subclass_name( m_ids.getSubClassName(device.baseclass_id, device.subclass_id) );
		switch( detLoadConfig(m_base_class_config, subclass_name) ) {
		case sdPCI::NEVER:
			return;		// ロードしない
		case sdPCI::FORCE:
			// pass through
		case sdPCI::AUTO:
			break;
		}
	}

	{  // Specific Deviceのコンフィグレーション
		known_class_config_map_t::const_iterator specific_config(
				m_known_class_config.find(sdPCI::SPECIFIC)
				);
		if( specific_config != m_known_class_config.end()  ||
				specific_config->second == sdPCI::AUTO ||
				specific_config->second == sdPCI::FORCE ) {
			// sdPCI::SPECIFICに設定がされていないか、
			// 設定されいてもAUTOかFORCE
			loadSpecificDrivers(loader, device);
		}
	}

	// FIXME: ModuleLoaderが不完全型なので、for_eachは使えない？
	for( std::list<std::string>::const_iterator end( device.modules.end() ), mod( device.modules.begin() );
			mod != end;
			++mod ) {
		loader.operator()(*mod);
	}
}


void sdPCIIMPL::showDetectedDevices(std::ostream& stream) const
{
	for( devices_t::const_iterator dev_pair(m_devices.begin()), end(m_devices.end());
			dev_pair != end;
			++dev_pair ) {
		const PCIDevice& dev( dev_pair->second );
		// XXX: ストリームのフラグを保存して、出力が終わったらリストアする
		const std::string* vendor_name( m_ids.getVendorName(dev.vendor_id) );
		const std::string* device_name( m_ids.getDeviceName(dev.vendor_id, dev.device_id) );
		const std::string* baseclass_name( m_ids.getBaseClassName(dev.baseclass_id) );
		const std::string* subclass_name ( m_ids.getSubClassName (dev.baseclass_id, dev.subclass_id) );
		stream	<< "----"
			<< std::endl << std::setfill('0')
			<< "Vendor      0x" << std::hex << std::setw(4) << dev.vendor_id
			<< " : " << (vendor_name != NULL ? *vendor_name : "(unknown)")
			<< std::endl
			<< "Device      0x" << std::hex << std::setw(4) << dev.device_id
			<< " : " << (device_name != NULL ? *device_name : "(unknown)")
			<< std::endl
			<< "BaseClass   0x" << std::hex << std::setw(2) << dev.baseclass_id
			<< "   : " << (baseclass_name != NULL ? *baseclass_name : "(unknown)")
			<< std::endl
			<< "SubClass    0x" << std::hex << std::setw(2) << dev.subclass_id
			<< "   : " << (subclass_name  != NULL ? *subclass_name  : "(unknown)")
			<< std::endl
			<< "Modules            : ";
		ModuleCollector collector;
		loadModules(collector, dev_pair->second);
		std::list<std::string> modules( collector.getList() );
		std::copy(modules.begin(), modules.end(), std::ostream_iterator<std::string>(stream, " "));
		stream << std::endl;
	}
}

void sdPCIIMPL::getDetectedDevices(std::vector<DetectedDevicePCI>& result) const
{
	for( devices_t::const_iterator dev_pair(m_devices.begin()), end(m_devices.end());
			dev_pair != end;
			++dev_pair ) {
		const PCIDevice& dev( dev_pair->second );
		const std::string* vendor_name( m_ids.getVendorName(dev.vendor_id) );
		const std::string* device_name( m_ids.getDeviceName(dev.vendor_id, dev.device_id) );
		const std::string* baseclass_name( m_ids.getBaseClassName(dev.baseclass_id) );
		const std::string* subclass_name ( m_ids.getSubClassName (dev.baseclass_id, dev.subclass_id) );

		DetectedDevicePCI r;
		r.vendor_id = dev.vendor_id;
		r.device_id = dev.device_id;
		r.baseclass_id = dev.baseclass_id;
		r.subclass_id = dev.subclass_id;
		r.vendor_name = (vendor_name != NULL ? *vendor_name : "");
		r.device_name = (device_name != NULL ? *device_name : "");
		r.baseclass_name = (baseclass_name != NULL ? *baseclass_name : "");
		r.subclass_name = (subclass_name != NULL ? *subclass_name : "");

		result.push_back(r);
	}
}


}  // namespace SimpleDetect
