#include "process.h"
#include "display.h"
#include "exception.h"
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>

namespace VIVER {


// コンストラクタ
Process::Process(const char* path) :
	m_pid(-1),
	m_waited(false)
{
	(*this) % path;		// args[0]
	m_path = m_args[0];
}


// デストラクタ
Process::~Process()
{
	// TODO: 例外を投げる可能性
	if( !m_waited ) wait();    // waitされていない場合は回収    // TODO: 回収しなくても良い？
	for( std::vector<char*>::iterator arg(m_args.begin()); arg != m_args.end(); ++arg ) {
		delete[] *arg;
	}
}


// public:
Process& Process::operator% (const char* arg)
{
	// XXX: strdup?
	char* tmp_arg = new char[ ::strlen(arg) + 1 ];
	::strcpy(tmp_arg, arg);
	m_args.push_back(tmp_arg);
	return *this;
}



std::pair<int, int> Process::exec(bool grub_cout, bool grub_cerr)
{
	// 引数を作成 ついでにデバッグメッセージを出力
	host::display.verbose() << "Executing";	// args[0] == m_path
	std::auto_ptr<char*> args( new char*[ m_args.size() + 1 ] );
	size_t i;
	for( i=0; i < m_args.size(); i++ ) {
		args.get()[i] = m_args[i];
		host::display.verbose() << " " << m_args[i];
	}
	args.get()[i] = NULL;
	host::display.verbose() << std::endl;

	// パイプ作成
	int pipe_cout[2];
	int pipe_cerr[2];
	if( !grub_cout ) {
		pipe_cout[0] = pipe_cout[1] = -1;
	} else if( ::pipe(pipe_cout) < 0 ) {
		// cout用のパイプの作成に失敗
		host::display.warn() << "Can't make cout pipe for " << m_path << errorDescribe() << std::endl;
		pipe_cout[0] = pipe_cout[1] = -1;
	}	// cout用のパイプの作成に成功

	if( !grub_cerr ) {
		pipe_cerr[0] = pipe_cerr[1] = -1;
	} else if( ::pipe(pipe_cerr) < 0 ) {
		// cerr用のパイプの作成に失敗
		host::display.warn() << "Can't make cerr pipe for " << m_path << errorDescribe() << std::endl;
		pipe_cerr[0] = pipe_cerr[1] = -1;
	}	// cerr用のパイプの作成に成功

	// fork
	if( (m_pid=::fork()) == 0 ) {
		// 子プロセス
		if( pipe_cout[1] != -1 ) {
			::close(pipe_cout[0]);		// 読み込み側は使わないのでクローズ
			::dup2(pipe_cout[1], 1);	// 子プロセスの標準出力 -> pipe_coutパイプの書き込み側
		}
		if( pipe_cerr[1] != -1 ) {
			::close(pipe_cerr[0]);		// 読み込み側は使わないのでクローズ
			::dup2(pipe_cerr[1], 2);	// 子プロセスの標準エラー出力 -> pipe_cerrパイプの書き込み側
		}
		::execv(m_path, args.get());
		// exec失敗
		// FIXME: 標準出力と標準エラー出力を開き直さないといけないのでは？
		std::cerr << "Can't execute " << m_path << errorDescribe() << std::endl;
		if( pipe_cout[1] != -1 ) ::close(pipe_cout[1]);
		if( pipe_cout[1] != -1 ) ::close(pipe_cerr[1]);
		::exit(127);
	} else if( m_pid == -1 ) {
		// fork失敗
		if( pipe_cout[1] != -1 ) {
			::close(pipe_cout[0]);
			::close(pipe_cout[1]);
			pipe_cout[0] = pipe_cout[1] = -1;
		}
		if( pipe_cerr[1] != -1 ) {
			::close(pipe_cerr[0]);
			::close(pipe_cerr[1]);
			pipe_cerr[0] = pipe_cerr[1] = -1;
		}
		throw ExecuteFailedError( CANNOT + "fork " + m_path + errorDescribe() );
	}


	// 親プロセス
	if( pipe_cout[0] != -1 ) {
		::close(pipe_cout[1]);		// 書き込み側は使わないのでクローズ
		host::display.debug() << "Picked up cout pipe of " << m_path << ": " << pipe_cout[0] << std::endl;
	} else {
		pipe_cout[0] = -1; 
	}
	if( pipe_cout[0] != -1 ) {
		::close(pipe_cerr[1]);		// 書き込み側は使わないのでクローズ
		host::display.debug() << "Picked up cerr pipe of " << m_path << ": " << pipe_cerr[0] << std::endl;
	} else {
		pipe_cerr[0] = -1;
	}

	return std::make_pair(pipe_cout[0], pipe_cerr[0]);
}



void Process::chroot_exec(const std::string& chroot_path, const std::string& cout_path, const std::string& cerr_path)
{  // XXX: コピー
	// 引数を作成 ついでにデバッグメッセージを出力
	host::display.verbose() << "Executing";	// args[0] == m_path
	std::auto_ptr<char*> args( new char*[ m_args.size() + 1 ] );
	size_t i;
	for( i=0; i < m_args.size(); i++ ) {
		args.get()[i] = m_args[i];
		host::display.verbose() << " " << m_args[i];
	}
	args.get()[i] = NULL;
	host::display.verbose() << " in chroot " << chroot_path << std::endl;

	// パイプ作成
	int pipe_cout[2];
	int pipe_cerr[2];
	if( ::pipe(pipe_cout) < 0 ) {
		// cout用のパイプの作成に失敗
		host::display.warn() << "Can't make cout pipe for " << m_path << errorDescribe() << std::endl;
		pipe_cout[0] = pipe_cout[1] = -1;	// XXX: throwする？
	}	// cout用のパイプの作成に成功

	if( ::pipe(pipe_cerr) < 0 ) {
		// cerr用のパイプの作成に失敗
		host::display.warn() << "Can't make cerr pipe for " << m_path << errorDescribe() << std::endl;
		pipe_cerr[0] = pipe_cerr[1] = -1;	// XXX: throwする？
	}	// cerr用のパイプの作成に成功

	// fork
	if( (m_pid=::fork()) == 0 ) {
		// 子プロセス
		if( pipe_cout[1] != -1 )
			::close(pipe_cout[0]);		// 読み込み側は使わないのでクローズ
		if( pipe_cerr[1] != -1 )
			::close(pipe_cerr[0]);		// 読み込み側は使わないのでクローズ
		// chroot
		if( ::chdir(chroot_path.c_str()) < 0 ) {
			std::cerr << "Can't chdir to " << chroot_path << errorDescribe() << std::endl;
			if( pipe_cout[1] != -1 ) ::close(pipe_cout[1]);
			if( pipe_cerr[1] != -1 ) ::close(pipe_cerr[1]);
			::exit(127);
		}
		if( ::chroot(".") < 0 ) {
			std::cerr << "Can't chroot to " << chroot_path << errorDescribe() << std::endl;
			if( pipe_cout[1] != -1 ) ::close(pipe_cout[1]);
			if( pipe_cerr[1] != -1 ) ::close(pipe_cerr[1]);
			::exit(127);
		}
		if( pipe_cout[1] != -1 ) {
			int cout_to = ::open(cout_path.c_str(), O_WRONLY | O_APPEND | O_CREAT);	// パイプ出力先のファイルをオープン
			if( cout_to == -1 )
				std::cerr << "Can't open " << cout_path << " for cout pipe" << errorDescribe() << std::endl;
			else
				::dup2(pipe_cout[1], cout_to);	// 子プロセスの標準出力 -> cout_path
		}
		if( pipe_cerr[1] != -1 ) {
			int cerr_to = ::open(cerr_path.c_str(), O_WRONLY | O_APPEND | O_CREAT);	// パイプ出力先のファイルをオープン
			if( cerr_to == -1 )
				std::cerr << "Can't open " << cerr_path << " for cerr pipe" << errorDescribe() << std::endl;
			else
				::dup2(pipe_cerr[1], cerr_to);	// 子プロセスの標準エラー出力 -> cerr_path
		}
		::execv(m_path, args.get());
		// exec失敗
		std::cerr << "Can't execute " << m_path << errorDescribe() << std::endl;
		if( pipe_cout[1] != -1 ) ::close(pipe_cout[1]);
		if( pipe_cout[1] != -1 ) ::close(pipe_cerr[1]);
		::exit(127);
	} else if( m_pid == -1 ) {
		// fork失敗
		if( pipe_cout[1] != -1 ) {
			::close(pipe_cout[0]);
			::close(pipe_cout[1]);
			pipe_cout[0] = pipe_cout[1] = -1;
		}
		if( pipe_cerr[1] != -1 ) {
			::close(pipe_cerr[0]);
			::close(pipe_cerr[1]);
			pipe_cerr[0] = pipe_cerr[1] = -1;
		}
		throw ExecuteFailedError( CANNOT + "fork " + m_path + errorDescribe() );
	}


	// 親プロセス
	if( pipe_cout[0] != -1 ) {
		::close(pipe_cout[1]);		// 書き込み側は使わないのでクローズ
		//host::display.debug() << "Picked up cout pipe of " << m_path << ": " << pipe_cout[0] << std::endl;
	} else {
		pipe_cout[0] = -1; 
	}
	if( pipe_cout[0] != -1 ) {
		::close(pipe_cerr[1]);		// 書き込み側は使わないのでクローズ
		//host::display.debug() << "Picked up cerr pipe of " << m_path << ": " << pipe_cerr[0] << std::endl;
	} else {
		pipe_cerr[0] = -1;
	}
}



/*
void Process::exec_r(void)
{  // XXX: コピー
	std::auto_ptr<char*> args( new char*[ m_args.size() + 1 ] );
	size_t i;
	for( i=0; i < m_args.size(); i++ ) {
		args.get()[i] = m_args[i];
	}
	args.get()[i] = NULL;

	// fork
	if( (m_pid=::fork()) == 0 ) {
		// 子プロセス
		::execv(m_path, args.get());
		// exec失敗
		// FIXME: 標準出力と標準エラー出力を開き直さないといけないのでは？
		std::cerr << "Can't execute " << m_path << errorDescribe() << std::endl;
		::exit(127);
	} else if( m_pid == -1 ) {
		// fork失敗
	}


	// 親プロセス
}
*/


int Process::wait(void)
{
	// 親プロセス
	if( m_pid < 0 ) return 127;    // 実行されていない
	if( m_waited ) {    // 既にwait済みである
		if( m_exit_status != 0 ) {
			host::display.verbose() << m_path << " returns " << m_exit_status << std::endl;
		}
		return m_exit_status;
	}

	int status_raw;
	int ret_waitpid;

	host::display.debug() << "Waiting " << m_path << std::endl;
	ret_waitpid = ::waitpid(m_pid, &status_raw, 0);
	m_exit_status = WEXITSTATUS(status_raw);
	m_waited = true;

	if( m_exit_status != 0 ) {
		host::display.verbose() << m_path << " returns " << m_exit_status << std::endl;
	}
	if( ret_waitpid == -1 ) {
		throw( CANNOT + "wait " + m_path + errorDescribe() );
	}

	return m_exit_status;
}



/*
int Process::wait_r(void)
{  // XXX: コピー
	// 親プロセス
	if( m_pid < 0 ) return 127;    // 実行されていない
	if( m_waited ) {    // 既にwait済みである
		return m_exit_status;
	}

	int status_raw;
	int ret_waitpid;

	ret_waitpid = ::waitpid(m_pid, &status_raw, 0);
	m_exit_status = WEXITSTATUS(status_raw);
	m_waited = true;

	return m_exit_status;
}
*/


bool Process::isActive(void)
{
	// TODO: Linux 2.6.9 以上であれば、waitid(P_PID, m_pid, pointer_to_siginfo, WNOHANG | WNOWAIT)で、waitさせずに判定可能

	if( m_pid < 0 ) return false;    // 実行されていない
	if( m_waited )  return false;    // 既にwaitされている

	int status_raw;
	int ret_waitpid;
	// プロセスがまだ実行中の場合は0が返ってくる
	ret_waitpid = waitpid(m_pid, &status_raw, WNOHANG);

	if( ret_waitpid == 0 ) {
		return true;
	} else {
		// waitが成功した
		m_exit_status = WEXITSTATUS(status_raw);
		m_waited = true;
		return false;
	}
}

int Process::signalTerm(void)
{
	if( m_pid < 0 ) return 0;
	return ::kill(m_pid, SIGTERM);
}


}  // namespace VIVER
