#include "constant.h"
#include "network.h"
#include "process.h"
#include "path.h"
#include "csv.h"
#include "display.h"
#include <fstream>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#include <set>

#include <algorithm>

namespace VIVER {


// コンストラクタ
NetworkInterface::NetworkInterface(unsigned short index, const std::string& name) :
	m_index(index),
	m_name(name),
	m_running_method(NONE),
	m_ps_dhcpc(NULL),
	m_ps_autoip(NULL),
	m_ps_staticip(NULL)
{}


// デストラクタ
NetworkInterface::~NetworkInterface()
{
	// FIXME: 例外を投げる可能性(bad_allocと~Process()が投げる例外)
	if( m_ps_dhcpc    ) waitDHCP();
	if( m_ps_autoip   ) waitAutoIP();
	if( m_ps_staticip ) waitStaticIP();
}


// public:
const std::string& NetworkInterface::getName(void) const
{
	return m_name;
}

std::pair<int,int> NetworkInterface::startDHCP(void)
{
	// XXX: 例外を飲み込むか？
	// zcipはインターフェースがリンクアップしていないといけないが、udhcpcが失敗したときにリンクアップされる
	if( m_running_method != NONE || m_ps_dhcpc != NULL ) return std::make_pair(-1,-1);
	m_running_method = DHCP;

	host::display.notice() << "Starting DHCP client for " << m_name << std::endl;
	m_ps_dhcpc = new Process(Component::Program::UDHCPC);
	(*m_ps_dhcpc) % "-nfq" % "-i" % m_name.c_str() % "-T" % "3" % "-t" % "5" % "-s" % Component::Resource::PATH_DHCP_SCRIPT;
	return m_ps_dhcpc->exec(true,true);
}
int NetworkInterface::waitDHCP(void)
{
	if( m_ps_dhcpc == NULL ) return -1;   // この返り値はテキトーである
	int retval = m_ps_dhcpc->wait();
	delete m_ps_dhcpc;
	m_ps_dhcpc = NULL;
	m_running_method = NONE;
	return retval;
}
inline bool NetworkInterface::isDHCPActive(void)
{
	if( m_ps_dhcpc == NULL ) return false;
	return m_ps_dhcpc->isActive();
}

std::pair<int,int> NetworkInterface::startAutoIP(void)
{
	if( m_running_method != NONE || m_ps_autoip != NULL ) return std::make_pair(-1,-1);
	m_running_method = AUTOIP;

	host::display.notice() << "Starting Auto-IP for " << m_name << std::endl;
	m_ps_autoip = new Process(Component::Program::ZCIP);
	(*m_ps_autoip) % "-f" % "-q" % "-r" % m_name.c_str() % Component::Resource::PATH_ZCIP_SCRIPT;
			// -f: foreground
			// -q: quit
	return m_ps_autoip->exec(true,true);
}
int NetworkInterface::waitAutoIP(void)
{
	if( m_ps_autoip == NULL ) return -1;   // この返り値はテキトーである
	int retval = m_ps_autoip->wait();
	delete m_ps_autoip;
	m_ps_autoip = NULL;
	m_running_method = NONE;
	return retval;
}
inline bool NetworkInterface::isAutoIPActive(void)
{
	if( m_ps_autoip == NULL ) return false;
	return m_ps_autoip->isActive();
}

std::pair<int,int> NetworkInterface::startStaticIP(const std::string& ipaddr, const std::string& netmask)
{
	if( m_running_method != NONE || m_ps_staticip != NULL ) return std::make_pair(-1,-1);
	m_running_method = STATICIP;

	host::display.notice() << "Setting " << ipaddr << " for " << m_name << std::endl; 
	m_ps_staticip = new Process(Component::Program::IFCONFIG);
	if( netmask.empty() ) {
		// netmask指定無し -> ifconfigに任せる
		(*m_ps_staticip) % m_name.c_str() % ipaddr.c_str() % "up";
	} else {
		// netmask指定あり
		(*m_ps_staticip) % m_name.c_str() % ipaddr.c_str() % "netmask" % netmask.c_str() % "up";
	}

	return m_ps_staticip->exec(true,true);
}
int NetworkInterface::waitStaticIP(void)
{
	if( m_ps_staticip == NULL ) return -1;   // この返り値はテキトーである
	int retval = m_ps_staticip->wait();
	delete m_ps_staticip;
	m_ps_staticip = NULL;
	m_running_method = NONE;
	return retval;
}
inline bool NetworkInterface::isStaticIPActive(void)
{
	if( m_ps_staticip == NULL ) return false;
	return m_ps_staticip->isActive();
}



// 定数
const std::string Network::KERNEL_MODULE("af_packet");


// コンストラクタ
Network::Network(const Parameters& _parameters) : parameters(_parameters)
{}


// デストラクタ
Network::~Network()
{}


// public:
void Network::wakeup(void)
{
	lookupInterfaces();

	// loを起動
	// /sbin/ifconfig lo 127.0.0.1 netmask 255.0.0.0 broadcast 127.255.25.255 up
	Process ifconfig_lo("/sbin/ifconfig");
	ifconfig_lo % "lo" % "127.0.0.1" % "netmask" % "255.0.0.0" % "broadcast" % "127.255.255.255" % "up";
	MultiPipe mp_lo( ifconfig_lo.exec(), host::display.debug(), host::display.debug() );
	mp_lo.display();
	ifconfig_lo.wait();

	// FIXME: インターフェースが一つもなくてもloは起動する？
	if( m_unconfigured_netif.empty() ) {
		// XXX: throwせずにreturn 0か？
		throw NoNetworkInterfaceAvailableError("Can't find any network interface");
	}

	// staticip指定分を設定
	if( parameters.getValueString("staticip", NULL) != NULL ) {
		// TODO: IPv4依存
		// XXX: ブートパラメータ通りにstaticipを設定できなかったらthorow()？
		// staticip=192.168.0.10
		// staticip=192.168.0.10/255.255.255.0
		// staticip=eth0:192.168.0.10
		// staticip=eth0:192.168.0.10/255.255.255.0
		// staticip=eth0:192.168.0.10,eth1:10.0.0.1/255.0.0.0
		typedef std::map<std::string, std::pair<std::string,std::string> > staticip_map_t;
			// key=netdev,  value=<ipaddr, netmask>
			// first=netdev,  first.first=ipaddr,  first.second=netmask
		staticip_map_t staticip_map;
		std::string global_ipaddr;
		std::string global_netmask;
		// staticip=パラメータをパース
		for( CSV specs( parameters.getValueString("staticip", NULL) );
				!specs.end();
				++specs ) {
			std::string static_netdev;
			std::string static_ipaddr;
			std::string static_netmask;
			analyzeStaticIPLine(*specs, &static_netdev, &static_ipaddr, &static_netmask);
			if( static_netdev.empty() ) {
				global_ipaddr = static_ipaddr;
				global_netmask = static_netmask;
			} else {
				staticip_map.insert(    // 重複を許さずinsert
						make_pair(
							static_netdev,
							make_pair(
								static_ipaddr,
								static_netmask
								)
							)
						);
			}
		}

		// startStaticIP & waitStaticIP
		// ifconfigはすぐに終わるので、同時一斉開始 -> 順次wait の処理は行わない
		netif_class_t staticip_failed;
		std::string unknown_netif;

		// netdev=を指定されたインターフェースの処理
		netif_map_t::iterator netif_map_ite;
		for( staticip_map_t::iterator ipinfo( staticip_map.begin() );
				ipinfo != staticip_map.end();
				++ipinfo ) {
			if( (netif_map_ite=m_netif_map.find( ipinfo->first )) == m_netif_map.end() ) {
				// 指定されたnetdevが見つからない
				// staticip_failed_netifに追加して続行し、後でthrow
				unknown_netif = unknown_netif + " " + ipinfo->first;
				continue;
			} else {
				// 指定されたnetdevを発見
				// startStaticIP & waitStaticIP
				NetworkInterface *target_netif = &(netif_map_ite->second);
				MultiPipe mp_staticip(
						target_netif->startStaticIP(ipinfo->second.first, ipinfo->second.second),
						host::display.notice(),
						host::display.warn()
						);
				mp_staticip.display();
				if( target_netif->waitStaticIP() == 0 ) {
					// StaticIP成功 -> m_unconfigured_netifから削除
					m_unconfigured_netif.remove(target_netif);
				} else {
					// StaticIP失敗 -> staticip_failed に追加
					staticip_failed.push_back(target_netif);
				}
			}
		}
		// globalに指定があった場合の処理
		if( !global_ipaddr.empty() ) {
			netif_class_t::iterator unconfigured_ite = m_unconfigured_netif.begin();
			while( unconfigured_ite != m_unconfigured_netif.end() ) {
				NetworkInterface *target_netif = *unconfigured_ite;
				if( std::find( staticip_failed.begin(), staticip_failed.end(), target_netif) != staticip_failed.end() ) {
					// 既に netdevを指定されたインターフェースの処理 で失敗済み
					++unconfigured_ite;
					continue;
				}
				MultiPipe mp_staticip(
						target_netif->startStaticIP(global_ipaddr, global_netmask),
						host::display.notice(),
						host::display.warn()
						);
				mp_staticip.display();
				if( target_netif->waitStaticIP() == 0 ) {
					// StaticIP成功 -> m_unconfigured_netifから削除
					unconfigured_ite = m_unconfigured_netif.erase( unconfigured_ite );
				} else {
					// StaticIP失敗 -> staticip_failedに追加
					staticip_failed.push_back(target_netif);
					++unconfigured_ite;
				}
			}
		}

		// unknonwn_netif, staticip_failedについてthrow
		if( !unknown_netif.empty() || !staticip_failed.empty() ) {
			std::string msg;
			if( !unknown_netif.empty() )
				msg = msg + "No such network interface:" + unknown_netif;
			if( !staticip_failed.empty() ) {
				msg = msg + (msg.empty() ? "" : "\n") + "Can't configure network interfaces:";
				for( netif_class_t::iterator fail( staticip_failed.begin() );
						fail != staticip_failed.end();
						++fail ) {
					msg = msg + " " + (*fail)->getName();
				}
			}
			throw NetworkInterfaceConfigurationFailedError( msg );
		}


	}

	if( m_unconfigured_netif.empty() ) goto all_interface_configured;



	{	// staticip=パラメータで指定されなかったインターフェースについて設定
		MultiPipe mp_dhcpc;
		MultiPipe mp_autoip;
		MultiPipe mp_staticip;
		netif_class_t trying_dhcpc;
		netif_class_t trying_autoip;
		netif_class_t trying_staticip;

		// DHCP -> Zeroconf Auto-IP -> 169.254.0.1 -> throw
		// 全インターフェースについて startDHCP
		for( netif_class_t::iterator nif( m_unconfigured_netif.begin() );
				nif != m_unconfigured_netif.end();
				++nif ) {
			mp_dhcpc.addPipe(
					(*nif)->startDHCP(),
					host::display.notice(),
					host::display.warn()
					);
			trying_dhcpc.push_back(*nif);
		}

		sleep(1);    // DHCPクライアント実行中 // TODO: 長すぎ？
		unsigned int rest_ps;

		// startDHCP -> waitDHCP (失敗したら -> startAutoIP)
		do {
			rest_ps = mp_dhcpc.display(true);
			// パイプが一つでもEOFに達したら(プログラムが終了したら)returnされる
			// 本当に終了したかどうかは Process::isActive() で判定
			// MultiPipe::display(true) と Process::isActive() の間でプロセスが終了する可能性もある
			//	(その場合でも trying_dhcpc.empty() == true になるだけ)
			netif_class_t::iterator dhif( trying_dhcpc.begin() );
			while( dhif != trying_dhcpc.end() ) {
				if( (!(*dhif)->isDHCPActive()) || rest_ps == 0 ) {
					// DHCPクライアントが終了したインターフェースについて、
					if( (*dhif)->waitDHCP() == 0 ) {
						// DHCP成功 -> m_unconfigured_neitfから削除
						m_unconfigured_netif.remove(*dhif);
					} else {
						// DHCP失敗 -> trying_autoipに追加, AutoIPを開始
						trying_autoip.push_back(*dhif);
						mp_autoip.addPipe(
								(*dhif)->startAutoIP(),
								host::display.notice(),
								host::display.warn()
								);
					}
					dhif = trying_dhcpc.erase(dhif);
				} else {
					// このインターフェースのDHCPクライアントは終了していない
					++dhif;
				}
			}
		} while( !trying_dhcpc.empty() );

		if( m_unconfigured_netif.empty() ) goto all_interface_configured;

		// stopAutoIP (失敗したら -> startStaticIP)
		do {
			rest_ps = mp_autoip.display(true);
			netif_class_t::iterator aiif( trying_autoip.begin() );
			while( aiif != trying_autoip.end() ) {
				if( ! (*aiif)->isAutoIPActive() || rest_ps == 0 ) {
					// AutoIPが終了したインターフェースについて
					if( (*aiif)->waitAutoIP() == 0 ) {
						// AutoIP成功 -> m_unconfigured_netifから削除
						m_unconfigured_netif.remove(*aiif);
					} else {
						// AutoIP失敗 -> trying_staticipに追加, StaticIPを開始
						trying_staticip.push_back(*aiif);
						mp_staticip.addPipe(
								(*aiif)->startStaticIP("169.254.0.1", "255.255.0.0"),
								host::display.notice(),
								host::display.warn()
								);
					}
					aiif = trying_autoip.erase(aiif);
				} else {
					// このインターフェースのAutoIPは終了していない
					++aiif;
				}
			}
		} while( !trying_autoip.empty() );

		if( m_unconfigured_netif.empty() ) goto all_interface_configured;

		// stopStaticIP (失敗したら -> throw)
		mp_staticip.display();     // trueは渡さず、すべて一度にwait
		for( netif_class_t::iterator stif( trying_staticip.begin() );
				stif != trying_staticip.end();
				++stif ) {
			if( (*stif)->waitStaticIP() == 0 ) {
				// StaticIP成功 -> m_unconfigured_netifから削除
				m_unconfigured_netif.remove(*stif);
			} else {
				// StaticIP失敗 (放置, m_unconfigured_netifに残す)
			}
		}
		trying_staticip.clear();
		if( !m_unconfigured_netif.empty() ) {
			std::string msg("Can't configure following network interfaces:");
			for( netif_class_t::const_iterator fail( m_unconfigured_netif.begin() );
					fail != m_unconfigured_netif.end();
					++fail ) {
				msg = msg + " " + (*fail)->getName();
			}
			throw NetworkInterfaceConfigurationFailedError( msg );    // XXX: throwする？放っておく？
		}
	}

all_interface_configured:
	// XXX: ここで設定されたIPアドレスやサブネットマスクをdisplay.notice()で表示する？
	return;
}


// private:
void Network::lookupInterfaces(void)
{
	DIR *netif_list_dir;
	struct dirent *netif_dirent;

	netif_list_dir = ::opendir( Component::Kernel::PATH_SYS_NET_DIR );
	if( netif_list_dir == NULL ) {
		throw NoSuchFileError(CANNOT + "open " + Component::Kernel::PATH_SYS_NET_DIR );
		// XXX: 例外を投げる？インターフェースなしとみなす？
	}

	// パラメータで起動しないように指定されたインターフェース
	std::set<std::string> black_netif;
	{
		for( CSV black( parameters.getValueString("blacknet", "") );
				!black.end();
				++black ) {
			black_netif.insert(*black);
		}
	}

	unsigned short netif_index;
	while( (netif_dirent = ::readdir(netif_list_dir)) != NULL ) {
		const char* dirname = netif_dirent->d_name;
		if( dirname[0] == '.') continue;    // "." と ".." を飛ばす
		if( black_netif.find(dirname) != black_netif.end() ) continue;

		std::ifstream netif_index_file(
				( Path(Component::Kernel::PATH_SYS_NET_DIR) + dirname + "ifindex" ).c_str()
				);
		if( !netif_index_file.is_open() ) continue;

		netif_index_file >> std::dec >> netif_index;    // FIXME: 本当に10進数か？
		if( netif_index_file.fail() ) continue;

		netif_index_file.close();

		// loは自明なのでm_netif_mapにもm_unconfigured_netifにも入れない
		// FIXME: fallback.toShellからのリトライに備え、m_netif_map/m_unconfigured_netifにも入れるべきか
		// FIXME: fallback.toShellからのデータ共有部分を
		if( ::strcmp(dirname,"lo") == 0 ) continue;

		std::pair<netif_map_t::iterator,bool> ins(
				m_netif_map.insert(
					netif_map_t::value_type(
						dirname,
						NetworkInterface(netif_index,dirname)
						)
					)
				);
		if( ins.second == true ) m_unconfigured_netif.push_back( &(ins.first->second) );

		host::display.notice() << "Found Network Interface " << netif_index << ": " << dirname << std::endl;
	}

	closedir( netif_list_dir );
}

void Network::analyzeStaticIPLine(const char* line, std::string* return_netdev, std::string* return_ipaddr, std::string* return_netmask) const
{
	// return_*には必ず何か代入する

	std::string progress(line);
	std::string::size_type pos_colon;
	std::string::size_type pos_slash;

	if( (pos_colon=progress.find(':')) != std::string::npos ) {
		*return_netdev = progress.substr(0, pos_colon);
		progress.erase(0, pos_colon+1);
	} else {
		return_netdev->erase();
	}

	if( (pos_slash=progress.find('/')) != std::string::npos ) {
		*return_ipaddr= progress.substr(0, pos_slash);
		progress.erase(0, pos_slash+1);
	} else {
		*return_ipaddr = progress;
		progress.erase();
	}

	*return_netmask = progress;
}


}  // namespace VIVER

