#include "linearizer.h"
#include "display.h"
#include "constant.h"
#include "process.h"
#include <sstream>

namespace VIVER {


// 定数
const char* Linearizer::KERNEL_MODULE("dm-mod");


// コンストラクタ
Linearizer::Linearizer(DeviceManager& _devmanager) :
	devmanager(_devmanager)
{}


// デストラクタ
Linearizer::~Linearizer()
{}


// public:
void Linearizer::addDevice(const DeviceNode* dn)
{
	dn_list.push_back(dn);
}

DeviceNode* Linearizer::linearize(const std::string& fstype, unsigned long mount_flags, const std::string& data_string)
{
	host::display.notice() << "Linearizing sparsed image" << std::endl;

	// テーブルファイルを作成
	{
		// 外部プログラム
		Process ps_tabling(Component::Program::TABLEFILING);
		ps_tabling % (Component::Resource::TEMPORARY_DIR + "/table").c_str();
		for( std::vector<const DeviceNode*>::const_iterator dn( dn_list.begin() );
				dn != dn_list.end();
				++dn ) {
			ps_tabling % (*dn)->getNodePath().c_str();
		}
		MultiPipe mp_tabling( ps_tabling.exec(true,true), host::display.debug(), host::display.verbose() );
		mp_tabling.display();
		ps_tabling.wait();
	}


	// DeviceMapperを実行
	// /dev/mapper/controlはあらかじめ作成しておく
	// XXX: "sparsed_fixed"はパラメータで指定できるようにする？
	{
		Process ps_dmsetup(Component::Program::DMSETUP);
		ps_dmsetup % "create"
			   % "sparsed_fixed"
			   % (Component::Resource::TEMPORARY_DIR + "/table").c_str();
		MultiPipe mp_dmsetup( ps_dmsetup.exec(true,true), host::display.debug(), host::display.warn() );
		mp_dmsetup.display();
		if( ps_dmsetup.wait() != 0 ) {
			throw SparseFailedError("Can't connect SparseImage to FixedImage");
		}
	}

	// DeviceMapperの結果を取得
	struct stat st_buf;
	if( ::stat("/dev/mapper/sparsed_fixed", &st_buf) < 0 ) {
		throw SparseFailedError(
				std::string("/dev/mapper/sparsed_fixed doesn't exist") + errorDescribe()
				);
	}
	if( ! S_ISBLK(st_buf.st_mode) ) {
		throw SparseFailedError(
			std::string("/dev/mapper/sparsed_fixed is not block device")
			);
	}
	FileSystem::major_t major = (st_buf.st_rdev >> 8) & 0x00ff;
	FileSystem::minor_t minor = (st_buf.st_rdev) & 0x00ff;


	// デバイスノードを作成
	std::ostringstream name_stream;
	name_stream << "dm-" << minor;
	DeviceNode* dn_linear = devmanager.requestNode(major, minor, name_stream.str().c_str(),
			fstype, mount_flags, data_string.c_str() );
	if( dn_linear->makeNode() < 0 ) {
		throw SparseFailedError( CANNOT + "make device node "
				+ dn_linear->getNodePath().str() + errorDescribe()
			);
	}

	return dn_linear;
}


}  // namespace VIVER
