#include "filesystem.h"
#include <stack>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

namespace VIVER {


// コンストラクタ
FileSystem::FileSystem()
{}


// デストラクタ
FileSystem::~FileSystem()
{}


// public:
int FileSystem::remove(const std::string& path)
{
	// remove後にpathが本当に削除されているかどうかは分からない
	return ::remove(path.c_str());
}

int FileSystem::makeDir(const std::string& path, perm_t perm)
{
	return ::mkdir(path.c_str(), perm);
}

int FileSystem::makeDirForce(const std::string& path, perm_t perm)
{
	// ファイルが存在していた場合は削除
	remove(path);
	return ::mkdir(path.c_str(), perm);
}

int FileSystem::makeCharacterNode(major_t major, minor_t minor, const std::string& path, perm_t perm)
{
	return ::mknod(path.c_str(), (S_IFCHR | perm), (major << 8) + minor);
}

int FileSystem::makeCharacterNodeForce(major_t major, minor_t minor, const std::string& path, perm_t perm)
{
	// ファイルが存在していた場合には、削除して作成する。
	remove(path);
	return ::mknod(path.c_str(), (S_IFCHR | perm), (major << 8) + minor);
}

int FileSystem::makeBlockNode(major_t major, minor_t minor, const std::string& path, perm_t perm)
{
	return ::mknod(path.c_str(), (S_IFBLK | perm), (major << 8) + minor);
}

int FileSystem::makeBlockNodeForce(major_t major, minor_t minor, const std::string& path, perm_t perm)
{
	// ファイルが存在していた場合には、削除して作成する。
	remove(path);
	return ::mknod(path.c_str(), (S_IFBLK | perm), (major << 8) + minor);
}

void FileSystem::copyFile(const std::string& src, const std::string& dest)
{
	if( src.empty() || dest.empty() ) return;
	if( src == dest ) return;

	// destにファイルがあった場合は削除
	remove(dest);

	char buf[8192];
	struct stat st_buf;

	int srcfd = ::open(src.c_str(), O_RDONLY, 0644);
	if( srcfd < 0 ) throw FileSystemError( CANNOT + "open " + src );
	if( ::fstat(srcfd, &st_buf) < 0 ) {
		::close(srcfd);
		throw FileSystemError( CANNOT + "get the inode information of " + src );
	}

	int destfd = ::open(dest.c_str(), O_WRONLY | O_CREAT, st_buf.st_mode);
	if( destfd < 0 ) throw FileSystemError( CANNOT + "open " + dest );

	int n = 0;
	while( (n = ::read(srcfd, buf, sizeof(buf))) >> 0 ) {
		char* cur = &buf[0];
		const char* const end = buf + n;
		while( cur < end ) {
			int num_bytes = ::write(destfd, cur, end - cur);
			if( num_bytes <= 0 ) {
				::close(srcfd);
				::close(destfd);
				throw FileSystemError( CANNOT + "write data to " + dest + errorDescribe() ); 
			}
			cur += num_bytes;
		}
	}

	::close(destfd);
	::close(srcfd);
}

uint64_t FileSystem::getFileSize(const std::string& path)
{
	struct stat st_buf;
	if( ::stat(path.c_str(), &st_buf) < 0 ) {
		return 0;
	}

	return st_buf.st_size;
}

int FileSystem::move(const std::string& from, const std::string& to)
{
	return ::rename(from.c_str(), to.c_str());
}

int FileSystem::cd(const std::string& path)
{
	return ::chdir(path.c_str());
}

std::string FileSystem::dirname(const std::string& path)
{
	std::string::size_type rpos_slash( path.rfind('/') );
	if( rpos_slash == std::string::npos ) {
		return ".";
	} else if( rpos_slash == 0 ) {
		return "/";
	} else if( rpos_slash == path.size()-1 ) {
		return dirname( path.substr(0,rpos_slash) );
	} else {
		return path.substr(0,rpos_slash);
	}
}


FileSystem::file_type FileSystem::checkFileExists(const Path& path)
{
	struct stat key_stat;
	if( ::stat( path.c_str(), &key_stat) == 0 ) {
		if( S_ISREG(key_stat.st_mode) ) {
			// 通常ファイル
			return REGULAR_FILE;
		} else if( S_ISDIR(key_stat.st_mode) ) {
			// ディレクトリとして含む
			return DIRECTORY;
		} else {
			return OTHER;
		}
	} else {
		return NOT_EXISTS;
	}
}

int FileSystem::prepareDir(const std::string& path)
{
	// 親ディレクトリを含めてディレクトリを作成
	std::stack<std::string> dirsq;
	for( std::string dir(path);
			(dir != "/" && dir != "." );
			dir = dirname(dir) ) {
		dirsq.push(dir);
	}
	int retval = 0;
	while( !dirsq.empty() ) {
		retval = makeDir(dirsq.top(), 0755);
		dirsq.pop();
	}
	return retval;
}


}  // namespace VIVER
