#include "devicenode_candidate.h"
#include "display.h"
#include "filesystem.h"
#include "constant.h"
#include <sstream>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mount.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

namespace VIVER {


// 定数
const unsigned long DeviceNodeCandidate::MOUNT_TEST_FLAGS = MS_RDONLY;
const std::string DeviceNodeCandidate::MOUNT_TEST_DATA_STRING("");


// コンストラクタ
DeviceNodeCandidate::DeviceNodeCandidate(FileSystem::major_t major, FileSystem::minor_t minor,
		const Path& dev_path, const std::string& node_name,
		media_type_t media) :
	m_major(major),
	m_minor(minor),
	m_dev_path(dev_path),
	m_node_name(node_name),
	m_media(media),
	m_threading(false),
	m_return_value(MOUNT_FAILED)
{
	// デバイスノードを作成
	// XXX: ノード作成に失敗したときは？
	FileSystem::makeBlockNodeForce(major, minor, getNodePath().str(), 0600);

	switch(media) {
	case FIXED_HDD:
		m_untested_fstype.push_back("ntfs");
		m_untested_fstype.push_back("vfat");
		m_untested_fstype.push_back("ext3");
		m_untested_fstype.push_back("reiserfs");
		m_untested_fstype.push_back("xfs");
		m_untested_fstype.push_back("hfsplus");
		m_untested_fstype.push_back("ext2");
		m_untested_fstype.push_back("jfs");
		m_untested_fstype.push_back("ufs");
		m_untested_fstype.push_back("udf");
		m_untested_fstype.push_back("iso9660");
		m_untested_fstype.push_back("hfs");
		break;
	case REMOBAVLE_HDD:
		m_untested_fstype.push_back("vfat");
		m_untested_fstype.push_back("ntfs");
		m_untested_fstype.push_back("hfsplus");
		m_untested_fstype.push_back("ext3");
		m_untested_fstype.push_back("reiserfs");
		m_untested_fstype.push_back("xfs");
		m_untested_fstype.push_back("ext2");
		m_untested_fstype.push_back("jfs");
		m_untested_fstype.push_back("ufs");
		m_untested_fstype.push_back("udf");
		m_untested_fstype.push_back("iso9660");
		m_untested_fstype.push_back("hfs");
		break;
	case CD:
	case DVD:
		m_untested_fstype.push_back("iso9660");
		m_untested_fstype.push_back("udf");
		m_untested_fstype.push_back("hfsplus");
		m_untested_fstype.push_back("ext2");
		m_untested_fstype.push_back("reiserefs");
		m_untested_fstype.push_back("vfat");
		m_untested_fstype.push_back("ntfs");
		m_untested_fstype.push_back("hfs");
		m_untested_fstype.push_back("xfs");
		m_untested_fstype.push_back("ufs");
		m_untested_fstype.push_back("ext3");
		m_untested_fstype.push_back("jfs");
		break;
	}
}


// デストラクタ
DeviceNodeCandidate::~DeviceNodeCandidate()
{}


// public
void DeviceNodeCandidate::startTestMount(void)
{
	if( !m_threading ) {
		if( m_untested_fstype.empty() ) {
			m_return_value = ALL_TESTED;
		} else {
			host::display.debug() << "Mount Test " << getNodePath().str()
					       << " (" << m_untested_fstype.front() << ")"
					       << std::endl;
			m_threading = true;
			::pthread_create(&m_test_thread, NULL, DeviceNodeCandidate::tc_testMount, static_cast<void*>(this));
		}
	} else {
		host::display.warn() << "Get Mount Test order but still threading" << std::endl;
	}
}

void* DeviceNodeCandidate::tc_testMount(void* target)
{
	// FIXME: ロックしていないので、同じオブジェクトのtc_testMount()が同時に2スレッド以上実行されてはいけない
	DeviceNodeCandidate* self( static_cast<DeviceNodeCandidate*>(target) );

	if( self->m_test_point.empty() ) {
		Path point(Component::Resource::MOUNT_TRIAL_POINT);
		std::ostringstream dir_name;
		dir_name << self->m_major << "-" << self->m_minor;
		point += dir_name.str();
		// Display::*().operator<<() はスレッドセーフでは無い
		// host::display.debug() << "Making mount trial point " << point.str() << std::endl;
		if( FileSystem::makeDirForce(point.str(), 0755) < 0 ) {
			self->m_return_value = DIRECTORY_ERROR;
			return NULL;
		}
		self->m_test_point = point;
	}

	const std::string test_fstype( self->m_untested_fstype.front() );

	int retval = ::mount(self->getNodePath().c_str(),
			self->m_test_point.c_str(),
			test_fstype.c_str(),
			MOUNT_TEST_FLAGS,
			( MOUNT_TEST_DATA_STRING.empty() ? NULL : MOUNT_TEST_DATA_STRING.c_str() )
			);

	self->m_untested_fstype.pop_front();

	if( retval == 0 ) {
		self->m_return_fstype = test_fstype;
		self->m_return_mount_flags = MOUNT_TEST_FLAGS;
		self->m_return_data_string = MOUNT_TEST_DATA_STRING;
		self->m_return_value = MOUNT_SUCCESS;
		return NULL;
	} else {
		self->m_return_value = MOUNT_FAILED;
		return NULL;
	}
}

DeviceNodeCandidate::test_stat_t DeviceNodeCandidate::waitTestMount(std::string* return_fstype,
		unsigned long* return_mount_flags, std::string* return_data_string)
{
	if( m_threading ) {
		::pthread_join(m_test_thread, NULL);
		m_threading = false;
		if( m_return_value == MOUNT_SUCCESS ) {
			*return_fstype = m_return_fstype;
			*return_mount_flags = m_return_mount_flags;
			*return_data_string = m_return_data_string;
			return m_return_value;
		} else {
			return m_return_value;
		}
	} else {
		return m_return_value;
	}
}


}  // namespace VIVER
