#include "devicenode.h"
#include "exception.h"
#include <fcntl.h>

namespace VIVER {


// コンストラクタ
DeviceNode::DeviceNode(FileSystem::major_t major, FileSystem::minor_t minor,
		const Path& dev_path, const std::string& node_name,
		const std::string& fstype, unsigned long mount_flags, const std::string& data_string) :
	Mountable(fstype, mount_flags, data_string),
	m_major(major),
	m_minor(minor),
	m_dev_path(dev_path),
	m_node_name(node_name)
{}

DeviceNode::DeviceNode(FileSystem::major_t major, FileSystem::minor_t minor,
		const Path& dev_path, const std::string& node_name,
		const std::string& fstype, unsigned long mount_flags, const std::string& data_string,
		const Path& mount_point) :
	Mountable(fstype, mount_flags, data_string, mount_point),
	m_major(major),
	m_minor(minor),
	m_dev_path(dev_path),
	m_node_name(node_name)
{}


// デストラクタ
DeviceNode::~DeviceNode()
{}


// public:
void DeviceNode::mount(const Path& mount_point)
{
	if( m_mounted ) {
		// 既にマウントされているので move
		return move(mount_point);
	}

	Mountable::mount(getNodePath().str(), mount_point.str(), m_fstype,
			m_mount_flags, m_data_string);
	m_mounted = true;
	m_mount_point = mount_point;
}

void DeviceNode::umount(void)
{
	// マウントされていなくても続行。Mountable::umount()の中で例外が発生する
	Mountable::umount( m_mount_point.str() );
	m_mounted = false;
}

void DeviceNode::move(const Path& mount_point)
{
	// マウントされていなくても続行
	Mountable::move(m_mount_point.str(), mount_point.str());
	m_mount_point = mount_point;
}

void DeviceNode::remount(unsigned long mount_flags, const std::string& data_string)
{
	// マウントされていなくても続行
	Mountable::remount(getNodePath().str(), m_mount_point.str(), mount_flags, data_string);
	m_mount_flags = mount_flags;
	m_data_string = data_string;
}

void DeviceNode::remount(const std::string& hand_options)
{
	// マウントされていなくても続行
	std::pair<unsigned long, std::string> opt( Mountable::splitHandOptions(hand_options) );
	Mountable::remount(getNodePath().str(), m_mount_point.str(), opt.first, opt.second);
	m_mount_flags = opt.first;
	m_data_string = opt.second;
}


// public:
int DeviceNode::makeNode(void)
{
	return FileSystem::makeBlockNodeForce(m_major, m_minor, getNodePath().str(), 0660);
}

int DeviceNode::makeNodeOn(const Path& dev_path)
{
	return FileSystem::makeBlockNodeForce(m_major, m_minor, getNodePathOn(dev_path).str(), 0660);
}

uint64_t DeviceNode::getSize(void)
{
	uint64_t size;
	int fd;
	if( (fd = ::open(getNodePath().c_str(), O_RDONLY | O_NONBLOCK)) < 0 ) {
		return 0;
	}
	if( ::ioctl(fd, BLKGETSIZE64, &size) < 0 ) {
		::close(fd);
		return 0;
	}
	::close(fd);
	return size;
}


}  // namespace VIVER
