#include "vtable.h"
#include "stream/stream.h"
#include "rpc/rpcclient.h"
#include "rw_lock.h"
#include "stdext_hash_map.h"
#include "node_hash.h"
#include "ip46.h"
#include "log.h"
#include "tools.h"
#include <algorithm>
#include <iterator>

namespace VFIELD {


////
// VTableMatch
VTableMatch::VTableMatch(const std::pair<DataRange, NodeIdentity>& inserter) :
	m_range(inserter.first), m_node(inserter.second) {}

void VTableMatch::operator= (const std::pair<DataRange, NodeIdentity>& inserter)
{
	m_range = inserter.first;
	m_node  = inserter.second;
}

VTableMatch::~VTableMatch() {}


////
// VTableIMPL
//
class VTableIMPL : private boost::noncopyable {
public:
	explicit VTableIMPL(RPCClient& rpcc);
	~VTableIMPL();
public:
	typedef VTable::matches_type matches_type;
	typedef VTable::duplex_type  duplex_type;
	inline void find(const DataRange& range, matches_type& result_nodes) const;
	inline void calcDuplex(pos_type image_size, duplex_type& result) const;
public:
	inline void strippedNodeDownDetect(const StrippedNodeIdentity& down_node);
public:
	inline void rpcNotifyUp(const NodeIdentity& up_node, const DataRange& up_range);
	inline void rpcNotifyDown(const NodeIdentity& down_node, StreamManager& smgr);
	inline void rpcPing(const NodeIdentity& up_node);
public:
	void setImageSize(pos_type size);
	void streamJoin(AutoSock& asock, uint16_t image_id, uint64_t image_size);
public:
	// std::multimapのイテレータは、insetやeraseで無効化されない
	typedef std::multimap<DataRange, NodeIdentity, CompareStartLess> table_type;
	typedef stdext::hash_map<NodeIdentity, table_type::iterator, NodeIdentityHash> exist_type;
private:
	mutable rw_mutex m_table_mutex;
	table_type m_table;
	boost::mutex m_exist_mutex;
	exist_type m_exist;

	pos_type m_image_size;		// m_image_sizeを超過したrpcNotifyUpは無視する

	RPCClient& m_rpcc;
private:
	VTableIMPL();
};


VTable::VTable(RPCClient& rpcc) : impl( new VTableIMPL(rpcc) ) {}
VTable::~VTable() {}
VTableIMPL::VTableIMPL(RPCClient& rpcc) :
	m_image_size(0),	// 0は確かに不正を示す
	m_rpcc(rpcc)  {}

VTableIMPL::~VTableIMPL()
{
	// VTableがデストラクトされるところでRPCNotifySelfDownをブロードキャストする
	// XXX: 欠損に備えて2回送る？
	try {
		m_rpcc.bcastNotifySelfDown();
	} catch (...) {
		LogError("Can't broadcast RPC Notify Self Down");
	}
}


namespace {
struct CopyIfRangeContains {
	CopyIfRangeContains(const DataRange& range, VTableIMPL::matches_type& target) :
		m_range(range), m_target(target) {}
	void operator() (const std::pair<DataRange, NodeIdentity>& pair)
	{
		//if( m_range.contains_part(pair.first) ) {
		if( pair.first.contains_part(m_range) ) {  // DataRange::contains_part()の実装的にこちらの方が早い
			// 逆順ソートはダメ SearchEngineがstart順ソートを前提に動いている
			m_target.push_back(pair);
			//m_target.push_front(pair);
		}
	}
private:
	const DataRange& m_range;
	VTableIMPL::matches_type& m_target;
	CopyIfRangeContains();
};
}  // noname namespace
void VTable::find(const DataRange& range, matches_type& result_nodes) const { return impl->find(range, result_nodes); }
void VTableIMPL::find(const DataRange& range, matches_type& result_nodes) const
{
	// m_tableをreadロック
	scoped_read_lock lk_table_r(m_table_mutex);

	// m_tableはCompareStartLessIfSameEndLess
	// 部分一致を検索する

	// SearchEngineはresult_nodesをbegin()から走査していく
	// CompareStartLessだと、SGI STLの場合、最初に起動したノードが常にm_tableの先頭にきてしまう

	// XXX: 0 - * のデータ範囲しか分散されない
	//      基本的には持っているデータ範囲が広いノードの方が基本性能が高いと思われる
	// テーブルはStartLessIfEndLessでソートしておき、start位置が同じなら
	// より持っているデータ範囲が狭いノードを使うようにする
	// これでデータ全体を持っている最初のノードにアクセスが集中しないようにする
	// （ただし２番目のノードにアクセスが集中する）

	// m_tableはCompareStartLessでソートされている
	// XXX: 後ろは2分検索、前は線形検索 エントリが多くなると遅くなる
	table_type::const_iterator match_upper(
			m_table.upper_bound( DataRange(range.end(),  range.end())   )
			);
	std::for_each(m_table.begin(), match_upper, CopyIfRangeContains(range, result_nodes));

	// m_tableをアンロック
}



DuplexRange::DuplexRange(pos_type start, pos_type end, duplex_type duplex) :
	DataRange(start, end), m_duplex(duplex) {}
DuplexRange::~DuplexRange() throw() {}

class OneSideRange {
public:
	OneSideRange(pos_type pos, bool is_start) :
		m_is_start(is_start), m_pos(pos) {}
	~OneSideRange() {}
public:
	bool is_start(void) const { return m_is_start; }
	pos_type pos(void)  const { return m_pos; }
public:
	bool operator< (const OneSideRange& rhl) const
	{
		// startの方がendより大きい
		if( m_pos != rhl.m_pos ) {
			return m_pos < rhl.m_pos;
		} else {
			if( m_is_start ) {
				return !rhl.m_is_start;
			} else {
				return false;
			}
		}
	}
friend std::ostream& operator<< (std::ostream& stream, const OneSideRange& side)
	{ return stream << (side.m_is_start ? "start: " : "  end: ") << side.m_pos; }
private:
	bool m_is_start;
	pos_type m_pos;
private:
	OneSideRange();
};

class SplitRange {
public:
	SplitRange(std::vector<OneSideRange>& result) : m_result(result) {}
	void operator() (const std::pair<DataRange, NodeIdentity>& entry)
	{
		m_result.push_back( OneSideRange(entry.first.start(), true) );
		m_result.push_back( OneSideRange(entry.first.end(),  false) );
	}
private:
	std::vector<OneSideRange>& m_result;
};

void VTable::calcDuplex(pos_type image_size, duplex_type& result) const { return impl->calcDuplex(image_size, result); }
void VTableIMPL::calcDuplex(pos_type image_size, duplex_type& result) const
{

	std::vector<OneSideRange> sides;

	{  // m_tableをreadロック
		scoped_read_lock lk_table_r(m_table_mutex);
		std::for_each(m_table.begin(), m_table.end(), SplitRange(sides));
	}  // m_tableをアンロック
	std::sort(sides.begin(), sides.end());

	pos_type cur = 0;
	DuplexRange::duplex_type duplex = 0;
	for(std::vector<OneSideRange>::const_iterator sd(sides.begin()), sd_end(sides.end());
			sd != sd_end;
			++sd ) {
		if( sd->is_start() ) {
			if( cur == sd->pos() ) {
				++duplex;
				continue;
			}
			result.push_back( DuplexRange(cur, sd->pos()-1, duplex) );
			cur = sd->pos();
			++duplex;
		} else {
			if( cur == sd->pos() + 1 ) {
				--duplex;
				continue;
			}
			result.push_back( DuplexRange(cur, sd->pos(), duplex) );
			cur = sd->pos() + 1;
			--duplex;
		}
	}

	// 端を処理
	if( cur <= (image_size - 1) ) {
		result.push_back( DuplexRange(cur, image_size - 1, 0) );
	}
}


void VTable::rpcNotifyUp(const NodeIdentity& node, const DataRange& range) { return impl->rpcNotifyUp(node, range); }
void VTableIMPL::rpcNotifyUp(const NodeIdentity& node, const DataRange& range)
{
	// m_image_sizeの範囲内のrangeかチェック
	if( m_image_size <= range.end() ) {
		// 無視する
		LogDebugError( Log::format("Received RPC Notify Up range exceeds image size: %1%") % range );
		return;
	}

	// m_existの中にあるか確認
	// m_existをロック
	boost::mutex::scoped_lock lk_exist(m_exist_mutex);

	exist_type::iterator found( m_exist.find(node) );
	if( found == m_exist.end() ) {
		// 登録されていない
		// m_tableに追加する
		scoped_write_lock lk_table_w(m_table_mutex);		// m_tableをwriteロック
		table_type::iterator ins( m_table.insert(table_type::value_type(range, node)) );
		lk_table_w.unlock();					// m_tableをアンロック

		m_exist.insert( std::make_pair(node, ins) );	// m_eixtsに追加

		LogDebug0( Log::format("Adding new entry (%1%: %2%), size of table: %3%") % addr46_t(node) % range % m_table.size() );

	} else if( found->second->first != range ) {
		// 登録されているが内容が違う
		// m_tableを更新する
		scoped_write_lock lk_table_w(m_table_mutex);		// m_tableをwriteロック
		m_table.erase(found->second);				// 更新するのはKeyなので、一度削除しないと更新できない
		table_type::iterator ins( m_table.insert(table_type::value_type(range, node)) );
		lk_table_w.unlock();					// m_tableをアンロック

		m_exist.erase(found);				// m_existから古い情報を削除
		m_exist.insert( std::make_pair(node, ins) );	// m_eixtsに追加

		LogDebug0( Log::format("Renew entry (%1%: %2%), size of table: %3%") % addr46_t(node) % range % m_table.size() );
	}
	// m_existをアンロック
}


void VTable::rpcNotifyDown(const NodeIdentity& node, StreamManager& smgr) { return impl->rpcNotifyDown(node, smgr); }
void VTableIMPL::rpcNotifyDown(const NodeIdentity& node, StreamManager& smgr)
{
	// Notify Downを受け取っても、接続しているノードであれば無視する
	if( smgr.isConnected(node) ) {
		LogDebugError( Log::format("Notified down node is connected node (%1%), ignoring")
				% addr46_t(node) );
		return;
	}

	// nodeが自分の場合、間違いなので、RPC Notify Self Upをブロードキャストする
	// （終了時には自分でbcastNotifySelfDownを送る前にRPCServerがデストラクトされてこの関数自体が呼ばれない）
	if( m_rpcc.getSelfIdentity() == node ) {
		LogDebugError("Notified down node is me, aliving. Broadcasting RPC Notify Self Up");
		m_rpcc.bcastNotifySelfUp();
		return;
	}

	// まずm_existの中にあるか確認
	// m_existをロック
	boost::mutex::scoped_lock lk_exist(m_exist_mutex);

	exist_type::iterator found( m_exist.find(node) );
	if( found != m_exist.end() ) {
		// 登録されている
		// m_tableから削除
		scoped_write_lock lk_table_w(m_table_mutex);		// m_tableをwriteロック
		m_table.erase(found->second);
		lk_table_w.unlock();					// m_tableをアンロック
		m_exist.erase(found);				// m_eixtsから削除

		LogDebug( Log::format("Remove entry (%1%), size of table: %2%") % addr46_t(node) % m_table.size() );
	}
	// m_existをアンロック
}


void VTable::rpcPing(const NodeIdentity& up_node) { return impl->rpcPing(up_node); }
void VTableIMPL::rpcPing(const NodeIdentity& up_node)
{
	m_rpcc.ucastNotifySelfUp(up_node);
}


void VTable::strippedNodeDownDetect(const StrippedNodeIdentity& down_node)
	{ return impl->strippedNodeDownDetect(down_node); }
void VTableIMPL::strippedNodeDownDetect(const StrippedNodeIdentity& down_node)
{
	std::vector<NodeIdentity> full_nodes;

	{
		// m_existをロック
		boost::mutex::scoped_lock lk_exist(m_exist_mutex);
		// m_tableをwriteロック
		scoped_write_lock lk_table_w(m_table_mutex);

		for(exist_type::iterator ex(m_exist.begin()), ex_end(m_exist.end()); ex != ex_end;  ) {
			if( down_node == ex->first ) {
				LogDebug( Log::format("Remove entry (%1%), sizeof table: %2%") % addr46_t(ex->first) % (m_table.size() - 1) );
				full_nodes.push_back(ex->first);
				// 後でRCPNotifyDownをブロードキャストする（自分にも届く）ので
				// ここで削除する必要は無いが、ここで削除した方が早い
				m_table.erase(ex->second);
				m_exist.erase(ex++);
			} else {
				++ex;
			}
		}

		// m_existをアンロック
		// m_tableをアンロック
	}

	// full_nodesについて、RPCNotifyOtherDownで他のノードにも知らせる。
	// 他のノードから既に知らされていた場合はVTableから既に除かれている（=full_nodesに入らない）ので、
	// 重複してブロードキャストすることはない
	// FIXME: VTableのエントリ全体にユニキャストにする？
	for( std::vector<NodeIdentity>::const_iterator nd(full_nodes.begin()), nd_end(full_nodes.end());
			nd != nd_end;
			++nd ) {
		m_rpcc.bcastNotifyOtherDown(*nd);
	}
}


void VTable::setImageSize(pos_type size) { return impl->setImageSize(size); }
void VTableIMPL::setImageSize(pos_type size)
{
	// XXX: 既存のテーブル内のデータをチェックする？
	m_image_size = size;
}


namespace {
struct SendTable {
	SendTable(AutoSock& asock, uint32_t send_limit) :
		m_asock(asock), m_send_limit(send_limit), m_sent(0) {}
	void operator() (const std::pair<DataRange, NodeIdentity>& pair)
	{
		// FIXME: 効率が悪い
		if(m_sent > m_send_limit) return;
		char buf[RPC_NOTIFY_UP_SIZE];
		::memcpy(&buf[0],  pair.second.getNetRaw(),      NodeIdentity::raw_size);
		::memcpy(&buf[19], pair.first.getNetRaw().get(), DataRange::raw_size);
		m_asock.write( buf, sizeof(buf) );		// 例外は呼出元でキャッチ
		++m_sent;
	}
private:
	AutoSock& m_asock;
	uint32_t m_send_limit;
	uint32_t m_sent;
};
}  // noname namespace

void VTable::streamJoin(AutoSock& asock, uint16_t image_id, uint64_t image_size)
	{ return impl->streamJoin(asock, image_id, image_size); }
void VTableIMPL::streamJoin(AutoSock& asock, uint16_t image_id, uint64_t image_size)
{
	LogDebug( Log::format("Proceeding Stream Join, size of table is %1%...") % m_table.size() );

	asock.setBlock();

	char buf[STREAM_BUFFER_SIZE];

	// 最初のデータはSTREAM_JOIN_ACK付きのSTREAM_JOIN_ACK_HEADER
	buf[0] = STREAM_JOIN_ACK_MAGIC;
	uint16_t net_image_id = htons( image_id );
	uint64_t net_image_size = htonll( image_size );
	uint32_t     table_size = m_table.size();
	uint32_t net_table_size = htonl( table_size );	// XXX: table_type::size_typeよりuint32_tが小さかったら？
	::memcpy(&buf[1+0],  &net_image_id,   sizeof(net_image_id)  );
	::memcpy(&buf[1+2],  &net_image_size, sizeof(net_image_size));
	::memcpy(&buf[1+10], &net_table_size, sizeof(net_table_size));

	asock.write(buf, 1 + STREAM_JOIN_ACK_HEADER_SIZE);	// 例外は呼出元でキャッチ

	std::for_each(m_table.begin(), m_table.end(), SendTable(asock, table_size));
								// 例外は呼出元でキャッチ

	LogDebug("...stream Join processed successfully");
}


}  // namespace VFIELD
