#include "storage.h"
#include "stream/stream.h"
#include "bufferedreq.h"
#include "search/controller.h"
#include "cache/receptor.h"
#include "bufferedreq.h"
#include "request.h"
#include "log.h"
#include "vfield.h"

namespace VFIELD {


namespace {
static char* TRASH_1BYTE = "";
}

void EdgeStorageToRAM::invalidate(void)
{
	LogDebug("EdgeStorageToRAM invalidating");
	m_ram = BufferedRequest(
			TRASH_1BYTE,
			DataRange(std::numeric_limits<pos_type>::max(), std::numeric_limits<pos_type>::max())
			);
	EdgeStorage::invalidate();
}


EdgeStorageToRAM::EdgeStorageToRAM(uint64_t store_limit, RPCClient& rpcc, VTable& vtable, ThreadPool& threadpool) :
	EdgeStorage(rpcc, vtable, store_limit, threadpool),
	m_ram(TRASH_1BYTE, DataRange(std::numeric_limits<pos_type>::max(), std::numeric_limits<pos_type>::max()))
		// m_ramは常にNULLでないバッファをさしている
{
	m_image_id = 0;		// XXX: image_idは？
}

EdgeStorageToRAM::~EdgeStorageToRAM() {}

void EdgeStorageToRAM::request(	DataRequest& request,
				std::vector<DataReceptor>* result_cached,
				std::vector<BufferedRequest>*  result_uncached	)
{
	// TODO: ここの効率はもう少し良くできるか？ （↓このif分と、calcUnmatchedBefore/After）
	if( m_self_range.contains_part( request ) ) {	// 部分一致しているなら
		DataReceptor receptor(m_ram, request);

		result_cached->push_back(receptor);

		{
			std::pair<BufferedRequest, bool> out( receptor.calcUnmatchedBefore() );
			if( out.second ) {
				result_uncached->push_back(out.first);
			}
		}
		{
			std::pair<BufferedRequest, bool> out( receptor.calcUnmatchedAfter() );
			if( out.second ) {
				result_uncached->push_back(out.first);
			}
		}
	} else {
		result_uncached->push_back( BufferedRequest(request) );
	}
}

void EdgeStorageToRAM::streamGetData(AutoSock& asock, const DataRange& range)
{
	LogDebug0( Log::format("request: %1%") % range);

	if( ! m_self_range.contains_all(range) ) {
		// 要求されたデータ範囲をすべて持っていない
		LogDebugError(
				Log::format("Invalid Stream Get Data (requested data range is out of stored range %1%)")
				% range );
		asock.close();  // 閉じる
		// TODO: rpcNotifyUpを送りたいが、相手のRPCのポート番号が分からない
	}

	// ブロッキングモードにする
	asock.setBlock();

	// 最初のデータはSTREAM_GET_DATA_ACK
	char header[STREAM_BUFFER_SIZE];
	header[0] = STREAM_GET_DATA_ACK_MAGIC;
	asock.write(header, STREAM_HEADER_SIZE);		// 例外はstream.ccがキャッチ

	// XXX: ロックが必要？（RPC KEEPを自分に送るような？）
	uint64_t pos = range.start();
	uint64_t pos_end = range.end();
	do {
		pos += asock.write( m_ram.data() + (pos - m_ram.start()), pos_end - pos + 1 );
			// 例外はstream.ccがキャッチ
		LogDebug0( Log::format("RAM to Network %1% bytes left") % (pos_end - pos + 1) );
	} while(pos <= pos_end);
}

void EdgeStorageToRAM::setSelfRange(const DataRange& range, SearchController& engine)
{
	// バッファを確保
	m_buffer.reset( new char[range.length()] );
	m_ram = BufferedRequest(m_buffer.get(), range);

	// データをダウンロードしてくる
	// キャッシュを介せずに直接ダウンロードする
	engine.submitRequest(m_ram);

	if( m_ram.getClient().wait() != BufferedRequest::SUCCESS ) {
		// FIXME: 再試行する？
		throw JoinFailedException("Failed to downlaod self data range");
	}

	// FIXME: m_self_rangeはダウンロードが完了した後？
	// 完了前だと、Join中に受け取ったFindに間違った応答（Notify Self Up）を返してしまう
	m_self_range = range;
}


}  // namespace VFIELD
