#include "interface.h"
#include "log.h"
#include "storage/storage.h"
#include "search/controller.h"
#include <algorithm>
#include <functional>
#include <boost/bind.hpp>

namespace VFIELD {


class InterfaceIMPL {
public:
	InterfaceIMPL() {}
	virtual ~InterfaceIMPL() {}
public:
	virtual Interface::result_type getData(char* buf, uint64_t from, uint32_t length) = 0;
	virtual uint64_t getImageSize(void) = 0;
};

class InterfaceRootIMPL : public InterfaceIMPL {
public:
	InterfaceRootIMPL(RootStorage& root_storage);
	~InterfaceRootIMPL();
public:
	Interface::result_type getData(char* buf, uint64_t from, uint32_t length);
	uint64_t getImageSize(void);
private:
	RootStorage& m_rstorage;
};

class InterfaceEdgeIMPL : public InterfaceIMPL {
public:
	InterfaceEdgeIMPL(EdgeStorage& edge_storage, SearchController& engine);
	~InterfaceEdgeIMPL();
public:
	Interface::result_type getData(char* buf, uint64_t from, uint32_t length);
	uint64_t getImageSize(void);
private:
	EdgeStorage& m_estorage;
	SearchController& m_engine;
};


////
// Interface
//
Interface::Interface(RootStorage& root_storage) :
	impl(new InterfaceRootIMPL(root_storage)), m_end_flag(false) {}

Interface::Interface(EdgeStorage& edge_storage, SearchController& engine) :
	impl(new InterfaceEdgeIMPL(edge_storage, engine)), m_end_flag(false) {}

Interface::~Interface()
{
	m_end_flag = true;
}

Interface::result_type Interface::getData(void* buffer, uint64_t from, uint32_t length)
	{ return impl->getData((char*)buffer, from, length); }

uint64_t Interface::getImageSize(void)
{
	// 0の間は待つ
	// FIXME: スマートでない
	uint64_t size;
	if( likely((size = impl->getImageSize()) != 0) ) {
		return size;
	} else {
		while( size == 0 && !m_end_flag ) {
			sleep(1);
			size = impl->getImageSize();
		}
		return size;
	}
}


////
// InterfaceMPL
//
InterfaceRootIMPL::InterfaceRootIMPL(RootStorage& root_storage) :
	m_rstorage(root_storage) {}
InterfaceRootIMPL::~InterfaceRootIMPL() {}

InterfaceEdgeIMPL::InterfaceEdgeIMPL(EdgeStorage& edge_storage, SearchController& engine) :
	m_estorage(edge_storage), m_engine(engine) {}
InterfaceEdgeIMPL::~InterfaceEdgeIMPL() {}

Interface::result_type InterfaceRootIMPL::getData(char* buf, uint64_t from, uint32_t length)
{
	try {
		m_rstorage.request(buf, from, length);
		return Interface::SUCCESS;
	} catch( const std::runtime_error& ex ) {
		LogError( Log::format("Can't get data: %1%") % ex.what() );
		return Interface::FAILED;
	} catch(...) {
		LogError("Unknown error is occurred while download data");
		return Interface::FAILED;
	}
}

Interface::result_type InterfaceEdgeIMPL::getData(char* buf, uint64_t from, uint32_t length)
{
	// FIXME: やけに無駄が多い
	try {
		if( unlikely(length == 0) ) return Interface::SUCCESS;

		std::vector<DataReceptor> stored;
		std::vector<BufferedRequest>  unstored;

		// 最初のリクエストを作る
		DataRequest original_request(buf, from, from + length - 1);

		// EdgeStorageにリクエスト
		m_estorage.request(original_request, &stored, &unstored);

		// EdgeStorageにヒットしなかったデータをSearchControllerにリクエスト
		m_engine.submitRequests(unstored);
		LogDebug( Log::format("stored number: %1%") % stored.size() );
		LogDebug( Log::format("request number: %1%") % unstored.size() );

		// BufferedRequestをDataReceptorに変換
		std::vector<DataReceptor> unstored_receptor;
		unstored_receptor.reserve( unstored.size() );
		for(std::vector<BufferedRequest>::iterator uns(unstored.begin()), uns_end(unstored.end());
				uns != uns_end;
				++uns ) {
			unstored_receptor.push_back( DataReceptor(*uns, original_request) );
		}

		// データの到着を待つ
		// まずEdgeStorageにヒットしたデータを待つ
		std::for_each(	stored.begin(),
				stored.end(),
				std::mem_fun_ref(&DataReceptor::waitAndCopy)
			     );
		// 次にSearchControllerにリクエストしたデータを待つ
		std::for_each(	unstored_receptor.begin(),
				unstored_receptor.end(),
				std::mem_fun_ref(&DataReceptor::waitAndCopy)
			     );
		return Interface::SUCCESS;

	} catch (const DataLostException& ex) {
		LogError("Can't download data");
		return Interface::FAILED;
	} catch (const std::runtime_error& ex) {
		LogError( std::string("Can't download data: ") + ex.what() );
		return Interface::FAILED;
	} catch (...) {
		LogError("Unknown error is occurred while download data");
		return Interface::FAILED;
	}
}

uint64_t InterfaceRootIMPL::getImageSize(void)
{
	return m_rstorage.getImageSize();
}

uint64_t InterfaceEdgeIMPL::getImageSize(void)
{
	return m_estorage.getImageSize();
}


}  // namespace VFIELD
