#ifndef VFIELD_LOCKED_QUEUE_H__
#define VFIELD_LOCKED_QUEUE_H__

#include "exception.h"
#include <queue>
#include <boost/shared_ptr.hpp>
#include <boost/thread.hpp>

// XXX: 例外処理が無い

namespace VFIELD {


template <typename T>
class LockedQueue {
public:
	LockedQueue();
	~LockedQueue();
public:
	typedef T value_type;
	typedef boost::shared_ptr<T> value_ptr;
public:
	void push(const value_type& x);
	value_ptr blocked_pop(void);					// 例外が発生した場合 value_ptr == false
	value_ptr timed_pop(const boost::xtime& xt);			// タイムアウトした場合 value_ptr == false
	value_ptr timed_pop(unsigned int sec, unsigned int nsec);	// タイムアウトした場合 value_ptr == false
public:
	size_t size(void)  { return m_queue.size();  }
	bool   empty(void) { return m_queue.empty(); }
private:
	std::queue<value_ptr> m_queue;		// Valueは共有されるので、boost::ptr_queueは使えない
	boost::condition m_cond;
	boost::mutex m_mutex;
};


template <typename T>
LockedQueue<T>::LockedQueue() {}
template <typename T>
LockedQueue<T>::~LockedQueue() {}

template <typename T>
void LockedQueue<T>::push(const value_type& x)
{
	boost::mutex::scoped_lock lk(m_mutex);
	m_queue.push( value_ptr(new value_type(x)) );
	m_cond.notify_all();
}


template <typename T>
typename LockedQueue<T>::value_ptr LockedQueue<T>::blocked_pop(void)
{
	boost::mutex::scoped_lock lk(m_mutex);
	while( m_queue.empty() ) {
		m_cond.wait(lk);
	}
	value_ptr item( m_queue.front() );
	m_queue.pop();
	return item;
}

template <typename T>
typename LockedQueue<T>::value_ptr LockedQueue<T>::timed_pop(const boost::xtime& xt)
{
	boost::mutex::scoped_lock lk(m_mutex);
	if( !m_queue.empty() ) {
		value_ptr item( m_queue.front() );
		m_queue.pop();
		return item;
	}

	while( m_queue.empty() ) {
		// キューにアイテムが追加されるか、タイムアウトするまで待つ
		if( !m_cond.timed_wait(lk, xt) ) {
			// タイムアウト
			return value_ptr();
		}
	}

	value_ptr item( m_queue.front() );
	m_queue.pop();
	return item;
}


template <typename T>
typename LockedQueue<T>::value_ptr LockedQueue<T>::timed_pop(unsigned int sec, unsigned int nsec)
{
	boost::mutex::scoped_lock lk(m_mutex);
	if( !m_queue.empty() ) {
		value_ptr item( m_queue.front() );
		m_queue.pop();
		return item;
	}

	boost::xtime xt;
	boost::xtime_get(&xt, boost::TIME_UTC);
	xt.sec += sec;
	xt.nsec += nsec;

	while( m_queue.empty() ) {
		// キューにアイテムが追加されるか、タイムアウトするまで待つ
		if( !m_cond.timed_wait(lk, xt) ) {
			// タイムアウト
			return value_ptr();
		}
	}

	value_ptr item( m_queue.front() );
	m_queue.pop();
	return item;
}


}  // namespace VFIELD

#endif /* locked_queue.h */
