#ifndef VFIELD_ASYNCLOG_H__
#define VFIELD_ASYNCLOG_H__

#include "locked_queue.h"
#include "exception.h"
#include <iostream>
#include <string>
#include <boost/format.hpp>
#include <boost/thread.hpp>
#include <boost/bind.hpp>

namespace VFIELD {


////
// AsyncLog
//
class AsyncLog {
public:
	explicit AsyncLog(bool syslog = true);
	~AsyncLog();
public:
	typedef boost::format format;
	typedef  enum {
		ASLOG_DEBUG	= 0,
		ASLOG_INFO	= 1,
		ASLOG_NOTICE	= 2,
		ASLOG_WARNING	= 3,
		ASLOG_ERR	= 4,
		ASLOG_CRIT	= 5,
		ASLOG_ALERT	= 6,
		ASLOG_EMERG	= 7,
	} level_type;
	template <level_type MessageLevel>
	inline void write(const format& message) throw();
	template <level_type MessageLevel>
	inline void write(const std::string& message) throw();
private:
	void WriterThread(void);
private:
	typedef LockedQueue< std::pair<format,level_type> > queue_type;
	queue_type queue;
	bool use_syslog;
	volatile bool end_flag;
	boost::thread writer;	// スレッドは最後に初期化
};


template <AsyncLog::level_type MessageLevel>
void AsyncLog::write(const boost::format& message) throw()
{
	try {
		queue.push( std::make_pair(message,MessageLevel) );
	} catch (const std::runtime_error& ex) {
		std::cerr << "Log Error: " << ex.what() << std::endl;
	} catch (...) {
		std::cerr << "Log Error" << std::endl;
	}
}

template <AsyncLog::level_type MessageLevel>
void AsyncLog::write(const std::string& message) throw()
{
	try {
		write<MessageLevel>( format("%1%") % message );
	} catch (const std::runtime_error& ex) {
		std::cerr << "Log Error: " << ex.what() << std::endl;
	} catch (...) {
		std::cerr << "Log Error" << std::endl;
	}
}


}  // namespace VFIELD

#endif /* asynclog.h */
